!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output of metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_meta
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature'' in metadynamics.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, temp, boltz, gkt, qmass, vxbath, &
     &   vybath, vzbath, xbath, ybath, zbath, hamiltonian, potential, &
     &   iprint_std, natom, nbead, nnhc, ncolor

      use meta_variables, only : &
     &   fictmass_meta, vmeta, ekin_meta, vbath_meta, rbath_meta, &
     &   ebath_meta, pot_ref_meta, potential_ref_meta, ekin_sys, &
     &   ebath_sys, potential_meta, qmass_meta, pot_meta, &
     &   pot_lifted, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   kinetic energy                                             */
!-----------------------------------------------------------------------

      ekin_sys = 0.d0

      do j = 1, nbead
      do i = 1, natom
         ekin_sys = ekin_sys + 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j)
         ekin_sys = ekin_sys + 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j)
         ekin_sys = ekin_sys + 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      temp = 2.d0*ekin_sys/dble(3*natom*nbead)/boltz

!-----------------------------------------------------------------------
!     /*   kinetic energy of collective variables                     */
!-----------------------------------------------------------------------

      ekin_meta = 0.d0

      do j = 1, nbead
      do i = 1, nmeta
         ekin_meta = ekin_meta &
     &             + 0.5d0*fictmass_meta(i)*vmeta(i,j)*vmeta(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_sys  =  thermostats attached to system               */
!-----------------------------------------------------------------------

      ebath_sys = 0.d0

      do k = 1, nbead
         do j = 1, nnhc
         do i = 1, natom
            ebath_sys = ebath_sys &
     &           + 0.5d0*qmass(k)*vxbath(i,j,k)*vxbath(i,j,k) &
     &           + 0.5d0*qmass(k)*vybath(i,j,k)*vybath(i,j,k) &
     &           + 0.5d0*qmass(k)*vzbath(i,j,k)*vzbath(i,j,k) &
     &           + gkt*xbath(i,j,k) &
     &           + gkt*ybath(i,j,k) &
     &           + gkt*zbath(i,j,k)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_meta  =  thermostats to collective variables         */
!-----------------------------------------------------------------------

      ebath_meta = 0.d0

      do k = 1, nbead
      do m = 1, ncolor
      do j = 1, nnhc
      do i = 1, nmeta
         ebath_meta = ebath_meta &
     &   + 0.5d0*qmass_meta(m)*vbath_meta(i,j,m,k)*vbath_meta(i,j,m,k) &
     &   + gkt*rbath_meta(i,j,m,k)
      end do
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   total energy                                               */
!-----------------------------------------------------------------------

      potential_meta = 0.d0

      do i = 1, nbead
         potential_meta = potential_meta + pot_meta(i)
      end do

      potential_ref_meta = 0.d0

      do i = 1, nbead
         potential_ref_meta = potential_ref_meta + pot_ref_meta(i)
      end do

!-----------------------------------------------------------------------
!     /*   total energy                                               */
!-----------------------------------------------------------------------

      hamiltonian = ekin_sys + potential + ekin_meta + potential_meta &
     &            + potential_ref_meta + ebath_sys + ebath_meta &
     &            - pot_lifted

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_md

      return
      end
