!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output of molecular dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_md_nve_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, temp, boltz, ekin, hamiltonian,  &
     &   potential, iprint_std, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   kinetic energy                                             */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = 1, nbead
      do i = 1, natom
         ekin = ekin + 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j)
         ekin = ekin + 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j)
         ekin = ekin + 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      temp = 2.d0*ekin/dble(3*natom*nbead)/boltz

!-----------------------------------------------------------------------
!     /*   total energy                                               */
!-----------------------------------------------------------------------

      hamiltonian = ekin + potential

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_md_MPI

      return
      end





!***********************************************************************
      subroutine standard_md_nvt_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, temp, boltz, gkt, ekin, ebath, &
     &   hamiltonian, potential, qmass_cart, vxbath_cart, vybath_cart, &
     &   vzbath_cart, xbath_cart, ybath_cart, zbath_cart, &
     &   ncolor, iprint_std, natom, nbead, nnhc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   kinetic energy                                             */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = 1, nbead
      do i = 1, natom
         ekin = ekin + 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j)
         ekin = ekin + 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j)
         ekin = ekin + 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      temp = 2.d0*ekin/dble(3*natom*nbead)/boltz

!-----------------------------------------------------------------------
!     /*   REVISED: energy of thermostats                             */
!-----------------------------------------------------------------------

!      ebath = 0.d0
!
!      do k = 1, nbead
!         do j = 1, nnhc
!         do i = 1, natom
!            ebath = ebath
!     &           + 0.5d0*qmass(k)*vxbath(i,j,k)*vxbath(i,j,k)
!     &           + 0.5d0*qmass(k)*vybath(i,j,k)*vybath(i,j,k)
!     &           + 0.5d0*qmass(k)*vzbath(i,j,k)*vzbath(i,j,k)
!     &           + gkt*xbath(i,j,k)
!     &           + gkt*ybath(i,j,k)
!     &           + gkt*zbath(i,j,k)
!         end do
!         end do
!      end do

      ebath = 0.d0

      do k = 1, nbead
      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath = ebath &
     &   + 0.5d0*qmass_cart(i,m) &
     &          *vxbath_cart(j,i,m,k)*vxbath_cart(j,i,m,k) &
     &   + 0.5d0*qmass_cart(i,m) &
     &          *vybath_cart(j,i,m,k)*vybath_cart(j,i,m,k) &
     &   + 0.5d0*qmass_cart(i,m) &
     &          *vzbath_cart(j,i,m,k)*vzbath_cart(j,i,m,k) &
     &   + gkt*xbath_cart(j,i,m,k) &
     &   + gkt*ybath_cart(j,i,m,k) &
     &   + gkt*zbath_cart(j,i,m,k)
      end do
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   REVISED: total energy                                      */
!-----------------------------------------------------------------------

      hamiltonian = ekin + potential + ebath

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_md_MPI

      return
      end





!***********************************************************************
      subroutine standard_md_nve_cons_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, temp, boltz,  hamiltonian, potential, &
     &   ekin, method, iprint_std, natom, nbead

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( method(1:5) .eq. 'AFED ' ) iset = 1

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   kinetic energy                                             */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = 1, nbead
      do i = 1, natom
         ekin = ekin + 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j)
         ekin = ekin + 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j)
         ekin = ekin + 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      temp = 2.d0*ekin/dble(3*natom*nbead)/boltz

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   total energy                                               */
!-----------------------------------------------------------------------

      hamiltonian = ekin + potential + potential_ref_cons

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_md_MPI

      return
      end





!***********************************************************************
      subroutine standard_md_nvt_cons_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, temp, boltz, gkt, ekin, ebath, &
     &   hamiltonian, potential, qmass_cart, vxbath_cart, vybath_cart, &
     &   vzbath_cart, xbath_cart, ybath_cart, zbath_cart, &
     &   ncolor, iprint_std, natom, nbead, nnhc

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   kinetic energy                                             */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = 1, nbead
      do i = 1, natom
         ekin = ekin + 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j)
         ekin = ekin + 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j)
         ekin = ekin + 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      temp = 2.d0*ekin/dble(3*natom*nbead)/boltz

!-----------------------------------------------------------------------
!     /*   REVISED: energy of thermostats                             */
!-----------------------------------------------------------------------

!      ebath = 0.d0
!
!      do k = 1, nbead
!         do j = 1, nnhc
!         do i = 1, natom
!            ebath = ebath
!     &           + 0.5d0*qmass(k)*vxbath(i,j,k)*vxbath(i,j,k)
!     &           + 0.5d0*qmass(k)*vybath(i,j,k)*vybath(i,j,k)
!     &           + 0.5d0*qmass(k)*vzbath(i,j,k)*vzbath(i,j,k)
!     &           + gkt*xbath(i,j,k)
!     &           + gkt*ybath(i,j,k)
!     &           + gkt*zbath(i,j,k)
!         end do
!         end do
!      end do

      ebath = 0.d0

      do k = 1, nbead
      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath = ebath &
     &   + 0.5d0*qmass_cart(i,m) &
     &          *vxbath_cart(j,i,m,k)*vxbath_cart(j,i,m,k) &
     &   + 0.5d0*qmass_cart(i,m) &
     &          *vybath_cart(j,i,m,k)*vybath_cart(j,i,m,k) &
     &   + 0.5d0*qmass_cart(i,m) &
     &          *vzbath_cart(j,i,m,k)*vzbath_cart(j,i,m,k) &
     &   + gkt*xbath_cart(j,i,m,k) &
     &   + gkt*ybath_cart(j,i,m,k) &
     &   + gkt*zbath_cart(j,i,m,k)
      end do
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   REVISED: total energy                                      */
!-----------------------------------------------------------------------

      hamiltonian = ekin + potential + potential_ref_cons + ebath

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_md_MPI

      return
      end
