!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     output for path integral hybrid monte carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_hmc_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature'':  no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, dnmmass, ux, uy, uz, omega_p2, &
     &   hamiltonian_sys, potential, ebath_cent, ebath_mode, au_time, &
     &   char_date, dt, au_time, hamiltonian, istep_start, iprint_std, &
     &   ndof, iounit, iounit_std, natom, nbead, istep, myrank

      use hmc_variables, only : &
     &   ratio

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, imode, iatom

      real(8) :: factqk, dt_fs

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         if ( myrank .ne. 0 ) return

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open( iounit_std, file = 'standard.out', access = 'append' )

            write( iounit_std, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( iounit_std, '(a)' ) &
     &      '    step dt [fs]   ratio  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( iounit_std, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            close( iounit_std )

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step dt [fs]   ratio  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         else

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step dt [fs]   ratio  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         end if

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( ( myrank .eq. 0) .and. (mod(istep,iprint_std) .eq. 0) ) then

!        /*   wall clock time   */
         call getdate

!        /*   dt in fs   */
         dt_fs = dt / (1.d-15/au_time)

!        /*   if initial step   */
         if ( istep .eq. istep_start ) then

!           /*   open   */
            open( iounit_std, file = 'standard.out', access = 'append' )

!           /*   output   */
            write( iounit_std, '(i8,f8.3,a8,f16.8,f10.2,2x,a26)' ) &
     &         istep, dt_fs, ' -------', potential, temp, char_date

!           /*   close   */
            close( iounit_std )

!           /*   output   */
            write( 6, '(i8,f8.3,a8,f16.8,f10.2,2x,a26)' ) &
     &         istep, dt_fs, ' -------', potential, temp, char_date

!           /*   make sure output   */
            flush( 6 )

!        /*   if not initial step   */
         else

!           /*   open   */
            open( iounit_std, file = 'standard.out', access = 'append' )

            write( iounit_std, '(i8,f8.3,f8.5,f16.8,f10.2,2x,a26)' ) &
     &         istep, dt_fs, ratio, potential, temp, char_date

!           /*   close   */
            close( iounit_std )

!           /*   output   */
            write( 6, '(i8,f8.3,f8.5,f16.8,f10.2,2x,a26)' ) &
     &         istep, dt_fs, ratio, potential, temp, char_date

!           /*   make sure output   */
            flush( 6 )

!        /*   end of if statement   */
         end if

!     /*   print interval   */
      end if

      return
      end
