!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output of geometry optimization
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_geoopt
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   iprint_std

      use lbfgs_variables, only : &
     &   pos, pos0, grad, frms, fmax, drms, dmax, ndim

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: k

      integer, save :: iset = 0

      real(8) :: posx, posy, posz

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_opt( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     //   check convergence:  root-mean-square of residual force
!-----------------------------------------------------------------------

      frms = 0.d0

      do k = 1, ndim
         frms = frms + grad(k)*grad(k)
      end do

      frms = sqrt(frms)/ndim

!-----------------------------------------------------------------------
!     //   check convergence:  maximum value of residual force
!-----------------------------------------------------------------------

      fmax = 0.d0

      do k = 1, ndim
         fmax = max ( abs( grad(k) ), fmax )
      end do

!-----------------------------------------------------------------------
!     //   check convergence:  root-mean-square of geometrical shift
!-----------------------------------------------------------------------

      drms = 0.d0

!      do k = 1, ndim
!         drms = drms + (pos(k)-pos0(k))*(pos(k)-pos0(k))
!      end do

      do k = 1, ndim, 3
         posx = pos(k+0) - pos0(k+0)
         posy = pos(k+1) - pos0(k+1)
         posz = pos(k+2) - pos0(k+2)
         call pbc_atom ( posx, posy, posz )
         drms = drms + posx*posx + posy*posy + posz*posz
      end do

      drms = sqrt(drms)/ndim

!-----------------------------------------------------------------------
!     //   check convergence:  maximum value of geometrical shift
!-----------------------------------------------------------------------

      dmax = 0.d0

!      do k = 1, ndim
!         dmax = max ( abs( pos(k)-pos0(k) ), dmax )
!      end do

      do k = 1, ndim, 3
         posx = pos(k+0) - pos0(k+0)
         posy = pos(k+1) - pos0(k+1)
         posz = pos(k+2) - pos0(k+2)
         call pbc_atom ( posx, posy, posz )
         dmax = max( abs(posx), dmax )
         dmax = max( abs(posy), dmax )
         dmax = max( abs(posz), dmax )
      end do

!-----------------------------------------------------------------------
!     //   output
!-----------------------------------------------------------------------

      call standard_output_opt

      return
      end





!***********************************************************************
      subroutine standard_init_opt( iset )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iprint_std, iounit, iounit_std

      use lbfgs_variables, only : &
     &   dmax_tol, drms_tol, fmax_tol, frms_tol

      implicit none

      integer :: iset, itest

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest, iounit )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write( iounit_std, '(a)' ) &
     &      '==================================================' // &
     &      '============================'

            write( iounit_std, '(a)' ) &
     &      '  step     energy [au]  maxdis   rmsdis   maxfrc  ' // &
     &      '  rmsfrc  wall clock time   '

            write( iounit_std, '(22x,f8.4,2f9.5,f10.6)' ) &
     &         dmax_tol, drms_tol, fmax_tol, frms_tol

            write( iounit_std, '(a)' ) &
     &      '--------------------------------------------------' // &
     &      '----------------------------'

            close( iounit_std )

            write( 6, '(a)' ) &
     &      '==================================================' // &
     &      '============================'

            write( 6, '(a)' ) &
     &      '  step     energy [au]  maxdis   rmsdis   maxfrc  ' // &
     &      '  rmsfrc  wall clock time   '

            write( 6, '(22x,f8.4,2f9.5,f10.6)' ) &
     &         dmax_tol, drms_tol, fmax_tol, frms_tol

            write( 6,'(a)' ) &
     &      '--------------------------------------------------' // &
     &      '----------------------------'

            flush( 6 )

         else

            write( 6, '(a)' ) &
     &      '==================================================' // &
     &      '============================'

            write( 6, '(a)' ) &
     &      '  step     energy [au]  maxdis   rmsdis   maxfrc  ' // &
     &      '  rmsfrc  wall clock time   '

            write( 6, '(22x,f8.4,2f9.5,f10.6)' ) &
     &         dmax_tol, drms_tol, fmax_tol, frms_tol

            write( 6,'(a)' ) &
     &      '--------------------------------------------------' // &
     &      '----------------------------'

            flush( 6 )

         end if

      end if

      return
      end





!***********************************************************************
      subroutine standard_output_opt
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      use common_variables, only :  &
     &   char_date, potential, iprint_std, iounit_std, istep

      use lbfgs_variables, only : &
     &   func, dmax, drms, fmax, frms

      use cons_variables, only : &
     &   ncons

      implicit none

!-----------------------------------------------------------------------
!     /*   print output                                               */
!-----------------------------------------------------------------------

      if ( iprint_std .gt. 0 ) then
      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   open file   */
         open( iounit_std, file = 'standard.out', access = 'append' )

!        /*   output   */
         if ( ncons .le. 0 ) then
            write( iounit_std, '(i6,f16.8,f8.4,2f9.5,f10.6,2x,a18)' ) &
     &         istep, func, dmax, drms, fmax, frms, char_date(6:23)
         else
            write( iounit_std, '(i6,f16.8,f8.4,2f9.5,f10.6,2x,a18)' ) &
     &         istep, potential, dmax, drms, fmax, frms, char_date(6:23)
         end if

!        /*   close file   */
         close( iounit_std )

!        /*   output   */
         if ( ncons .le. 0 ) then
            write( 6, '(i6,f16.8,f8.4,2f9.5,f10.6,2x,a18)' ) &
     &         istep, func, dmax, drms, fmax, frms, char_date(6:23)
         else
            write( 6, '(i6,f16.8,f8.4,2f9.5,f10.6,2x,a18)' ) &
     &         istep, potential, dmax, drms, fmax, frms, char_date(6:23)
         end if

!        /*   make sure output   */
         flush( 6 )

      end if
      end if

      return
      end
