!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     extensive MPI parallelization
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_nve_0_XMPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, dnmmass, &
     &   hamiltonian_sys, omega_p2, ux, uy, uz, ebath_cent, ebath_mode, &
     &   fictmass, ndof, iprint_std, vux, vuy, vuz

      use XMPI_variables, only : &
     &   jstart_bead, jend_bead, jstart_atom, jend_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom
         ekin = ekin + fictmass(i,j)*vux(i,j)*vux(i,j) &
     &               + fictmass(i,j)*vuy(i,j)*vuy(i,j) &
     &               + fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      ekin = 0.5d0 * ekin

      call my_mpi_allreduce_real_0( ekin )
!      call my_mpi_allreduce_real_0_sub( ekin )
!      call my_mpi_allreduce_real_0_main( ekin )

      temp = 2.d0 * ekin / dble(ndof) / boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do j = jstart_bead, jend_bead
         if ( j .eq. 1 ) cycle
         do i = jstart_atom, jend_atom
            factqk = 0.5d0*dnmmass(i,j)*omega_p2
            qkin = qkin &
     &           + factqk*ux(i,j)*ux(i,j) &
     &           + factqk*uy(i,j)*uy(i,j) &
     &           + factqk*uz(i,j)*uz(i,j)
         end do
      end do

      call my_mpi_allreduce_real_0( qkin )
!      call my_mpi_allreduce_real_0_sub( ekin )
!      call my_mpi_allreduce_real_0_main( ekin )

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end





!***********************************************************************
      subroutine standard_pibcmd_0_XMPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, &
     &   hamiltonian_sys, hamiltonian_cor, omega_p2, dnmmass, ux, uy, &
     &   uz, ebath_cent, ebath_mode, vux, vuy, vuz, fictmass, &
     &   ndof, iprint_std, iounit, iounit_std

      use XMPI_variables, only : &
     &   jstart_bead, jend_bead, jstart_atom, jend_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom
         ekin = ekin + fictmass(i,j)*vux(i,j)*vux(i,j) &
     &               + fictmass(i,j)*vuy(i,j)*vuy(i,j) &
     &               + fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      ekin = 0.5d0 * ekin

      call my_mpi_allreduce_real_0( ekin )
!      call my_mpi_allreduce_real_0_sub( ekin )
!      call my_mpi_allreduce_real_0_main( ekin )

      temp = 2.d0 * ekin / dble(ndof) / boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do j = jstart_bead, jend_bead
         if ( j .eq. 1 ) cycle
         do i = jstart_atom, jend_atom
            factqk = 0.5d0*dnmmass(i,j)*omega_p2
            qkin = qkin &
     &           + factqk*ux(i,j)*ux(i,j) &
     &           + factqk*uy(i,j)*uy(i,j) &
     &           + factqk*uz(i,j)*uz(i,j)
         end do
      end do

      call my_mpi_allreduce_real_0( qkin )
!      call my_mpi_allreduce_real_0_sub( qkin )
!      call my_mpi_allreduce_real_0_main( qkin )

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

      hamiltonian = hamiltonian + hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end





!***********************************************************************
      subroutine standard_nvt_mnhc_XMPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type III.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   xbath_cent, ybath_cent, zbath_cent, qmass, ebath_cent, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath,  ebath_mode, ndof, iprint_std, &
     &   ncolor, nnhc, vux, vuy, vuz, fictmass

      use XMPI_variables, only : &
     &   jstart_bead, jend_bead, jstart_atom, jend_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom
         ekin = ekin + fictmass(i,j)*vux(i,j)*vux(i,j) &
     &               + fictmass(i,j)*vuy(i,j)*vuy(i,j) &
     &               + fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      ekin = 0.5d0 * ekin

      call my_mpi_allreduce_real_0( ekin )
!      call my_mpi_allreduce_real_0_sub( ekin )
!      call my_mpi_allreduce_real_0_main( ekin )

      temp = 2.d0 * ekin / dble(ndof) / boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do j = jstart_bead, jend_bead
         if ( j .eq. 1 ) cycle
         do i = jstart_atom, jend_atom
            factqk = 0.5d0*dnmmass(i,j)*omega_p2
            qkin = qkin &
     &           + factqk*ux(i,j)*ux(i,j) &
     &           + factqk*uy(i,j)*uy(i,j) &
     &           + factqk*uz(i,j)*uz(i,j)
         end do
      end do

      call my_mpi_allreduce_real_0( qkin )
!      call my_mpi_allreduce_real_0_sub( qkin )
!      call my_mpi_allreduce_real_0_main( qkin )

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do k = jstart_bead, jend_bead
         if ( k .ne. 1 ) cycle
      do m = 1, ncolor
      do i = 1, nnhc
      do j = jstart_atom, jend_atom
         ebath_cent = ebath_cent &
     &   + 0.5d0*qmass_cent(i,m)*vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &   + gkt*xbath_cent(j,i,m) &
     &   + gkt*ybath_cent(j,i,m) &
     &   + gkt*zbath_cent(j,i,m)
      end do
      end do
      end do
      end do

      call my_mpi_allreduce_real_0( ebath_cent )
!      call my_mpi_allreduce_real_0_sub( ebath_cent )
!      call my_mpi_allreduce_real_0_main( ebath_cent )

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = jstart_bead, jend_bead
         if ( i .eq. 1 ) cycle
         do j = 1, nnhc
         do k = jstart_atom, jend_atom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

      call my_mpi_allreduce_real_0( ebath_mode )
!      call my_mpi_allreduce_real_0_sub( ebath_mode )
!      call my_mpi_allreduce_real_0_main( ebath_mode )

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end





!***********************************************************************
      subroutine standard_cmd_0_XMPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type IV.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ebath_mode, ux, uy, uz, &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, ebath_cent, qmass, &
     &   nnhc, ndof, iprint_std, fictmass, vux, vuy, vuz

      use XMPI_variables, only : &
     &   jstart_bead, jend_bead, jstart_atom, jend_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer ::  i, j, k

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do j = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom
         ekin = ekin + fictmass(i,j)*vux(i,j)*vux(i,j) &
     &               + fictmass(i,j)*vuy(i,j)*vuy(i,j) &
     &               + fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      ekin = 0.5d0 * ekin

      call my_mpi_allreduce_real_0( ekin )

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do j = jstart_bead, jend_bead
         if ( j .eq. 1 ) cycle
         do i = jstart_atom, jend_atom
            factqk = 0.5d0*dnmmass(i,j)*omega_p2
            qkin = qkin &
     &           + factqk*ux(i,j)*ux(i,j) &
     &           + factqk*uy(i,j)*uy(i,j) &
     &           + factqk*uz(i,j)*uz(i,j)
         end do
      end do

      call my_mpi_allreduce_real_0( qkin )

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = jstart_bead, jend_bead
         if ( i .eq. 1 ) cycle
         do j = 1, nnhc
         do k = jstart_atom, jend_atom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

      call my_mpi_allreduce_real_0( ebath_mode )

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end
