!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 10, 2022 by M. Shiga
!      Description:     standard output of path integrals
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine standard_init( iset )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      use common_variables, only : iprint_std, iounit, iounit_std

      implicit none

      integer :: iset, itest

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest, iounit )

         if ( itest .eq. 1 ) then

            open( iounit_std, file = 'standard.out', access = 'append' )

            write( iounit_std, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( iounit_std, '(a)' ) &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( iounit_std, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            close( iounit_std )

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         else

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         end if

      end if

      return
      end





!***********************************************************************
      subroutine standard_output
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      use common_variables, only :  &
     &   hamiltonian, potential, temp, char_date, iprint_std, &
     &   iounit_std, istep

      implicit none

!-----------------------------------------------------------------------
!     /*   print output                                               */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   open file   */
         open( iounit_std, file = 'standard.out', access = 'append' )

!        /*   output to file   */
         write( iounit_std, '(i8,2f16.8,f10.2,2x,a26)' ) &
     &      istep, hamiltonian, potential, temp, char_date

!        /*   close file   */
         close( iounit_std )

!        /*   output   */
         write( 6, '(i8,2f16.8,f10.2,2x,a26)' ) &
     &      istep, hamiltonian, potential, temp, char_date

!        /*   make sure the output   */
         flush( 6 )

      end if

      return
      end





!***********************************************************************
      subroutine standard_output_md
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      use common_variables, only :  &
     &   hamiltonian, potential, temp, char_date, iprint_std, nbead, &
     &   iounit_std, istep

      implicit none

!-----------------------------------------------------------------------
!     /*   print output                                               */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   open file   */
         open( iounit_std, file = 'standard.out', access = 'append' )

!        /*   output to file   */
         write( iounit_std, '(i8,2f16.8,f10.2,2x,a28)' ) &
     &      istep, hamiltonian/nbead, potential/nbead, temp, char_date

!        /*   close file   */
         close( iounit_std )

!        /*   output   */
         write( 6, '(i8,2f16.8,f10.2,2x,a26)' ) &
     &      istep, hamiltonian/nbead, potential/nbead, temp, char_date

!        /*   make sure the output   */
         flush( 6 )

      end if

      return
      end





!***********************************************************************
      subroutine standard_nve_0
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, dnmmass, &
     &   hamiltonian_sys, omega_p2, ux, uy, uz, ebath_cent, ebath_mode, &
     &   ndof, iprint_std, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nvt_nhc
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type I.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   gnkt, rbath_cent, vbath_cent, qmass, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, ebath_cent, ebath_mode, ndof, nnhc, &
     &   iprint_std, nbead, natom, ncolor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
         ebath_cent = ebath_cent &
     &      + 0.5d0*qmass_cent(1,m)*vbath_cent(1,m)*vbath_cent(1,m) &
     &      + gnkt*rbath_cent(1,m)
         do i = 2, nnhc
             ebath_cent = ebath_cent &
     &      + 0.5d0*qmass_cent(i,m)*vbath_cent(i,m)*vbath_cent(i,m) &
     &      + gkt*rbath_cent(i,m)
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
      do j = 1, nnhc
      do k = 1, natom
         ebath_mode = ebath_mode &
     &         + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &         + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &         + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &         + gkt*xbath(k,j,i) &
     &         + gkt*ybath(k,j,i) &
     &         + gkt*zbath(k,j,i)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nvt_nhcs
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type II.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   xbath_cent, ybath_cent, zbath_cent, qmass, ebath_cent, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, ebath_mode, ndof, nnhc, iprint_std, &
     &   nbead, natom, ncolor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
         ebath_cent = ebath_cent &
     &    + 0.5d0*qmass_cent(1,m)*vxbath_cent(1,1,m)*vxbath_cent(1,1,m) &
     &    + 0.5d0*qmass_cent(1,m)*vybath_cent(1,1,m)*vybath_cent(1,1,m) &
     &    + 0.5d0*qmass_cent(1,m)*vzbath_cent(1,1,m)*vzbath_cent(1,1,m) &
     &    + natom*gkt*xbath_cent(1,1,m) &
     &    + natom*gkt*ybath_cent(1,1,m) &
     &    + natom*gkt*zbath_cent(1,1,m)
         do i = 2, nnhc
            ebath_cent = ebath_cent &
     &    + 0.5d0*qmass_cent(i,m)*vxbath_cent(1,i,m)*vxbath_cent(1,i,m) &
     &    + 0.5d0*qmass_cent(i,m)*vybath_cent(1,i,m)*vybath_cent(1,i,m) &
     &    + 0.5d0*qmass_cent(i,m)*vzbath_cent(1,i,m)*vzbath_cent(1,i,m) &
     &    + gkt*xbath_cent(1,i,m) &
     &    + gkt*ybath_cent(1,i,m) &
     &    + gkt*zbath_cent(1,i,m)
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nvt_mnhc
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type III.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   xbath_cent, ybath_cent, zbath_cent, qmass, ebath_cent, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath,  ebath_mode, ndof, iprint_std, &
     &   nbead, natom, ncolor, nnhc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath_cent = ebath_cent &
     &   + 0.5d0*qmass_cent(i,m)*vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &   + gkt*xbath_cent(j,i,m) &
     &   + gkt*ybath_cent(j,i,m) &
     &   + gkt*zbath_cent(j,i,m)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_cmd_0
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type IV.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ebath_mode, ux, uy, uz, &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, ebath_cent, qmass, &
     &   nnhc, ndof, iprint_std, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
      do j = 1, nnhc
      do k = 1, natom
         ebath_mode = ebath_mode &
     &         + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &         + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &         + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &         + gkt*xbath(k,j,i) &
     &         + gkt*ybath(k,j,i) &
     &         + gkt*zbath(k,j,i)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nve_cons
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, ebath_cent, &
     &   ebath_mode, ndof, iprint_std, nbead, natom

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + potential_ref_cons &
     &            + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nvt_nhc_cons
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type I.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   gnkt, rbath_cent, vbath_cent, qmass, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, ebath_cent, ebath_mode, ndof, nnhc, &
     &   iprint_std, nbead, natom, ncolor

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
         ebath_cent = ebath_cent &
     &      + 0.5d0*qmass_cent(1,m)*vbath_cent(1,m)*vbath_cent(1,m) &
     &      + gnkt*rbath_cent(1,m)
         do i = 2, nnhc
             ebath_cent = ebath_cent &
     &      + 0.5d0*qmass_cent(i,m)*vbath_cent(i,m)*vbath_cent(i,m) &
     &      + gkt*rbath_cent(i,m)
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
      do j = 1, nnhc
      do k = 1, natom
         ebath_mode = ebath_mode &
     &         + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &         + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &         + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &         + gkt*xbath(k,j,i) &
     &         + gkt*ybath(k,j,i) &
     &         + gkt*zbath(k,j,i)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + potential_ref_cons &
     &            + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nvt_nhcs_cons
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type II.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   xbath_cent, ybath_cent, zbath_cent, qmass, ebath_cent, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, ebath_mode, ndof, iprint_std, &
     &   nbead, natom, ncolor, nnhc

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
         ebath_cent = ebath_cent &
     &    + 0.5d0*qmass_cent(1,m)*vxbath_cent(1,1,m)*vxbath_cent(1,1,m) &
     &    + 0.5d0*qmass_cent(1,m)*vybath_cent(1,1,m)*vybath_cent(1,1,m) &
     &    + 0.5d0*qmass_cent(1,m)*vzbath_cent(1,1,m)*vzbath_cent(1,1,m) &
     &    + natom*gkt*xbath_cent(1,1,m) &
     &    + natom*gkt*ybath_cent(1,1,m) &
     &    + natom*gkt*zbath_cent(1,1,m)
         do i = 2, nnhc
            ebath_cent = ebath_cent &
     &    + 0.5d0*qmass_cent(i,m)*vxbath_cent(1,i,m)*vxbath_cent(1,i,m) &
     &    + 0.5d0*qmass_cent(i,m)*vybath_cent(1,i,m)*vybath_cent(1,i,m) &
     &    + 0.5d0*qmass_cent(i,m)*vzbath_cent(1,i,m)*vzbath_cent(1,i,m) &
     &    + gkt*xbath_cent(1,i,m) &
     &    + gkt*ybath_cent(1,i,m) &
     &    + gkt*zbath_cent(1,i,m)
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + potential_ref_cons &
     &            + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_nvt_mnhc_cons
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type III.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   xbath_cent, ybath_cent, zbath_cent, qmass, ebath_cent, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, ebath_mode, ndof, iprint_std, &
     &   nbead, natom, ncolor, nnhc

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath_cent = ebath_cent &
     &   + 0.5d0*qmass_cent(i,m)*vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &   + gkt*xbath_cent(j,i,m) &
     &   + gkt*ybath_cent(j,i,m) &
     &   + gkt*zbath_cent(j,i,m)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + potential_ref_cons &
     &            + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_rpmd_nvt
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type I.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   gnkt, rbath_cent, vbath_cent, ebath_cent, ebath_mode, ndof, &
     &   iprint_std, nbead, natom, ncolor, nnhc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
         ebath_cent = ebath_cent &
     &      + 0.5d0*qmass_cent(1,m)*vbath_cent(1,m)*vbath_cent(1,m) &
     &      + gnkt*rbath_cent(1,m)
         do i = 2, nnhc
             ebath_cent = ebath_cent &
     &      + 0.5d0*qmass_cent(i,m)*vbath_cent(i,m)*vbath_cent(i,m) &
     &      + gkt*rbath_cent(i,m)
         end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!      do i = 2, nbead
!      do j = 1, nnhc
!      do k = 1, natom
!         ebath_mode = ebath_mode
!     &         + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i)
!     &         + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i)
!     &         + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i)
!     &         + gkt*xbath(k,j,i)
!     &         + gkt*ybath(k,j,i)
!     &         + gkt*zbath(k,j,i)
!      end do
!      end do
!      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_cmd_cons
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type IV.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ebath_mode, ux, uy, uz, &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, ebath_cent, qmass, &
     &   nnhc, ndof, iprint_std, nbead, natom

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
      do j = 1, nnhc
      do k = 1, natom
         ebath_mode = ebath_mode &
     &         + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &         + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &         + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &         + gkt*xbath(k,j,i) &
     &         + gkt*ybath(k,j,i) &
     &         + gkt*zbath(k,j,i)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + potential_ref_cons &
     &            + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine standard_pibcmd_0
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, &
     &   hamiltonian_sys, hamiltonian_cor, omega_p2, dnmmass, ux, uy, &
     &   uz, ebath_cent, ebath_mode, char_date, ndof, iprint_std, &
     &   iounit, iounit_std, nbead, natom, istep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, imode, iatom

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest, iounit )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

      hamiltonian = hamiltonian + hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then
!        /*   wall clock time   */
         call getdate
!        /*   output   */
         write(iounit_std,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date
         write(         6,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date
      end if

      return
      end





!***********************************************************************
      subroutine standard_pibcmd_cons
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, &
     &   hamiltonian_sys, hamiltonian_cor, omega_p2, dnmmass, ux, uy, &
     &   uz, ebath_cent, ebath_mode, char_date, ndof, iprint_std, &
     &   iounit, iounit_std, nbead, natom, istep

      use cons_variables, only : &
     &   pot_ref_cons, potential_ref_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, imode, iatom, i

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest, iounit )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   energy of constraints                                      */
!-----------------------------------------------------------------------

      potential_ref_cons = 0.d0

      do i = 1, nbead
         potential_ref_cons = potential_ref_cons + pot_ref_cons(i)
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + potential_ref_cons &
     &   + ebath_mode + ebath_cent

      hamiltonian = hamiltonian + hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then
!        /*   wall clock time   */
         call getdate
!        /*   output   */
         write(iounit_std,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date
         write(         6,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date
      end if

      return
      end





!***********************************************************************
      subroutine standard_pibcmd_platen
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, &
     &   hamiltonian_sys, hamiltonian_cor, omega_p2, dnmmass, ux, uy, &
     &   uz, char_date, ndof, iprint_std, &
     &   iounit, iounit_std, nbead, natom, istep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, imode, iatom

      real(8) :: factqk

      integer, save :: istep_avg = 0
      real(8), save :: hamiltonian_avg = 0.d0

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest, iounit )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   average                                                    */
!-----------------------------------------------------------------------

      istep_avg = istep_avg + 1

      hamiltonian_avg = &
     &     hamiltonian_avg*dble(istep_avg-1)/dble(istep_avg) &
     &   + hamiltonian/dble(istep_avg)

!-----------------------------------------------------------------------
!     /*   hamiltonian removing linear drifts                         */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian - 2.d0*hamiltonian_avg

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then
!        /*   wall clock time   */
         call getdate
!        /*   output   */
         write(iounit_std,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date
         write(         6,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date
      end if

      return
      end





!***********************************************************************
      subroutine standard_pibcmd_gillespie
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, dnmmass, &
     &   hamiltonian_sys, hamiltonian_cor, omega_p2, ux, uy, uz, &
     &   ebath_cent, ebath_mode, char_date, ndof, iprint_std, tau_bcmd, &
     &   iounit, iounit_std, nbead, natom, istep, dt

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, imode, iatom

      real(8) :: factqk, g4

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest, iounit )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      g4 = ( 1.d0 - exp(-dt/tau_bcmd) ) / (dt/tau_bcmd)

      hamiltonian_sys = ekin + qkin*g4 + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

      hamiltonian = hamiltonian + hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then
!        /*   wall clock time   */
         call getdate
!        /*   output   */
         write(iounit_std,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date
         write(         6,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date
      end if

      return
      end