!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    July 1, 2020 by M. Shiga
!      Description:     spline intrapolation of string
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine spline_init_pot_string
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : pot, nbead

      use string_variables, only : s_ref, pot1_ref, pot2_ref

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: j

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      do j = 1, nbead

         s_ref(j,1)  = dble(j-1) / dble(nbead-1)
         pot1_ref(j,1) = pot(j)

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_string( s_ref, pot1_ref, pot2_ref, nbead, 1 )

      return
      end





!***********************************************************************
      subroutine spline_init_crd_string
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : x, y, z, fictmass, natom, nbead

      use string_variables, only : s_ref, r1_ref, r2_ref

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k

!     /*   real number   */
      real(8) :: factor

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      do j = 1, nbead

         k = 0

         do i = 1, natom

            factor = sqrt(fictmass(i,1))

            k = k + 1

            s_ref(j,k)  = dble(j-1) / dble(nbead-1)
            r1_ref(j,k) = factor * x(i,j)

            k = k + 1

            s_ref(j,k)  = dble(j-1) / dble(nbead-1)
            r1_ref(j,k) = factor * y(i,j)

            k = k + 1

            s_ref(j,k)  = dble(j-1) / dble(nbead-1)
            r1_ref(j,k) = factor * z(i,j)

         end do

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_string( s_ref, r1_ref, r2_ref, nbead, 3*natom )

      return
      end





!***********************************************************************
      subroutine spline_init_frc_string
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : fx, fy, fz, natom, nbead

      use string_variables, only : s_ref, f1_ref, f2_ref

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      do j = 1, nbead

         k = 0

         do i = 1, natom

            k = k + 1

            f1_ref(j,k) = fx(i,j) * dble(nbead)

            k = k + 1

            f1_ref(j,k) = fy(i,j) * dble(nbead)

            k = k + 1

            f1_ref(j,k) = fz(i,j) * dble(nbead)

         end do

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_string( s_ref, f1_ref, f2_ref, nbead, 3*natom )

      return
      end





!***********************************************************************
      subroutine spline_init_arc_string
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use string_variables, only : arc_ref, s1_ref, s2_ref, narc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: j

!-----------------------------------------------------------------------
!     /*   reference data                                             */
!-----------------------------------------------------------------------

      do j = 1, narc

         s1_ref(j,1) = dble(j-1) / dble(narc-1)

      end do

!-----------------------------------------------------------------------
!     /*   initialize spline                                          */
!-----------------------------------------------------------------------

      call spline_init_string( arc_ref, s1_ref, s2_ref, narc, 1 )

      return
      end





!***********************************************************************
      subroutine spline_pot_string( pot_s, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : nbead

      use string_variables, only : s_ref, pot1_ref, pot2_ref

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: k

!     /*   real values   */
      real(8) :: s, pot_s

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      k = nbead/2

      call spline_hunt( s_ref(:,1), nbead, s, k )

      pot_s = spline_string_func &
     &      ( s_ref(:,1), pot1_ref(:,1), pot2_ref(:,1), s, k )

      return
      contains
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_crd_string( r, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom, nbead

      use string_variables, only : s_ref, r1_ref, r2_ref

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   mass weighted coordinates   */
      real(8), dimension(3*natom) :: r

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      k = nbead/2

      do i = 1, 3*natom

         call spline_hunt( s_ref(:,i), nbead, s, k )

         r(i) = spline_string_func &
     &            ( s_ref(:,i), r1_ref(:,i), r2_ref(:,i), s, k )

      end do

      return
      contains
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_frc_string( f, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom, nbead

      use string_variables, only : s_ref, f1_ref, f2_ref

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   forces   */
      real(8), dimension(3*natom) :: f

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      k = nbead/2

      do i = 1, 3*natom

         call spline_hunt( s_ref(:,i), nbead, s, k )

         f(i) = spline_string_func &
     &      ( s_ref(:,i), f1_ref(:,i), f2_ref(:,i), s, k )

      end do

      return
      contains
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_arc_string( s, arc )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use string_variables, only : arc_ref, s1_ref, s2_ref, narc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: k

!     /*   real values   */
      real(8) :: s, arc

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      k = narc/2

      call spline_hunt( arc_ref(:,1), narc, arc, k )

      s = spline_string_func &
     &       ( arc_ref(:,1), s1_ref(:,1), s2_ref(:,1), arc, k )

      return
      contains
      include 'spline_string_func.F90'
      end





!***********************************************************************
      subroutine spline_init_string ( x, y, y2, n, m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, n, m, k

      real(8) :: yp1, ypn, sig, p, qn, un

      real(8) :: x(n,m), y(n,m), y2(n,m), u(n)

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do j = 1, m

         yp1 = (y(2,j)-y(1,j))/(x(2,j)-x(1,j))
         ypn = (y(n,j)-y(n-1,j))/(x(n,j)-x(n-1,j))

         y2(1,j) = -0.5d0
         u(1) = (3.d0/(x(2,j)-x(1,j)))*((y(2,j)-y(1,j)) &
     &           /(x(2,j)-x(1,j))-yp1)

         do i = 2, n-1
            sig = (x(i,j)-x(i-1,j))/(x(i+1,j)-x(i-1,j))
            p = sig*y2(i-1,j)+2.d0
            y2(i,j) = (sig-1.d0)/p
            u(i) = (6.d0*((y(i+1,j)-y(i,j)) &
     &              /(x(i+1,j)-x(i,j))-(y(i,j)-y(i-1,j)) &
     &              /(x(i,j)-x(i-1,j)))/(x(i+1,j)-x(i-1,j)) &
     &              -sig*u(i-1))/p
         end do

         qn = 0.5d0
         un = (3.d0/(x(n,j)-x(n-1,j)))*(ypn-(y(n,j)-y(n-1,j)) &
     &       /(x(n,j)-x(n-1,j)))

         y2(n,j) = (un-qn*u(n-1))/(qn*y2(n-1,j)+1.d0)

         do k = n-1, 1, -1
            y2(k,j) = y2(k,j)*y2(k+1,j) + u(k)
         end do

      end do

      return
      end





!***********************************************************************
      subroutine spline_crd_frc_string( r, f, pot_s, s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead

      use string_variables, only : &
     &   s_ref, r1_ref, r2_ref, f1_ref, f2_ref, pot1_ref, pot2_ref

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k

!     /*   real values   */
      real(8) :: s

!     /*   mass weighted coordinates and forces   */
      real(8), dimension(3*natom) :: r, f

!     /*   real values   */
      real(8) :: pot_s

!-----------------------------------------------------------------------
!     /*   interpolation                                              */
!-----------------------------------------------------------------------

      k = nbead/2

      do i = 1, 3*natom

         call spline_hunt( s_ref(:,i), nbead, s, k )

         r(i) = spline_string_func &
     &             ( s_ref(:,i), r1_ref(:,i), r2_ref(:,i), s, k )

         f(i) = spline_string_func &
     &             ( s_ref(:,i), f1_ref(:,i), f2_ref(:,i), s, k )

      end do

         pot_s = spline_string_func &
     &             ( s_ref(:,1), pot1_ref(:,1), pot2_ref(:,1), s, k )

      return
      contains
      include 'spline_string_func.F90'
      end





!***********************************************************************
      SUBROUTINE spline_hunt(xx,n,x,jlo)
!***********************************************************************

      IMPLICIT NONE

      INTEGER,INTENT(INOUT) :: jlo
      INTEGER,INTENT(IN) :: n
      REAL(8),INTENT(IN) :: x
      REAL(8),INTENT(IN) :: xx(n)
      INTEGER :: inc,jhi,jm
      LOGICAL :: ascnd

      if ( x .lt. xx(1) ) then
         jlo = 1
      else if ( x .gt. xx(n) ) then
         jlo = n-1
      else

         ascnd = (xx(n) >= xx(1))
         if (jlo <= 0 .or. jlo > n) then
            jlo=0
            jhi=n+1
         else
            inc=1
            if (x >= xx(jlo) .eqv. ascnd) then
               do
                  jhi=jlo+inc
                  if (jhi > n) then
                     jhi=n+1
                     exit
                  else
                     if (x < xx(jhi) .eqv. ascnd) exit
                     jlo=jhi
                     inc=inc+inc
                  end if
               end do
            else
               jhi=jlo
               do
                  jlo=jhi-inc
                  if (jlo < 1) then
                     jlo=0
                     exit
                  else
                     if (x >= xx(jlo) .eqv. ascnd) exit
                     jhi=jlo
                     inc=inc+inc
                  end if
               end do
            end if
         end if
         do
            if (jhi-jlo <= 1) then
               if (x == xx(n)) jlo=n-1
               if (x == xx(1)) jlo=1
               exit
            else
               jm=(jhi+jlo)/2
               if (x >= xx(jm) .eqv. ascnd) then
                  jlo=jm
               else
                  jhi=jm
               end if
            end if
         end do
      end if

      END SUBROUTINE spline_hunt

