!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     scaled hypersphere search
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module shs_variables
!***********************************************************************

!     /*   shs positions   */
      real(8), dimension(:,:), allocatable :: q_shs

!     /*   shs forces   */
      real(8), dimension(:,:), allocatable :: fq_shs

!     /*   shs forces normal to hypersphere    */
      real(8), dimension(:), allocatable :: fn_shs

!     /*   shs velocities   */
      real(8), dimension(:,:), allocatable :: vq_shs

!     /*   old shs positions   */
      real(8), dimension(:,:), allocatable :: qold_shs

!     /*   old shs velocities   */
      real(8), dimension(:,:), allocatable :: vqold_shs

!     /*   shs step size   */
      real(8), dimension(:), allocatable :: dt_shs

!     /*   initial shs step size   */
      real(8) :: dtmax_shs = 0.050d0

!     /*   minimum shs step size   */
      real(8) :: dtmin_shs = 0.001d0

!     /*   shs to cartesian conversion matrix   */
      real(8), dimension(:,:), allocatable :: a_shs

!     /*   cartesian to shs conversion matrix   */
      real(8), dimension(:,:), allocatable :: ainv_shs

!     /*   active/inactive index of shs coordinates   */
      integer, dimension(:), allocatable :: imode_shs

!     /*   cut off frequency of active shs coordinates in cm^-1   */
      real(8) :: eigcut_shs = 10.d0

!     /*   harmonic energy at shs hypersphere   */
      real(8), dimension(:), allocatable :: edev_shs

!     /*   initial value of harmonic energy at shs hypersphere   */
      real(8) :: eini_shs  = 0.010d0

!     /*   increment of harmonic energy at shs hypersphere   */
      real(8) :: dedev_shs = 0.001d0

!     /*   potential energy at shs minimum   */
      real(8) :: potmin_shs

!     /*   shs hypersphere radius   */
      real(8), dimension(:), allocatable :: qdev_shs

!     /*   shs convergence flag   */
      integer, dimension(:), allocatable :: iconv_shs

!     /*   shs termination flag   */
      integer :: jconv_shs

!     /*   maximum value of potential energy at shs minima   */
      real(8), dimension(:), allocatable :: potmax_shs

!     /*   saved cartesian coordinates   */
      real(8), dimension(:,:), allocatable :: x_shs
      real(8), dimension(:,:), allocatable :: y_shs
      real(8), dimension(:,:), allocatable :: z_shs

!***********************************************************************
      end module shs_variables
!***********************************************************************





!***********************************************************************
      subroutine nma_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, pi, speedlight_SI, x, y, z, ux, uy, uz, potential, &
     &   au_time, au_length, hessian, natom, nbead, iounit

      use nma_variables, only : &
     &   hess, redmass, eigval, eigvec

      use shs_variables, only : &
     &   x_shs, y_shs, z_shs, potmin_shs, potmax_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: i, j, k, l, ij, kl, m, ierr, itest, jtest, ktest

!     /*   real numbers   */
      real(8) :: factor, factor_cminv, sum

!-----------------------------------------------------------------------
!     /*   restart files                                              */
!-----------------------------------------------------------------------

!     /*   check existence of shs restart file   */
      call testfile ( 'shs.ini', 7, itest )

!     /*   check existence of geometry restart file   */
      call testfile ( 'geometry.ini', 11, jtest )

!     /*   check existence of hessian file   */
      call testfile ( 'hessian.out', 11, ktest )

!-----------------------------------------------------------------------
!     /*   last position                                              */
!-----------------------------------------------------------------------

!     /*   if geometry restart file exists   */
      if ( jtest .eq. 0 ) then

!        /*   read cartesian coordinates of last shs minimum   */
         call restart_shs( 1 )

!        /*   substitute them to current cartesian coordinates   */

         do j = 1, nbead
         do i = 1, natom
            x(i,j) = x_shs(i,j)
            y(i,j) = y_shs(i,j)
            z(i,j) = z_shs(i,j)
         end do
         end do

!     /*   if geometry restart file does not exist   */
      else

!        //   read centroid coordinates
         call init_centroid

!        /*   substitute them to current cartesian for all beads  */

         do m = 1, nbead
            x(:,m) = ux(:,1)
            y(:,m) = uy(:,1)
            z(:,m) = uz(:,1)
         end do

!     /*   if geometry restart file exists or not   */
      end if

!-----------------------------------------------------------------------
!     /*   position, potential, hessian of sphere center              */
!-----------------------------------------------------------------------

!     /*   if shs restart file exists   */
      if ( itest .eq. 0 ) then

!        /*   read data of last shs minimum   */
         call restart_shs( 5 )

!        /*   potential at hypersphere center   */
         potmin_shs = potential

!        /*   maximum value of potential energy at shs minima   */
         potmax_shs(:) = potential

!     /*   if shs restart file does not exist   */
      else

!        /*   calculate potential at shs minima   */
         call getforce

!        /*   potential at hypersphere center   */
         potmin_shs = potential

!        /*   maximum value of potential energy at shs minima   */
         potmax_shs(:) = potential

!        /*   if hessian file exists   */
         if ( ktest .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'hessian.out' )

!           /*   read hessian   */

            do i = 1, 3*natom
            do j = 1, 3*natom
               read( iounit, *, iostat=ierr ) k, l, hess(i,j)
            end do
            end do

!           /*   file close   */
            close( iounit )

!           /*   error handling   */
            call error_handling ( ierr, 'subroutine nma_shs', 18 )

!        /*   if hessian file does not exist   */
         else

!           /*   calculate hessian   */
            call gethess

!           /*   use hessian of the first bead   */
            hess(:,:) = hessian(:,:,1)

!        /*   if hessian file exists or not   */
         end if

!     /*   if shs restart file exists or not   */
      end if

!-----------------------------------------------------------------------
!     /*   symmetrize hessian matrix                                  */
!-----------------------------------------------------------------------

      do i = 1, 3*natom
      do j = i, 3*natom

         hess(i,j) = 0.5d0 * ( hess(i,j) + hess(j,i) )
         hess(j,i) = hess(i,j)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   project out translation and rotation                       */
!-----------------------------------------------------------------------

      call project_out_nma( 1 )

!-----------------------------------------------------------------------
!     /*   mass weighted hessian                                      */
!-----------------------------------------------------------------------

      ij = 0

      do i = 1, natom
      do j = 1, 3

         ij = ij + 1

         kl = 0

         do k = 1, natom
         do l = 1, 3

            kl = kl + 1

            factor = 1.d0 / sqrt( physmass(i) * physmass(k) )

            hess(ij,kl) = hess(ij,kl) * factor

         end do
         end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   diagonalize mass weighted hessian                          */
!-----------------------------------------------------------------------

      call ddiag ( hess, eigval, eigvec, 3*natom )

!-----------------------------------------------------------------------
!     /*   harmonic frequencies (convert imaginary to negative)       */
!-----------------------------------------------------------------------

      do i = 1, 3*natom

         if ( eigval(i) .gt. 0.d0 ) then

            eigval(i) = + sqrt(+eigval(i))

         else

            eigval(i) = - sqrt(-eigval(i))

         end if

      end do

!-----------------------------------------------------------------------
!     /*   harmonic frequencies in wave numbers [cm^-1]               */
!-----------------------------------------------------------------------

      factor_cminv = 2.d0 * pi * speedlight_SI * au_time * 100.d0

      do i = 1, 3*natom
         eigval(i) = eigval(i) / factor_cminv
      end do

!-----------------------------------------------------------------------
!     /*   normal modes                                               */
!-----------------------------------------------------------------------

!     /*   multiply mass factor   */

      k = 0

      do i = 1, natom

         k = k + 1
         eigvec(k,:) = eigvec(k,:) / sqrt(physmass(i))

         k = k + 1
         eigvec(k,:) = eigvec(k,:) / sqrt(physmass(i))

         k = k + 1
         eigvec(k,:) = eigvec(k,:) / sqrt(physmass(i))

      end do

!     /*   normalize   */

      do j = 1, 3*natom

         sum = 0.d0

         do i = 1, 3*natom
            sum = sum + eigvec(i,j) * eigvec(i,j)
         end do

         if ( sum .ne. 0.d0 ) then
            redmass(j) = 1.d0 / sum / 1822.88853d0
         else
            redmass(j) = 0.d0
         end if

         sum = 1.d0 / sqrt(sum)

         do i = 1, 3*natom
            eigvec(i,j) = eigvec(i,j) * sum
         end do

      end do

      return
      end





!***********************************************************************
      subroutine shscycle
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

      use shs_variables, only : &
     &   jconv_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     //   starting step
      istep = istep_start

!     //   current step
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce

!     /*   cartesian forces -> shs forces   */
      call nm_trans_force_shs( 1 )

!     /*   projection of forces   */
      call project_fq_shs

!     /*   update qdev   */
      call update_qdev_shs

!     /*   standard output   */
      call standard_shs

!     /*   main loop   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update position   */
         call update_q_shs

!        /*   shs position -> cartesian position   */
         call nm_trans_shs( 0 )

!        /*   get interatomic forces   */
         call getforce

!        /*   cartesian forces -> shs forces   */
         call nm_trans_force_shs( 1 )

!        /*   projection of forces   */
         call project_fq_shs

!        /*   update qdev   */
         call update_qdev_shs

!        /*   standard output   */
         call standard_shs

!        /*   save data   */
         call backup_shs

!        /*   exit by termination flag   */
         if ( jconv_shs .eq. 1 ) exit

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   main loop   */
      end do

      return
      end





!***********************************************************************
      subroutine update_qdev_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, natom, nbead

      use shs_variables, only : &
     &   qdev_shs, edev_shs, dedev_shs, dt_shs, dtmax_shs, potmax_shs, &
     &   x_shs, y_shs, z_shs, iconv_shs, jconv_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: l, m

!-----------------------------------------------------------------------
!     /*   update shs harmonic energy                                 */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   at shs minimum   */
         if ( iconv_shs(m) .eq. 2 ) then

!           /*   if shs potential minimum was increased   */
            if ( pot(m) .gt. potmax_shs(m) ) then

!              /*   reset convergence flag   */
               iconv_shs(m) = 0

!              /*   maximum value of shs potential minimum   */
               potmax_shs(m) = pot(m)

!              /*   position of shs potential minimum   */
               do l = 1, natom
                  x_shs(l,m) = x(l,m)
                  y_shs(l,m) = y(l,m)
                  z_shs(l,m) = z(l,m)
               end do

!              /*   update shs harmonic energy   */
               edev_shs(m) = edev_shs(m) + dedev_shs

!              /*   update shs radius   */
               qdev_shs(m) = sqrt( 2.d0 * edev_shs(m) )

!              /*   reset step size   */
               dt_shs(m) = dtmax_shs

!           /*   if potential was decreased   */
            else

!              /*   finish the calculation for this bead   */
               iconv_shs(m) = 3

            end if

!        /*   at shs minimum   */
         end if

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   norm of q is scaled                                        */
!-----------------------------------------------------------------------

      call scale_q_shs

!-----------------------------------------------------------------------
!     /*   shs coordinates -> cartesian coordinates                   */
!-----------------------------------------------------------------------

      call nm_trans_shs( 0 )

!-----------------------------------------------------------------------
!     /*   termination flag                                           */
!-----------------------------------------------------------------------

!     /*   count number of finished bead   */

      l = 0

      do m = 1, nbead
         if ( iconv_shs(m) .eq. 3 ) l = l + 1
      end do

!     /*   terminate if all is finished   */
      if ( l .eq. nbead ) jconv_shs = 1

      return
      end





!***********************************************************************
      subroutine backup_shs
!***********************************************************************
!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep_end, iprint_rest, iounit, nstep

      implicit none

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( mod(istep_end,iprint_rest) .eq. 0 ) then
         if ( iprint_rest .le. 0 ) then
            return
         else
            continue
         end if
      else if ( istep_end .eq. nstep ) then
         continue
      else
         return
      end if

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

!     /*   write geometry.ini   */
      call restart_shs( 2 )

!     /*   write shs.ini   */
      call restart_shs( 6 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
      close( iounit )

      return
      end





!***********************************************************************
      subroutine standard_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, istep, nbead, iounit, char_date, iounit_std, iprint_std

      use shs_variables, only : &
     &   qdev_shs, potmin_shs, fn_shs, iconv_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: m, itest

!     /*   integers   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '  step s   rad  potential [au]   dpot [au]   force  ' // &
     &      'wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '  step s   rad  potential [au]    dev [au]   force  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '  step s   rad  potential [au]    dev [au]   force  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   print output                                               */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

         do m = 1, nbead

            write( 6, '(i6,i2,f6.3,f16.8,f12.8,f8.3,2x,a26)' ) &
     &         istep, iconv_shs(m), qdev_shs(m), &
     &         pot(m), pot(m)-potmin_shs, fn_shs(m), char_date

         end do

      end if

      return
      end

!***********************************************************************
      subroutine restart_shs ( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file     irw = 1 :   read bead positions
!
!                                 irw = 2 :   write bead positions
!
!                                 irw = 3 :   read minimum position
!
!                                 irw = 4 :   write minimum position
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, potential, iounit, natom, nbead, istep_end, mbox

      use shs_variables, only : &
     &   x_shs, y_shs, z_shs

      use nma_variables, only : &
     &   hess

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer:: irw, i, j, k, l, m

!     /*   real numbers   */
      real(8) :: dummy

!-----------------------------------------------------------------------
!     /*   read bead positions                                        */
!-----------------------------------------------------------------------

!     /*   option   */
      if ( irw .eq. 1 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do j = 1, nbead
         do i = 1, natom
            read( iounit, * ) k, x_shs(i,j), y_shs(i,j), z_shs(i,j), &
     &                        dummy, dummy, dummy, mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!        /*   current step   */
         istep_end = k

!-----------------------------------------------------------------------
!     /*   write bead positions                                       */
!-----------------------------------------------------------------------

!     /*   option   */
      else if ( irw .eq. 2 ) then

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   write   */
         do j = 1, nbead
         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         k, x_shs(i,j), y_shs(i,j), z_shs(i,j), &
     &         0.d0, 0.d0, 0.d0, mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   read minimum position                                      */
!-----------------------------------------------------------------------

!     /*   option   */
      else if ( irw .eq. 3 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do i = 1, natom
            read( iounit, * ) k, ux(i,1), uy(i,1), uz(i,1)
         end do

!        /*   close file   */
         close( iounit )

!        /*   current step   */
         istep_end = k

!-----------------------------------------------------------------------
!     /*   write minimum position                                     */
!-----------------------------------------------------------------------

!     /*   option   */
      else if ( irw .eq. 4 ) then

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   write   */
         do i = 1, natom
            write( iounit, '(i8,3e24.16)' ) k, ux(i,1), uy(i,1), uz(i,1)
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   real minimum position, potential, hessian                  */
!-----------------------------------------------------------------------

!     /*   option   */
      else if ( irw .eq. 5 ) then

!        /*   open file   */
         open ( iounit, file = 'shs.ini', status = 'unknown' )

!        /*   read minimum position   */

         do i = 1, natom
            read( iounit, * ) k, ux(i,1), uy(i,1), uz(i,1)
         end do

!        /*   write minimum energy   */

         do i = 1, natom
            read( iounit, '(e24.16)' ) potential
         end do

!        /*   read hessian   */

         do i = 1, 3*natom
         do j = 1, 3*natom
             read( iounit, * ) l, m, hess(i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!        /*   current step   */
         istep_end = k

!-----------------------------------------------------------------------
!     /*   write minimum position, potential, hessian                 */
!-----------------------------------------------------------------------

!     /*   option   */
      else if ( irw .eq. 6 ) then

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'shs.ini', status = 'unknown' )

!        /*   write minimum position   */

         do i = 1, natom
            write( iounit, '(i8,3e24.16)' ) k, ux(i,1), uy(i,1), uz(i,1)
         end do

!        /*   write minimum energy   */

         do i = 1, natom
            write( iounit, '(e24.16)' ) potential
         end do

!        /*   write hessian   */

         do i = 1, 3*natom
         do j = 1, 3*natom
             write( iounit, '(2i6,e24.16)' ) i, j, hess(i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!     /*   option   */
      else

!        /*   error handling   */
         call error_handling ( 1, 'subroutine restart_shs', 22 )

!     /*   option   */
      end if

      return
      end




!***********************************************************************
      subroutine setup_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead, iounit, ipos_start

      use shs_variables, only : &
     &   fq_shs, q_shs, a_shs, ainv_shs, eigcut_shs, edev_shs, dt_shs, &
     &   qdev_shs, dtmax_shs, fn_shs, vq_shs, qold_shs, vqold_shs, &
     &   eini_shs, potmax_shs, x_shs, y_shs, z_shs, imode_shs, &
     &   iconv_shs, jconv_shs

      use nma_variables, only : &
     &   hess, redmass, eigval, eigvec

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: l, itest

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   hessian   */
      if ( .not. allocated( hess ) ) &
     &   allocate( hess(3*natom,3*natom) )

!     /*   reduced masss   */
      if ( .not. allocated( redmass ) ) &
     &   allocate( redmass(3*natom) )

!     /*   eigenvalues   */
      if ( .not. allocated( eigval ) ) &
     &   allocate( eigval(3*natom) )

!     /*   eigenvectors    */
      if ( .not. allocated( eigvec ) ) &
     &   allocate( eigvec(3*natom,3*natom) )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   shs positions   */
      if ( .not. allocated( q_shs ) ) &
     &   allocate( q_shs(3*natom,nbead) )

!     /*   shs forces   */
      if ( .not. allocated( fq_shs ) ) &
     &   allocate( fq_shs(3*natom,nbead) )

!     /*   shs forces normal to hypersphere   */
      if ( .not. allocated( fn_shs ) ) &
     &   allocate( fn_shs(nbead) )

!     /*   shs step size   */
      if ( .not. allocated( dt_shs ) ) &
     &   allocate( dt_shs(nbead) )

!     /*   shs to cartesian conversion matrix   */
      if ( .not. allocated( a_shs ) ) &
     &   allocate( a_shs(3*natom,3*natom) )

!     /*   cartesian to shs conversion matrix   */
      if ( .not. allocated( ainv_shs ) ) &
     &   allocate( ainv_shs(3*natom,3*natom) )

!     /*   active shs coordinates   */
      if ( .not. allocated( imode_shs ) ) &
     &   allocate( imode_shs(3*natom) )

!     /*   shs velocities   */
      if ( .not. allocated( vq_shs ) ) &
     &   allocate( vq_shs(3*natom,nbead) )

!     /*   old shs positions   */
      if ( .not. allocated( qold_shs ) ) &
     &   allocate( qold_shs(3*natom,nbead) )

!     /*   old shs velocities   */
      if ( .not. allocated( vqold_shs ) ) &
     &   allocate( vqold_shs(3*natom,nbead) )

!     /*   shs convergence flag   */
      if ( .not. allocated( iconv_shs ) ) &
     &   allocate( iconv_shs(nbead) )

!     /*   shs hypersphere radius   */
      if ( .not. allocated( qdev_shs ) ) &
     &   allocate( qdev_shs(nbead) )

!     /*   harmonic energy at shs hypersphere   */
      if ( .not. allocated( edev_shs ) ) &
     &   allocate( edev_shs(nbead) )

!     /*   maximum value of shs potential minimum  */
      if ( .not. allocated( potmax_shs ) ) &
     &   allocate( potmax_shs(nbead) )

!     /*   saved coordinates   */
      if ( .not. allocated( x_shs ) ) &
     &   allocate( x_shs(natom,nbead) )
      if ( .not. allocated( y_shs ) ) &
     &   allocate( y_shs(natom,nbead) )
      if ( .not. allocated( z_shs ) ) &
     &   allocate( z_shs(natom,nbead) )

!-----------------------------------------------------------------------
!     /*   normal mode analysis                                       */
!-----------------------------------------------------------------------

      call nma_shs

!-----------------------------------------------------------------------
!     /*   initial harmonic energy at shs hypersphere                 */
!-----------------------------------------------------------------------

      do l = 1, nbead
         edev_shs(l) = eini_shs
      end do

!-----------------------------------------------------------------------
!     /*   initial hypersphere radius                                 */
!-----------------------------------------------------------------------

      do l = 1, nbead
         qdev_shs(l) = sqrt( 2.d0 * edev_shs(l) )
      end do

!-----------------------------------------------------------------------
!     /*   check eigenvalues                                          */
!-----------------------------------------------------------------------

!     /*   loop of shs space   */
      do l = 1, 3*natom

!        /*   if eigenvalue is small or negative   */
         if ( eigval(l) .lt. eigcut_shs ) then

!           /*   the shs coordinate is inactive   */
            imode_shs(l) = 0

!        /*   if eigenvalue is large   */
         else

!           /*   the shs coordinate is active   */
            imode_shs(l) = 1

!        /*   if eigenvalue is large or small   */
         end if

!     /*   loop of shs space   */
      end do

!-----------------------------------------------------------------------
!     /*   mode matrix                                                */
!-----------------------------------------------------------------------

      call nm_matrix_shs

!-----------------------------------------------------------------------
!     /*   check existence of shs restart file                        */
!-----------------------------------------------------------------------

!     /*   check shs.ini   */
      call testfile ( 'shs.ini', 7, itest )

!     /*   if exists   */
      if ( itest .eq. 0 ) then

!        /*   flag set to one  */
         ipos_start = 1

!     /*   if shs.ini does not exist   */
      else

!        /*   flag set to zero  */
         ipos_start = 0

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   if shs.ini exists   */
      if     ( ipos_start .eq. 0 ) then

!        /*   random deviation of q   */
         call init_random_shs

!        /*   q -> x, y, z   */
         call nm_trans_shs( 0 )

!     /*   if shs.ini does not exist   */
      else if ( ipos_start .eq. 1 ) then

!        /*   read positions   */
         call restart_shs( 1 )

!        /*   x, y, z -> q   */
         call nm_trans_shs( 1 )

!     /*   otherwise   */
      else

!        /*   error termination   */
         call error_handling( 1, 'subroutine setup_shs', 20 )

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     //   initialize convergence flag
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do l = 1, nbead

!        /*   convergence flag   */
         iconv_shs(l) = 0

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     //   initialize termination flag
!-----------------------------------------------------------------------

      jconv_shs = 0

!-----------------------------------------------------------------------
!     //   initialize step size
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do l = 1, nbead

!        /*   all step size set to initial value   */
         dt_shs(l) = dtmax_shs

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine update_q_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : pi, natom, nbead

      use shs_variables, only : &
     &   fq_shs, q_shs, vq_shs, qold_shs, vqold_shs, dt_shs, &
     &   dtmin_shs, iconv_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: l, m

!     /*   norm of mode force   */
      real(8) :: abs_fq, aa, ab, bb, theta

!-----------------------------------------------------------------------
!     /*   initial setting                                            */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   skip finished bead   */
         if ( iconv_shs(m) .eq. 3 ) cycle

!        /*   save old position   */

         do l = 1, 3*natom
            qold_shs(l,m) = q_shs(l,m)
         end do

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   update position                                            */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   skip finished bead   */
         if ( iconv_shs(m) .eq. 3 ) cycle

!        /*   calculate |fq|   */

         abs_fq = 0.d0

         do l = 1, 3*natom
            abs_fq = abs_fq + fq_shs(l,m) * fq_shs(l,m)
         end do

         abs_fq = sqrt( abs_fq )

!        /*   update q = q + fq/|fq|*dt   */

         do l = 1, 3*natom
            q_shs(l,m) = q_shs(l,m) + fq_shs(l,m) / abs_fq * dt_shs(m)
         end do

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   norm of q is scaled                                        */
!-----------------------------------------------------------------------

      call scale_q_shs

!-----------------------------------------------------------------------
!     /*   velocity                                                   */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   skip finished bead   */
         if ( iconv_shs(m) .eq. 3 ) cycle

!        /*   velocity = q - qold   */
         vq_shs(:,m) = q_shs(:,m) - qold_shs(:,m)

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   convergence flag                                           */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   skip finished bead   */
         if ( iconv_shs(m) .eq. 3 ) cycle

!        /*   initial step   */
         if ( iconv_shs(m) .eq. 0 ) then

!           /*   save velocity   */

            do l = 1, 3*natom
               vqold_shs(l,m) = vq_shs(l,m)
            end do

!        /*   unconverged step   */
         else if ( iconv_shs(m) .eq. 1 ) then

!           /*   angle theta   */

            aa = 0.d0
            ab = 0.d0
            bb = 0.d0

            do l = 1, 3*natom
               aa = aa + vq_shs(l,m) * vq_shs(l,m)
               ab = ab + vq_shs(l,m) * vqold_shs(l,m)
               bb = bb + vqold_shs(l,m) * vqold_shs(l,m)
            end do

            theta = acos( ab / sqrt(aa*bb) ) * 180.d0 / pi

!           /*   diminish step size if theta is small   */

            if ( theta .ge. 90.d0 ) then
               dt_shs(m) = dt_shs(m) * 0.70d0
            end if

!           /*   save old velocity   */

            do l = 1, 3*natom
               vqold_shs(l,m) = vq_shs(l,m)
            end do

!        /*   converged step   */
         else

!           /*   set step size   */
            dt_shs(m) = 0.d0

         end if

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   convergence flag                                           */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   skip finished bead   */
         if ( iconv_shs(m) .eq. 3 ) cycle

!        /*   if dt is large   */
         if ( dt_shs(m) .gt. dtmin_shs ) then

!           /*   unconverged flag   */
            iconv_shs(m) = 1

!        /*   if dt is large   */
         else

!           /*   converged flag   */
            iconv_shs(m) = 2

         end if

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine project_fq_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom, nbead

      use shs_variables, only : fq_shs, q_shs, fn_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: l, m

!     /*   real numbers   */
      real(8) :: qq, fq

!-----------------------------------------------------------------------
!     /*   project out mode forces along q vector                     */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   calculate fq = (f,q)   */

         fq = 0.d0

         do l = 1, 3*natom
            fq = fq + fq_shs(l,m) * q_shs(l,m)
         end do

!        /*   calculate qq = (q,q)   */

         qq = 0.d0

         do l = 1, 3*natom
            qq = qq + q_shs(l,m) * q_shs(l,m)
         end do

!        /*   calculate fn = (f,q)/|q|   */

         fn_shs(m) = fq / sqrt(qq)

!        /*   projected fq = fq - q (f,q)/(q,q)   */

         do l = 1, 3*natom
            fq_shs(l,m) = fq_shs(l,m) - fq / qq * q_shs(l,m)
         end do

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine init_random_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom, nbead

      use shs_variables, only : q_shs, imode_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: l, m

!     /*   real numbers   */
      real(8) :: ranf1, randomno

!-----------------------------------------------------------------------
!     /*   random shift from minimum                                  */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   loop of shs coordinates   */
         do l = 1, 3*natom

!           /*   inactive modes   */
            if ( imode_shs(l) .eq. 0 ) then

!              /*   shs coordinates set to zero   */
               q_shs(l,m) = 0.d0

!           /*   active modes   */
            else

!              /*   uniform random number   */
               randomno = ranf1()

!              /*   shs coordinates set to random values   */
               q_shs(l,m) = 2.d0*randomno - 1.d0

!           /*   modes   */
            end if

!        /*   loop of shs coordinates   */
         end do

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   norm of q is scaled                                        */
!-----------------------------------------------------------------------

      call scale_q_shs

      return
      end





!***********************************************************************
      subroutine scale_q_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom, nbead

      use shs_variables, only : q_shs, qdev_shs, imode_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: l, m

!     /*   real numbers   */
      real(8) :: qnorm, factor

!-----------------------------------------------------------------------
!     /*   scale q such that q is on hypersphere                      */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   inactive shs coordinates set to zero   */

         do l = 1, 3*natom
            if ( imode_shs(l) .eq. 0 ) q_shs(l,m) = 0.d0
         end do

!        /*   calculate norm of shs coordinates   */

         qnorm = 0.d0

         do l = 1, 3*natom
            qnorm = qnorm + q_shs(l,m) * q_shs(l,m)
         end do

         qnorm = sqrt( qnorm )

!        /*   scale q such that norm is qdev   */

         factor = qdev_shs(m) / qnorm

         do l = 1, 3*natom
            q_shs(l,m) = q_shs(l,m) * factor
         end do

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine nm_matrix_shs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, speedlight_SI, au_time, physmass, natom

      use shs_variables, only : &
     &   a_shs, ainv_shs

      use nma_variables, only : &
     &   eigval, eigvec, redmass

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: i, k, l

!     /*   real numbers   */
      real(8) :: factor_1, factor_2, factor_3, factor_4, factor_cminv

!-----------------------------------------------------------------------
!     /*   conversion from cm-1 to hartree                            */
!-----------------------------------------------------------------------

      factor_cminv = 2.d0 * pi * speedlight_SI * au_time * 100.d0

!-----------------------------------------------------------------------
!     /*   make a and ainv matrices                                   */
!-----------------------------------------------------------------------

!     /*   loop of shs space   */
      do l = 1, 3*natom

!        /*   factor: reduced mass from amu to au   */
         factor_1 = sqrt( 1.d0 / redmass(l) / 1822.88853d0 )

!        /*   factor: eigenvalue from cm-1 to hartree  */
         factor_2 = 1.d0 / ( eigval(l) * factor_cminv )

!        /*   factor   */
         factor_3 = factor_1 * factor_2

!        /*   factor   */
         factor_4 = factor_1 / factor_2

!        /*   cartesian index   */
         k = 0

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   cartesian index   */
            k = k + 1

!           /*   a: matrix converting from x, y, z to q   */
            a_shs(k,l)    = eigvec(k,l) * factor_3

!           /*   a: matrix converting from q to x, y, z   */
            ainv_shs(l,k) = eigvec(k,l) * factor_4 * physmass(i)

!           /*   cartesian index   */
            k = k + 1

!           /*   a: matrix converting from x, y, z to q   */
            a_shs(k,l)    = eigvec(k,l) * factor_3

!           /*   a: matrix converting from q to x, y, z   */
            ainv_shs(l,k) = eigvec(k,l) * factor_4 * physmass(i)

!           /*   cartesian index   */
            k = k + 1

!           /*   a: matrix converting from x, y, z to q   */
            a_shs(k,l)    = eigvec(k,l) * factor_3

!           /*   a: matrix converting from q to x, y, z   */
            ainv_shs(l,k) = eigvec(k,l) * factor_4 * physmass(i)

!        /*   loop of atoms   */
         end do

!     /*   loop of shs space   */
      end do

      return
      end





!***********************************************************************
      subroutine nm_trans_shs( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, speedlight_SI, au_time, x, y, z, ux, uy, uz, &
     &   natom, nbead

      use shs_variables, only : &
     &   q_shs, a_shs, ainv_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: ioption, i, k, l, m

!     /*   real numbers   */
      real(8) :: dx, dy, dz

!-----------------------------------------------------------------------
!     /*   q -> x, y, z                                               */
!-----------------------------------------------------------------------

!     /*   option   */
      if ( ioption .eq. 0 ) then

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   cartesian coordinates at hypersphere center   */
            x(:,m) = ux(:,1)
            y(:,m) = uy(:,1)
            z(:,m) = uz(:,1)

!           /*   loop of shs space   */
            do l = 1, 3*natom

!              /*   cartesian index   */
               k = 0

!              /*   loop of atoms   */
               do i = 1, natom

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift cartesian coordinates   */
                  x(i,m) = x(i,m) + a_shs(k,l) * q_shs(l,m)

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift cartesian coordinates   */
                  y(i,m) = y(i,m) + a_shs(k,l) * q_shs(l,m)

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift cartesian coordinates   */
                  z(i,m) = z(i,m) + a_shs(k,l) * q_shs(l,m)

!              /*   loop of atoms   */
               end do

!           /*   loop of shs space   */
            end do

!        /*   loop of beads   */
         end do

!     /*   option   */
      end if

!-----------------------------------------------------------------------
!     /*   x, y, z -> q                                               */
!-----------------------------------------------------------------------

!     /*   option   */
      if ( ioption .eq. 1 ) then

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   loop of shs space   */
            do l = 1, 3*natom

!              /*   initialize shs coordinates   */
               q_shs(l,m) = 0.d0

!              /*   cartesian index   */
               k = 0

!              /*   loop of atoms   */
               do i = 1, natom

!                 /*   x deviation from minimum   */
                  dx = x(i,m) - ux(i,1)

!                 /*   y deviation from minimum   */
                  dy = y(i,m) - uy(i,1)

!                 /*   z deviation from minimum   */
                  dz = z(i,m) - uz(i,1)

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift shs coordinates   */
                  q_shs(l,m) = q_shs(l,m) + ainv_shs(l,k) * dx

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift shs coordinates   */
                  q_shs(l,m) = q_shs(l,m) + ainv_shs(l,k) * dy

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift shs coordinates   */
                  q_shs(l,m) = q_shs(l,m) + ainv_shs(l,k) * dz

!              /*   loop of atoms   */
               end do

!           /*   loop of shs space   */
            end do

!        /*   loop of beads   */
         end do

!     /*   option   */
      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_force_shs( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, speedlight_SI, au_time, fx, fy, fz, natom, nbead

      use shs_variables, only : &
     &   fq_shs, a_shs, ainv_shs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: ioption, i, k, l, m

!-----------------------------------------------------------------------
!     /*   fq -> fx, fy, fz                                           */
!-----------------------------------------------------------------------

!     /*   option   */
      if ( ioption .eq. 0 ) then

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   initialize cartesian forces   */
            fx(:,m) = 0.d0
            fy(:,m) = 0.d0
            fz(:,m) = 0.d0

!           /*   loop of shs space   */
            do l = 1, 3*natom

!              /*   cartesian index   */
               k = 0

!              /*   loop of atoms   */
               do i = 1, natom

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift cartesian forces   */
                  fx(i,m) = fx(i,m) + ainv_shs(l,k) * fq_shs(l,m)

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift cartesian forces   */
                  fy(i,m) = fy(i,m) + ainv_shs(l,k) * fq_shs(l,m)

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift cartesian forces   */
                  fz(i,m) = fz(i,m) + ainv_shs(l,k) * fq_shs(l,m)

!              /*   loop of atoms   */
               end do

!           /*   loop of modes   */
            end do

!        /*   loop of beads   */
         end do

!     /*   option   */
      end if

!-----------------------------------------------------------------------
!     /*   fx, fy, fz -> fq                                           */
!-----------------------------------------------------------------------

!     /*   option   */
      if ( ioption .eq. 1 ) then

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   loop of shs space   */
            do l = 1, 3*natom

!              /*   initialize shs forces   */
               fq_shs(l,m) = 0.d0

!              /*   cartesian index   */
               k = 0

!              /*   loop of atoms   */
               do i = 1, natom

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift shs forces   */
                  fq_shs(l,m) = fq_shs(l,m) + a_shs(k,l) * fx(i,m)

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift shs forces   */
                  fq_shs(l,m) = fq_shs(l,m) + a_shs(k,l) * fy(i,m)

!                 /*   cartesian index   */
                  k = k + 1

!                 /*   shift shs forces   */
                  fq_shs(l,m) = fq_shs(l,m) + a_shs(k,l) * fz(i,m)

!              /*   loop of atoms   */
               end do

!           /*   loop of shs space   */
            end do

!        /*   loop of beads   */
         end do

!     /*   option   */
      end if

      return
      end
