!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     temperature accelerated sliced sampling
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_tass_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, ipos_start, ibath_start

      use tass_variables, only : &
     &   icv_start, ihills_start, icons_start

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   read input file and memory allocation                      */
!-----------------------------------------------------------------------

      call setmdparams_MPI

      call setup_tass_param_MPI

!-----------------------------------------------------------------------
!     /*   atomic mass                                                */
!-----------------------------------------------------------------------

      call init_mass_md_MPI

!-----------------------------------------------------------------------
!     /*   atom position                                              */
!-----------------------------------------------------------------------

      if      ( ipos_start .eq. 0 ) then

!        /*   initialize position   */
         call init_position_cart_MPI

!        /*   ux --> x   */
         call nm_trans_cart( 2 )

      else if ( ipos_start .eq. 1 ) then

!        /*   restart position   */
         call restart_position_MPI( 2 )

      else

!        /*   check error   */
         call error_handling_MPI( 1, 'subroutine setup_tass_MPI', 25 )

      end if

!-----------------------------------------------------------------------
!     /*   atom velocity                                              */
!-----------------------------------------------------------------------

      if      ( ivel_start .eq. 0 ) then

!        /*   initialize velocity   */
         call init_velocity_MPI

      else if ( ivel_start .eq. 1 ) then

!        /*   restart velocity   */
         call restart_velocity_MPI( 2 )

      else

!        /*   check error   */
         call error_handling_MPI( 1, 'subroutine setup_tass_MPI', 25 )

      end if

!-----------------------------------------------------------------------
!     /*   thermostat masses                                          */
!-----------------------------------------------------------------------

      call setup_qmass_tass

!-----------------------------------------------------------------------
!     /*   position and velocity of all thermostats                   */
!-----------------------------------------------------------------------

      if ( ibath_start .le. 0 ) then

!        /*   initialize thermostats   */
         call init_bath_tass_MPI

      else if ( ibath_start .eq. 1 ) then

!        /*   restart thermostats   */
         call restart_bath_tass_MPI( 1 )

      end if

!-----------------------------------------------------------------------
!     /*   actual position of cv                                      */
!-----------------------------------------------------------------------

      call get_cv_tass_MPI

!-----------------------------------------------------------------------
!     /*   initialize/restart cv                                      */
!-----------------------------------------------------------------------

      if      ( icv_start .le. 0 ) then

!        /*   initialize cv   */
         call init_cv_tass_MPI

      else if ( icv_start .eq. 1 ) then

!        /*   restart cv   */
         call restart_cv_tass_MPI( 1 )

      end if

!-----------------------------------------------------------------------
!     /*   initialize/restart hills                                   */
!-----------------------------------------------------------------------

      if      ( ihills_start .le. 0 ) then

!        /*   initialize hills   */
         call init_hills_tass_MPI

      else if ( ihills_start .eq. 1 ) then

!        /*   initialize hills   */
         call restart_hills_tass_MPI( 1 )

      end if

!-----------------------------------------------------------------------
!     /*   initialize/restart constraints                             */
!-----------------------------------------------------------------------

      if      ( icons_start .le. 0 ) then

!        /*   initialize hills   */
         call init_cons_tass_MPI

      else if ( icons_start .eq. 1 ) then

!        /*   initialize hills   */
         call restart_cons_tass_MPI( 1 )

      end if

      return
      end





!***********************************************************************
      subroutine setup_tass_param_MPI
!***********************************************************************
!=======================================================================
!
!     read parameters for tass
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_time, dt, pi, boltz, beta, gkt, temperature, iounit, natom, &
     &   nbead, ncolor, nnhc, istep_start, myrank

      use tass_variables, only : &
     &   rtass, vtass, stass, fictmass_tass, rbath_tass, fbath_tass, &
     &   vbath_tass, fref_tass, fc_tass, pot_tass, ftass, gc_meta, &
     &   gh_meta, gw_meta, gw_meta_type, gw_meta_save, gc_meta_save, &
     &   gh_meta_save, time_limit_meta, pot_ref_tass, time_tass_save, &
     &   qmass_tass, req_tass, params_rec_meta, dtemp_meta, &
     &   fx_tass, fy_tass, fz_tass, fux_tass, fuy_tass, fuz_tass, &
     &   omega_cv_bath, time_cv_bath, time_cv_meta, time_fc_meta, &
     &   dt_ref_tass, time_cv_meta, dmeta_max, params_meta, gkt_tass, &
     &   temp_tamd_tass, params_tass, stass_avg, fref_tass_avg, &
     &   ntass_rec, mu_tass, nu_tass, ntass_meta, ntass_afed, &
     &   ntass_cons, i_tass, j_tass, k_tass, l_tass, idim_rec_tass, &
     &   nref_tass, iprint_rec_tass, ihills_start, itype_tass, &
     &   icv_start, ntype_tass, mg_meta, ipbc_tass, ntass, ikind_tass, &
     &   ioption_meta, joption_meta, jtass_rec, icons_start, &
     &   iprint_hills_tass

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, ierr, jerr, itest

      real(8) :: huge = 1.d+38

      character(len=8) :: params_char, char_tass, char_meta, char_cons

      character(len=120) :: char_line

!-----------------------------------------------------------------------
!     /*   option for tass                                            */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_int1_MPI &
     &   ( ioption_meta, '<ioption_meta>', 14, iounit )

!     /*   read value   */
      call read_int1_MPI &
     &   ( joption_meta, '<joption_meta>', 14, iounit )

!-----------------------------------------------------------------------
!     /*   read characteristic time scale for cvs                     */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI ( time_cv_meta, '<time_cv_meta>', 14, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_cv_meta = time_cv_meta *(1.d-15/au_time)

!-----------------------------------------------------------------------
!     /*   read gaussian height                                       */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_real1_MPI ( gh_meta_save, '<gh_meta>', 9, iounit )

!     /*   kelvin ->  au   */
      gh_meta_save = gh_meta_save * boltz

!-----------------------------------------------------------------------
!     /*   read characteristic time scale for force constant          */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI ( time_fc_meta, '<time_fc_meta>', 14, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_fc_meta = time_fc_meta *(1.d-15/au_time)

!-----------------------------------------------------------------------
!     /*   time since last gaussian has been added                    */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI &
     &   ( time_limit_meta, '<time_limit_meta>', 17, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_limit_meta = time_limit_meta *(1.d-15/au_time)

!-----------------------------------------------------------------------
!     /*   step interval of gaussian deposition                       */
!-----------------------------------------------------------------------

      iprint_hills_tass = nint( time_limit_meta / dt )

!-----------------------------------------------------------------------
!     /*   time scale of thermostat attached to cv                    */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI ( time_cv_bath, '<time_cv_bath>', 14, iounit )

!     /*   [fs] -> [a.u.]   */
      time_cv_bath  = time_cv_bath * (1.d-15/au_time)

!     /*   change unit  [fs]  -->  [au]   */
      omega_cv_bath = 2.d0*pi/time_cv_bath

!-----------------------------------------------------------------------
!     /*   initial/restart conditions:  hills                         */
!-----------------------------------------------------------------------

!     /*   default:  new collective variables with random numbers     */
      ihills_start = 0

!     /*   check if file called `hills.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'hills.ini', 9, itest )
      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( itest )

!     /*   if the file exists, restart    */
      if ( itest .eq. 0 )  ihills_start = 1

!-----------------------------------------------------------------------
!     /*   initial/restart conditions:  constraints                   */
!-----------------------------------------------------------------------

!     /*   default:  new collective variables with random numbers     */
      icons_start = 0

!     /*   check if file called `cons.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'cons.ini', 8, itest )
      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( itest )

!     /*   if the file exists, restart    */
      if ( itest .eq. 0 )  icons_start = 1

!-----------------------------------------------------------------------
!     /*   initial/restart conditions:  tass                          */
!-----------------------------------------------------------------------

!     /*   default:  new collective variables with random numbers     */
      icv_start = 0

!     /*   check if file called `cv.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'cv.ini', 6, itest )
      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( itest )

!     /*   if the file exists, restart    */
      if ( itest .eq. 0 )  icv_start = 1

!-----------------------------------------------------------------------
!     /*   scaling factor                                             */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_real1_MPI ( dtemp_meta, '<dtemp_meta>', 12, iounit )

!-----------------------------------------------------------------------
!     /*   read maximum number of gaussians                           */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( mg_meta, '<mg_meta>', 9, iounit )

!-----------------------------------------------------------------------
!     /*   default                                                    */
!-----------------------------------------------------------------------

      ierr = 0

!-----------------------------------------------------------------------
!     /*   read number of collective variables                        */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( ntass, '<ntass>', 7, iounit )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   cv position   */
      if ( .not. allocated( rtass ) ) &
     &   allocate( rtass(ntass,nbead) )

!     /*   cv velocity   */
      if ( .not. allocated( vtass ) ) &
     &   allocate( vtass(ntass,nbead) )

!     /*   actual cv position   */
      if ( .not. allocated( stass ) ) &
     &   allocate( stass(ntass,nbead) )

!     /*   cv mass   */
      if ( .not. allocated( fictmass_tass ) ) &
     &   allocate( fictmass_tass(ntass) )

!     /*   thermostat position attached to cv   */
      if ( .not. allocated( rbath_tass ) ) &
     &   allocate( rbath_tass(ntass,nnhc,ncolor,nbead) )

!     /*   thermostat velocity attached to cv   */
      if ( .not. allocated( vbath_tass ) ) &
     &   allocate( vbath_tass(ntass,nnhc,ncolor,nbead) )

!     /*   thermostat force attached to cv   */
      if ( .not. allocated( fbath_tass ) ) &
     &   allocate( fbath_tass(ntass,nnhc,ncolor,nbead) )

!     /*   thermostat mass attached to cv   */
      if ( .not. allocated( qmass_tass ) ) &
     &   allocate( qmass_tass(ncolor) )

!     /*   type of cv   */
      if ( .not. allocated( itype_tass ) ) &
     &   allocate( itype_tass(ntass) )

!     /*   type of cv   */
      if ( .not. allocated( ikind_tass ) ) &
     &   allocate( ikind_tass(ntass) )

!     /*   boundary condition of cv   */
      if ( .not. allocated( ipbc_tass ) ) &
     &   allocate( ipbc_tass(ntass) )

!     /*   harmonic potential, each bead    */
      if ( .not. allocated( pot_ref_tass ) ) &
     &   allocate( pot_ref_tass(nbead) )

!     /*   cv force: harmonic   */
      if ( .not. allocated( fref_tass ) ) &
     &   allocate( fref_tass(ntass,nbead) )

!     /*   force constant of harmonic potential   */
      if ( .not. allocated( fc_tass ) ) &
     &   allocate( fc_tass(ntass) )

!     /*   gaussian potential, each bead   */
      if ( .not. allocated( pot_tass ) ) &
     &   allocate( pot_tass(nbead) )

!     /*   cv force: gaussian   */
      if ( .not. allocated( ftass ) ) &
     &   allocate( ftass(ntass,nbead) )

!     /*   gaussian center   */
      if ( .not. allocated( gc_meta ) ) &
     &   allocate( gc_meta(ntass,mg_meta) )

!     /*   gaussian height   */
      if ( .not. allocated( gh_meta ) ) &
     &   allocate( gh_meta(mg_meta) )

!     /*   gaussian width   */
      if ( .not. allocated( gw_meta ) ) &
     &   allocate( gw_meta(ntass,mg_meta) )

!     /*   gaussian width   */
      if ( .not. allocated( gw_meta_type ) ) &
     &   allocate( gw_meta_type(ntype_tass) )

!     /*   current value of gaussian width   */
      if ( .not. allocated( gw_meta_save ) ) &
     &   allocate( gw_meta_save(ntass) )

!     /*   latest gaussian center   */
      if ( .not. allocated( gc_meta_save ) ) &
     &   allocate( gc_meta_save(ntass,nbead) )

!     /*   time since last gaussian has been added   */
      if ( .not. allocated( time_tass_save ) ) &
     &   allocate( time_tass_save(nbead) )

!     /*   atoms i, j, k, l of cv   */
      if ( .not. allocated( i_tass ) ) &
     &   allocate( i_tass(ntass) )
      if ( .not. allocated( j_tass ) ) &
     &   allocate( j_tass(ntass) )
      if ( .not. allocated( k_tass ) ) &
     &   allocate( k_tass(ntass) )
      if ( .not. allocated( l_tass ) ) &
     &   allocate( l_tass(ntass) )

!     /*   rational function parameters for coordination number   */
      if ( .not. allocated( nu_tass ) ) &
     &   allocate( nu_tass(ntass,2) )
      if ( .not. allocated( mu_tass ) ) &
     &   allocate( mu_tass(ntass,2) )
      if ( .not. allocated( req_tass ) ) &
     &   allocate( req_tass(ntass,2) )

!     /*   parameters for reconstruction of pes   */
      if ( .not. allocated( params_rec_meta ) ) &
     &   allocate( params_rec_meta(3,ntype_tass) )

!     /*   axes of hills reconstruction   */
      if ( .not. allocated( idim_rec_tass ) ) &
     &   allocate( idim_rec_tass(ntass) )

!     /*   forces    */
      if ( .not. allocated( fx_tass ) ) &
     &   allocate( fx_tass(natom,nbead) )
      if ( .not. allocated( fy_tass ) ) &
     &   allocate( fy_tass(natom,nbead) )
      if ( .not. allocated( fz_tass ) ) &
     &   allocate( fz_tass(natom,nbead) )

!     /*   forces    */
      if ( .not. allocated( fux_tass ) ) &
     &   allocate( fux_tass(natom,nbead) )
      if ( .not. allocated( fuy_tass ) ) &
     &   allocate( fuy_tass(natom,nbead) )
      if ( .not. allocated( fuz_tass ) ) &
     &   allocate( fuz_tass(natom,nbead) )

!     /*   minimum and maximum   */
      if ( .not. allocated( params_meta ) ) &
     &   allocate( params_meta(2,ntass) )

!     /*   cv parameters   */
      if ( .not. allocated( params_tass ) ) &
     &   allocate( params_tass(ntype_tass,2) )

!     /*   cvs for reconstruction  */
      if ( .not. allocated( jtass_rec ) ) &
     &   allocate( jtass_rec(3) )

!     /*   average actual position   */
      if ( .not. allocated( stass_avg ) ) &
     &   allocate( stass_avg(ntass,nbead) )

!     /*   average harmonic force for constraint   */
      if ( .not. allocated( fref_tass_avg ) ) &
     &   allocate( fref_tass_avg(ntass,nbead) )

!-----------------------------------------------------------------------
!     /*   initialize cv                                              */
!-----------------------------------------------------------------------

      rtass(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   initialize cv kind                                         */
!-----------------------------------------------------------------------

!     /*   kind of cv   */
      ikind_tass(:) = 0

!     /*   number of collective variables for afed   */
      ntass_afed = 0

!     /*   number of afed collective variables for constrained md  */
      ntass_cons = 0

!     /*   number of afed collective variables for metadynamics  */
      ntass_meta = 0

!-----------------------------------------------------------------------
!     /*   initialize last gaussian                                   */
!-----------------------------------------------------------------------

      time_tass_save(:) = istep_start*dt

!-----------------------------------------------------------------------
!     /*   number of reference steps for harmonic potential           */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( nref_tass, '<nref_tass>', 11, iounit )

!-----------------------------------------------------------------------
!     /*   step size for harmonic potential                           */
!-----------------------------------------------------------------------

      dt_ref_tass = dt/dble(nref_tass)

!-----------------------------------------------------------------------
!     /*   read parameters for tass: gaussian width                   */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<gw_meta>', 9, iounit, ierr )

!     /*   parameters of constraint type   */

      if ( ierr .eq. 0 ) then

         do i = 1, ntype_tass

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_tass(k,1:1)

            if ( ierr .ne. 0 ) exit

         end do

      end if

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */

      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<gw_meta>', 9, iounit, ierr )

!        /*   parameters of constraint type   */

         do i = 1, ntype_tass

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_tass(k,1:1)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <gw_meta> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   read parameters for tass: spring constant                  */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<params_cons>', 13, iounit, ierr )

!     /*   parameters of constraint type   */

      if ( ierr .eq. 0 ) then

         do i = 1, ntype_tass

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_tass(k,2:2)

            if ( ierr .ne. 0 ) exit

         end do

      end if

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */

      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<params_cons>', 13, iounit, ierr )

!        /*   parameters of constraint type   */

         do i = 1, ntype_tass

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_tass(k,2:2)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <params_cons> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!     /*   broadcast   */
      call my_mpi_bcast_real_2( params_tass, ntype_tass, 2 )

!-----------------------------------------------------------------------
!     /*   read type of collective variables: mean force dynamics     */
!-----------------------------------------------------------------------

      /*   initialize   */
      k = 0

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<ntass>', 7, iounit, ierr )

!     /*   read integer   */
      read ( iounit, *, iostat=ierr )

      do k = 1, ntass

         read ( iounit, *, iostat=ierr ) char_tass, char_cons

         if ( char_tass(1:5) .ne. 'AFED ' ) cycle

         ikind_tass(k) = 1

         ntass_afed = ntass_afed + 1

         params_meta(1,k) = - huge
         params_meta(2,k) = + huge

         backspace( iounit )

         if      ( ( char_cons(1:6) .eq. '1     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIST  ' ) ) then
            itype_tass(k) = 1
         else if ( ( char_cons(1:6) .eq. '2     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'ANGL  ' ) ) then
            itype_tass(k) = 2
         else if ( ( char_cons(1:6) .eq. '3     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIH   ' ) ) then
            itype_tass(k) = 3
         else if ( ( char_cons(1:6) .eq. '4     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIFF  ' ) ) then
            itype_tass(k) = 4
         else if ( ( char_cons(1:6) .eq. '5     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'CN    ' ) ) then
            itype_tass(k) = 5
         else if ( ( char_cons(1:6) .eq. '6     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DCN   ' ) ) then
            itype_tass(k) = 6
         else if ( ( char_cons(1:6) .eq. '7     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'XYZ   ' ) ) then
            itype_tass(k) = 7
         else if ( ( char_cons(1:6) .eq. '8     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DXYZ  ' ) ) then
            itype_tass(k) = 8
         else
            ierr = 1
            exit
         end if

         if ( itype_tass(k) .eq. 1 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 2 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), k_tass(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 3 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), k_tass(k), &
     &         l_tass(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 4 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), k_tass(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 5 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), &
     &         nu_tass(k,1), mu_tass(k,1), req_tass(k,1)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 6 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), &
     &         nu_tass(k,1), mu_tass(k,1), req_tass(k,1), &
     &         k_tass(k), l_tass(k), &
     &         nu_tass(k,2), mu_tass(k,2), req_tass(k,2)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 7 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 8 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), k_tass(k)
            if ( ierr .ne. 0 ) exit
         else
            ierr = 1
            exit
         end if

      end do

!     /*   file close   */
      close( iounit )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <ntass> is incorrect (AFED).'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   read type of collective variables: constrained dynamics    */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<ntass>', 7, iounit, ierr )

!     /*   read integer   */
      read ( iounit, *, iostat=ierr )

      do k = 1, ntass

         read ( iounit, *, iostat=ierr ) char_tass, char_cons

         if ( char_tass(1:5) .ne. 'CONS ' ) cycle

         ikind_tass(k) = 2

         ntass_cons = ntass_cons + 1

         params_meta(1,k) = - huge
         params_meta(2,k) = + huge

         backspace( iounit )

         if      ( ( char_cons(1:6) .eq. '1     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIST  ' ) ) then
            itype_tass(k) = 1
         else if ( ( char_cons(1:6) .eq. '2     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'ANGL  ' ) ) then
            itype_tass(k) = 2
         else if ( ( char_cons(1:6) .eq. '3     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIH   ' ) ) then
            itype_tass(k) = 3
         else if ( ( char_cons(1:6) .eq. '4     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIFF  ' ) ) then
            itype_tass(k) = 4
         else if ( ( char_cons(1:6) .eq. '5     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'CN    ' ) ) then
            itype_tass(k) = 5
         else if ( ( char_cons(1:6) .eq. '6     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DCN   ' ) ) then
            itype_tass(k) = 6
         else if ( ( char_cons(1:6) .eq. '7     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'XYZ   ' ) ) then
            itype_tass(k) = 7
         else if ( ( char_cons(1:6) .eq. '8     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DXYZ  ' ) ) then
            itype_tass(k) = 8
         else
            ierr = 1
            exit
         end if

         if ( itype_tass(k) .eq. 1 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), &
     &         rtass(k,1), rtass(k,nbead)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 2 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), k_tass(k), &
     &         rtass(k,1), rtass(k,nbead)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 3 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), k_tass(k), &
     &         l_tass(k), rtass(k,1), rtass(k,nbead)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 4 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), k_tass(k), &
     &         rtass(k,1), rtass(k,nbead)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 5 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), &
     &         nu_tass(k,1), mu_tass(k,1), req_tass(k,1), &
     &         rtass(k,1), rtass(k,nbead)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 6 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), &
     &         nu_tass(k,1), mu_tass(k,1), req_tass(k,1), &
     &         k_tass(k), l_tass(k), &
     &         nu_tass(k,2), mu_tass(k,2), req_tass(k,2), &
     &         rtass(k,1), rtass(k,nbead)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 7 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), &
     &         rtass(k,1), rtass(k,nbead)
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 8 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_tass, char_cons, i_tass(k), j_tass(k), k_tass(k), &
     &         rtass(k,1), rtass(k,nbead)
            if ( ierr .ne. 0 ) exit
         else
            ierr = 1
            exit
         end if

      end do

!     /*   file close   */
      close( iounit )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <ntass> is incorrect (CONS).'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   read type of collective variables: metadynamics            */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<ntass>', 7, iounit, ierr )

!     /*   read integer   */
      read ( iounit, *, iostat=ierr )

      do k = 1, ntass

         read ( iounit, *, iostat=ierr ) char_tass, char_meta

         if ( char_tass(1:5) .ne. 'META ' ) cycle

         ikind_tass(k) = 3

         ntass_meta = ntass_meta + 1

         backspace( iounit )

         params_meta(1,k) = - huge
         params_meta(2,k) = + huge

         if      ( ( char_meta(1:6) .eq. '1     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIST  ' ) ) then
            itype_tass(k) = 1
         else if ( ( char_meta(1:6) .eq. '2     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'ANGL  ' ) ) then
            itype_tass(k) = 2
         else if ( ( char_meta(1:6) .eq. '3     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIH   ' ) ) then
            itype_tass(k) = 3
         else if ( ( char_meta(1:6) .eq. '4     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIFF  ' ) ) then
            itype_tass(k) = 4
         else if ( ( char_meta(1:6) .eq. '5     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'CN    ' ) ) then
            itype_tass(k) = 5
         else if ( ( char_meta(1:6) .eq. '6     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DCN   ' ) ) then
            itype_tass(k) = 6
         else if ( ( char_meta(1:6) .eq. '7     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'XYZ   ' ) ) then
            itype_tass(k) = 7
         else if ( ( char_meta(1:6) .eq. '8     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DXYZ  ' ) ) then
            itype_tass(k) = 8
         else
            ierr = 1
            exit
         end if

         read ( iounit, '(a)', iostat=ierr ) char_line

         if      ( itype_tass(k) .eq. 1 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 2 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), k_tass(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), k_tass(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 3 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), k_tass(k), &
     &         l_tass(k), params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), k_tass(k), &
     &         l_tass(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 4 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), k_tass(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), k_tass(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 5 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), &
     &         nu_tass(k,1), mu_tass(k,1), req_tass(k,1), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), &
     &         nu_tass(k,1), mu_tass(k,1), req_tass(k,1)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 6 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), &
     &         nu_tass(k,1), mu_tass(k,1), req_tass(k,1), &
     &                    k_tass(k), l_tass(k), &
     &         nu_tass(k,2), mu_tass(k,2), req_tass(k,2), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), &
     &         nu_tass(k,1), mu_tass(k,1), req_tass(k,1), &
     &                    k_tass(k), l_tass(k), &
     &         nu_tass(k,2), mu_tass(k,2), req_tass(k,2)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 7 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_tass(k) .eq. 8 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), k_tass(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_tass, char_meta, i_tass(k), j_tass(k), k_tass(k)
            end if
            if ( ierr .ne. 0 ) exit
         else
            ierr = 1
            exit
         end if

      end do

!     /*   file close   */
      close( iounit )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <ntass> is incorrect (META).'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   mpi communication                                          */
!-----------------------------------------------------------------------

!     /*   broadcast   */
      call my_mpi_bcast_int_1 ( i_tass, ntass )
      call my_mpi_bcast_int_1 ( j_tass, ntass )
      call my_mpi_bcast_int_1 ( k_tass, ntass )
      call my_mpi_bcast_int_1 ( l_tass, ntass )
      call my_mpi_bcast_int_2 ( nu_tass, ntass, 2 )
      call my_mpi_bcast_int_2 ( mu_tass, ntass, 2 )
      call my_mpi_bcast_real_2( req_tass, ntass, 2 )
      call my_mpi_bcast_int_1 ( itype_tass, ntass )
      call my_mpi_bcast_int_1 ( ikind_tass, ntass )
      call my_mpi_bcast_real_2( params_meta, 2, ntass )
      call my_mpi_bcast_int_0 ( ntass_afed )
      call my_mpi_bcast_int_0 ( ntass_cons )
      call my_mpi_bcast_int_0 ( ntass_meta )

!-----------------------------------------------------------------------
!     /*   check cv kind                                              */
!-----------------------------------------------------------------------

      ierr = 0

      do k = 1, ntass
         if ( ikind_tass(k) .eq. 0 ) ierr = 1
      end do

      if ( ierr .ne. 0 ) then
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) &
     &      'Error - in keyword <ntass>. AFED, CONS or META.'
         end if
      end if

      if ( ntass_cons .gt. 1 ) then
         ierr = 1
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) &
     &      'Error - in keyword <ntass>. CONS is more than 1.'
         end if
      end if

      if ( ntass_meta .gt. 3 ) then
         ierr = 1
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) &
     &      'Error - in keyword <ntass>. META is more than 3.'
         end if
      end if

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   cv dimension for free energy reconstruction                */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( ntass_rec, '<ntass_rec>', 11, iounit )

!     /*   initialize error flag   */
      ierr = 0
      jerr = 0

!     /*   default value   */
      if ( ntass_rec .le. 0 ) then
         ntass_rec = ntass
         if ( ntass_rec .gt. 3 ) jerr = 1
      end if

!     /*   error flag   */
      if ( ntass_rec .gt. 3 ) ierr = 1
      if ( ntass_rec .gt. ntass ) ierr = 1

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - keyword <ntass_rec> is incorrect.'
      end if

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   cvs for free energy reconstruction                         */
!-----------------------------------------------------------------------

!     /*   default   */
      if ( jerr .eq. 0 ) then

         do i = 1, ntass_rec
            jtass_rec(i) = i
         end do

!     /*   manual   */
      else

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input.dat' )

!           /*   search for tag    */
            call search_tag ( '<ntass_rec>', 11, iounit, ierr )

!           /*   read integer   */
            read ( iounit, *, iostat=ierr )

!           /*   read integer   */
            read ( iounit, *, iostat=ierr ) jtass_rec(1:ntass)

!           /*   file close   */
            close( iounit )

!           /*   error message   */
            if ( ierr .ne. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - keyword <ntass_rec> is incorrect.'
            end if

!        /*   master process only   */
         end if

!        /*   broadcast   */
         call my_mpi_bcast_int_0 ( ierr )

!        /*   check error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!        /*   broadcast   */
         call my_mpi_bcast_int_1 ( jtass_rec, 3 )

!     /*   default or manual  */
      end if

!-----------------------------------------------------------------------
!     /*   read print inverval for reconstruction of pes              */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI &
     &   ( iprint_rec_tass, '<iprint_rec_tass>', 17, iounit )

!-----------------------------------------------------------------------
!     /*   read parameters for reconstruction of pes                  */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<params_rec_meta>', 17, iounit, ierr )

!     /*   min, max, mesh of each cv type   */

      do i = 1, ntype_tass

         read ( iounit, *, iostat=ierr ) params_char

         backspace( iounit )

         if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIST  ' ) ) then
            k = 1
         else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &             ( params_char(1:6) .eq. 'ANGL  ' ) ) then
            k = 2
         else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIH   ' ) ) then
            k = 3
         else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIFF  ' ) ) then
            k = 4
         else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &             ( params_char(1:6) .eq. 'CN    ' ) ) then
            k = 5
         else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DCN   ' ) ) then
            k = 6
         else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &             ( params_char(1:6) .eq. 'XYZ   ' ) ) then
            k = 7
         else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
            k = 8
         else
            ierr = 1
         end if

         if ( ierr .ne. 0 ) exit

         read ( iounit, *, iostat=ierr ) &
     &      params_char, &
     &      params_rec_meta(1,k), &
     &      params_rec_meta(2,k), &
     &      params_rec_meta(3,k)

         if ( ierr .ne. 0 ) exit

      end do

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<params_rec_meta>', 17, iounit, ierr )

!        /*   min, max, mesh of each cv type   */

         do i = 1, ntype_tass

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, &
     &         params_rec_meta(1,k), &
     &         params_rec_meta(2,k), &
     &         params_rec_meta(3,k)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <params_rec_meta> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_tass_param_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   mpi communication                                          */
!-----------------------------------------------------------------------

!     /*   broadcast   */
      call my_mpi_bcast_real_2( params_rec_meta, 3, ntype_tass )

!-----------------------------------------------------------------------
!     /*   gaussian width                                             */
!-----------------------------------------------------------------------

!     /*   loop of cv   */
      do k = 1, ntass

!        /*   cv: mean force dynamics   */
         if      ( ikind_tass(k) .eq. 1 ) then

!           /*   given value   */
            gw_meta_save(k) = params_tass(itype_tass(k),1) * huge

!        /*   cv: constrained dynamics   */
         else if ( ikind_tass(k) .eq. 2 ) then

!           /*   given value   */
            gw_meta_save(k) = params_tass(itype_tass(k),1) * huge

!        /*   cv: metadynamics   */
         else if ( ikind_tass(k) .eq. 3 ) then

!           /*   given value   */
            gw_meta_save(k) = params_tass(itype_tass(k),1)

!        /*   cv   */
         end if

!     /*   loop of cv   */
      end do

!-----------------------------------------------------------------------
!     /*   fictitious mass of cv                                      */
!-----------------------------------------------------------------------

!     /*   loop of cv   */
      do k = 1, ntass

!        /*   cv: mean force dynamics   */
         if      ( ikind_tass(k) .eq. 1 ) then

!           /*   converted   */
            fictmass_tass(k) = 1.d0 / beta &
     &        / ( dmeta_max * params_tass(itype_tass(k),1) &
     &        / time_cv_meta )**2

!        /*   cv: constrained dynamics   */
         else if ( ikind_tass(k) .eq. 2 ) then

!           /*   huge value   */
            fictmass_tass(k) = huge

!        /*   cv: metadynamics   */
         else if ( ikind_tass(k) .eq. 3 ) then

!           /*   converted   */
            fictmass_tass(k) = 1.d0 / beta &
     &        / ( dmeta_max * params_tass(itype_tass(k),1) &
     &        / time_cv_meta )**2

!        /*   cv   */
         end if

!     /*   loop of cv   */
      end do

!-----------------------------------------------------------------------
!     /*   force constant of harmonic term                            */
!-----------------------------------------------------------------------

!     /*   loop of cv   */
      do k = 1, ntass

!        /*   cv: mean force dynamics   */
         if      ( ikind_tass(k) .eq. 1 ) then

!           /*   given value   */
            fc_tass(k) = params_tass(itype_tass(k),2)

!        /*   cv: constrained dynamics   */
         else if ( ikind_tass(k) .eq. 2 ) then

!           /*   given value   */
            fc_tass(k) = params_tass(itype_tass(k),2)

!        /*   cv: metadynamics   */
         else if ( ikind_tass(k) .eq. 3 ) then

!           /*   converted value   */
            fc_tass(k) = fictmass_tass(k) * (2.d0*pi/time_fc_meta)**2

!        /*   cv   */
         end if

!     /*   loop of cv   */
      end do

!-----------------------------------------------------------------------
!     /*   tamd temperature for tass                                  */
!-----------------------------------------------------------------------

!     /*   read real number   */
      call read_real1_MPI &
     &   ( temp_tamd_tass, '<temp_tamd_tass>', 16, iounit )

      if ( temp_tamd_tass .le. 0.d0 ) temp_tamd_tass = temperature

!-----------------------------------------------------------------------
!     /*   gkt for tass                                               */
!-----------------------------------------------------------------------

      gkt_tass = gkt * temp_tamd_tass / temperature

      return
      end





!***********************************************************************
      subroutine init_cons_tass_MPI
!***********************************************************************

      use tass_variables, only : dstep_tass, stass_avg, fref_tass_avg

      implicit none

      dstep_tass = 0.d0

      stass_avg(:,:)        = 0.d0
      fref_tass_avg(:,:)    = 0.d0

      return
      end





!***********************************************************************
      subroutine restart_cons_tass_MPI( ioption )
!***********************************************************************

      use common_variables, only : &
     &   iounit, nbead, myrank

      use tass_variables, only : &
     &   dstep_tass, stass_avg, fref_tass_avg, ntass

      implicit none

      integer :: ioption, i, j

      if ( ioption .eq. 1 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'cons.ini' )

            read ( iounit, * ) dstep_tass

            do j = 1, nbead
            do i = 1, ntass
               read ( iounit, * ) stass_avg(i,j)
            end do
            end do

            do j = 1, nbead
            do i = 1, ntass
               read ( iounit, * ) fref_tass_avg(i,j)
            end do
            end do

            close( iounit )

         end if

         call my_mpi_bcast_real_0( dstep_tass )
         call my_mpi_bcast_real_2( stass_avg, ntass, nbead )
         call my_mpi_bcast_real_2( fref_tass_avg, ntass, nbead )

      end if

      if ( ioption .eq. 2 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'cons.ini' )

            write( iounit, '(e24.16)' ) dstep_tass

            do j = 1, nbead
            do i = 1, ntass
               write( iounit, '(e24.16)' ) stass_avg(i,j)
            end do
            end do

            do j = 1, nbead
            do i = 1, ntass
               write( iounit, '(e24.16)' ) fref_tass_avg(i,j)
            end do
            end do

            close( iounit )

         end if

      end if

      return
      end

