!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Feb 12, 2022 by M. Shiga
!      Description:     set up ring polymer molecular dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_rpmd
!***********************************************************************
!=======================================================================
!
!     set up rpmd
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, ipos_start, iounit, nstep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_rpmd

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file, and               */
!     /*   non-centroid modes are in Gaussian distribution            */
      if     ( ipos_start .eq. 0 ) then
         call init_centroid
         call init_mode
!     /*   read normal mode position                                  */
      else if ( ipos_start .eq. 1 ) then
         call restart_position( 1 )
!     /*   read Cartesian position                                    */
      else if ( ipos_start .eq. 2 ) then
         call restart_position( 2 )
      else
         call error_handling( 1, 'subroutine setup_rpmd', 21)
      end if

!     /*   Maxwell distribution of velocities                         */
      if     ( ivel_start .eq. 0 ) then
         call init_velocity
      else if ( ivel_start .eq. 1 ) then
!        /*   read normal mode momentum (scaled)                      */
         call restart_velocity( 1 )
      else if ( ivel_start .eq. 2 ) then
!        /*   read Cartesian momentum                                 */
         call restart_velocity( 2 )
      else
         call error_handling( 1, 'subroutine setup_rpmd', 21)
      end if

!-----------------------------------------------------------------------
!     /*   subtract translational and rotational part of velocities   */
!-----------------------------------------------------------------------

      call correct_velocity

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         call backup_pimd_nve

         stop

      end if

!-----------------------------------------------------------------------
!     /*   constraints                                                */
!-----------------------------------------------------------------------

      call setup_cons

      return
      end





!***********************************************************************
      subroutine setup_rpmd_nvt
!***********************************************************************
!=======================================================================
!
!     set up rpmd (NVT)
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, iounit, ibath_start, ipos_start, nstep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix

!-----------------------------------------------------------------------
!     /*   thermostat masses for RPMD (NVT)                           */
!-----------------------------------------------------------------------

      call setup_qmass_rpmd_nvt

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_rpmd_nvt

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file, and               */
!     /*   non-centroid modes are in Gaussian distribution            */
      if     ( ipos_start .eq. 0 ) then
         call init_centroid
         call init_mode
!     /*   read normal mode position                                  */
      else if ( ipos_start .eq. 1 ) then
         call restart_position( 1 )
!     /*   read Cartesian position                                    */
      else if ( ipos_start .eq. 2 ) then
         call restart_position( 2 )
      else
         call error_handling( 1, 'subroutine setup_cmd', 20 )
      end if

!     /*   Maxwell distribution of velocities                         */
      if     ( ivel_start .eq. 0 ) then
         call init_velocity
      else if ( ivel_start .eq. 1 ) then
!        /*   read normal mode momentum (scaled)                      */
         call restart_velocity( 1 )
      else if ( ivel_start .eq. 2 ) then
!        /*   read Cartesian momentum                                 */
         call restart_velocity( 2 )
      else
         call error_handling( 1, 'subroutine setup_cmd', 20 )
      end if

!-----------------------------------------------------------------------
!     /*   subtract translational and rotational part of velocities   */
!-----------------------------------------------------------------------

      call correct_velocity

!-----------------------------------------------------------------------
!     /*   set up thermostat positions and velocities                 */
!-----------------------------------------------------------------------

!     /*   for initial start   */

      if ( ibath_start .le. 0 ) then
!        /*   Maxwell distribution of thermostat velocities   */
         call init_bath_nhc

!     /*   for restart   */

      else if ( ibath_start .eq. 1 ) then
!        /*   read in thermostat position and momentum   */
         call restart_bath_nhc( 1 )
      else if ( ibath_start .eq. 2 ) then
!        /*   read in thermostat position and momentum   */
         call restart_bath_nhc( 1 )
      else
         call error_handling( 1, 'subroutine setup_rpmd_nvt', 25 )
      end if

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         call backup_cmd

         stop

      end if

!-----------------------------------------------------------------------
!     /*   constraints                                                */
!-----------------------------------------------------------------------

      call setup_cons

      return
      end





!***********************************************************************
      subroutine setup_trpmd
!***********************************************************************
!=======================================================================
!
!     set up thermostatted rpmd
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, ipos_start, iounit, nstep, integrator_trpmd, &
     &   scale_trpmd, hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none
      integer :: ierr, itest

!-----------------------------------------------------------------------
!     /*   bcmd integrator                                            */
!-----------------------------------------------------------------------

      call read_char &
     &   ( integrator_trpmd, 12, '<integrator_trpmd>', 18, iounit )

!-----------------------------------------------------------------------
!     /*   bcmd decoherence time ( relative to beta hbar )            */
!-----------------------------------------------------------------------

      call read_real1 &
     &   ( scale_trpmd, '<scale_trpmd>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_rpmd

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file, and               */
!     /*   non-centroid modes are in Gaussian distribution            */
      if     ( ipos_start .eq. 0 ) then
         call init_centroid
         call init_mode
!     /*   read normal mode position                                  */
      else if ( ipos_start .eq. 1 ) then
         call restart_position( 1 )
!     /*   read Cartesian position                                    */
      else if ( ipos_start .eq. 2 ) then
         call restart_position( 2 )
      else
         call error_handling( 1, 'subroutine setup_rpmd', 21)
      end if

!     /*   Maxwell distribution of velocities                         */
      if     ( ivel_start .eq. 0 ) then
         call init_velocity
      else if ( ivel_start .eq. 1 ) then
!        /*   read normal mode momentum (scaled)                      */
         call restart_velocity( 1 )
      else if ( ivel_start .eq. 2 ) then
!        /*   read Cartesian momentum                                 */
         call restart_velocity( 2 )
      else
         call error_handling( 1, 'subroutine setup_rpmd', 21)
      end if

!-----------------------------------------------------------------------
!     /*   subtract translational and rotational part of velocities   */
!-----------------------------------------------------------------------

      call correct_velocity

!-----------------------------------------------------------------------
!     /*   initialize energy correction                               */
!-----------------------------------------------------------------------

      call testfile ( 'step.ini', 8, itest, iounit )

      if ( itest .eq. 0 ) then

         open ( iounit, file = 'step.ini' )

         read ( iounit, *, iostat=ierr )
         read ( iounit, *, iostat=ierr ) hamiltonian_cor

         close( iounit )

         if ( ierr .ne. 0 ) hamiltonian_cor = 0.d0

      else

         hamiltonian_cor = 0.d0

      end if

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         call backup_pimd_nve

         stop

      end if

!-----------------------------------------------------------------------
!     /*   constraints                                                */
!-----------------------------------------------------------------------

      call setup_cons

      return
      end
