!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     replica exchange Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_remc
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temperature, boltz, tnm, tnminv, box_bead, vir_bead, &
     &   boxinv_bead, volume_bead, ipos_start, iounit, nbead, natom, &
     &   irem_type, ibox_start

      use rehmc_variables, only : &
     &   x_save, y_save, z_save, beta_bead, temperature_bead, &
     &   pot_save, fx_save, fy_save, fz_save, vir_bead_save, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, bfactor_rem

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ierr

      real(8) :: t1, tn, alpha

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   preserved coordinates   */
      if ( .not. allocated( x_rehmc_last ) ) &
     &   allocate( x_rehmc_last(natom,nbead) )
      if ( .not. allocated( y_rehmc_last ) ) &
     &   allocate( y_rehmc_last(natom,nbead) )
      if ( .not. allocated( z_rehmc_last ) ) &
     &   allocate( z_rehmc_last(natom,nbead) )

!     /*   preserved potential   */
      if ( .not. allocated( pot_rehmc_last ) ) &
     &   allocate( pot_rehmc_last(nbead) )

!     /*   preserved coordinates   */
      if ( .not. allocated( fx_rehmc_last ) ) &
     &   allocate( fx_rehmc_last(natom,nbead) )
      if ( .not. allocated( fy_rehmc_last ) ) &
     &   allocate( fy_rehmc_last(natom,nbead) )
      if ( .not. allocated( fz_rehmc_last ) ) &
     &   allocate( fz_rehmc_last(natom,nbead) )

!     /*   box of each bead   */
      if ( .not. allocated(box_bead) ) &
     &   allocate( box_bead(3,3,nbead) )

!     /*   box of each bead   */
      if ( .not. allocated(boxinv_bead) ) &
     &   allocate( boxinv_bead(3,3,nbead) )

!     /*   volume of each bead   */
      if ( .not. allocated(volume_bead) ) &
     &   allocate( volume_bead(nbead) )

!     /*   virial of each bead   */
      if ( .not. allocated(vir_bead) ) &
     &   allocate( vir_bead(3,3,nbead) )

!     /*   temperature of each bead  */
      if ( .not. allocated( temperature_bead ) ) &
     &   allocate( temperature_bead(nbead) )

!     /*   beta values of each bead  */
      if ( .not. allocated( beta_bead ) ) &
     &   allocate( beta_bead(nbead) )

!     /*   beta times energy difference   */
      if ( .not. allocated( bfactor_rem ) ) &
     &   allocate( bfactor_rem(nbead) )

!     //   saved coordinates
      if ( .not. allocated(x_save) ) &
     &    allocate( x_save(natom,nbead) )
      if ( .not. allocated(y_save) ) &
     &    allocate( y_save(natom,nbead) )
      if ( .not. allocated(z_save) ) &
     &    allocate( z_save(natom,nbead) )

!     //   saved forces
      if ( .not. allocated(fx_save) ) &
     &    allocate(fx_save(natom,nbead))
      if ( .not. allocated(fy_save) ) &
     &    allocate(fy_save(natom,nbead))
      if ( .not. allocated(fz_save) ) &
     &    allocate(fz_save(natom,nbead))

!     //   saved potentials
      if ( .not. allocated(pot_save) ) &
     &    allocate( pot_save(nbead) )

!     //   saved virials
      if ( .not. allocated(vir_bead_save) ) &
     &   allocate( vir_bead_save(3,3,nbead) )

!-----------------------------------------------------------------------
!     /*   temperature                                                */
!-----------------------------------------------------------------------

      if      ( irem_type(1:3) .eq. 'T  ' ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<temprange_rem>', 15, iounit, ierr )

!        /*   temperature of first and last replica   */
         read ( iounit, *, iostat=ierr ) t1, tn

!        /*   file close   */
         close( iounit )

!        /*   exponent   */
         if ( ierr .eq. 0 ) then
            if ( nbead .eq. 1 ) then
               alpha = 0.d0
            else
               alpha = log ( tn / t1 ) / dble(nbead-1)
            end if
         end if

!        /*   temperature of each replica   */
         if ( ierr .eq. 0 ) then
            do i = 1, nbead
               temperature_bead(i) = exp( alpha * dble(i-1) ) * t1
            end do
         end if

      else

!        /*   temperature of each bead   */
         do i = 1, nbead
            temperature_bead(i) = temperature
         end do

         ierr = 0

      end if

      if ( ierr .ne. 0 ) then

         write( 6, '(a)' ) &
     &      'Error - <temprange_rem> read incorrectly.'
         write( 6, '(a)' )

      end if

!     /*   error termination   */
      call error_handling ( ierr, 'subroutine setup_remc', 21 )

!     /*   beta value of each bead   */

      do i = 1, nbead
         beta_bead(i) = 1.d0 / ( boltz * temperature_bead(i) )
      end do

!-----------------------------------------------------------------------
!     /*   unit matrix (just for convenience)                         */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, nbead
         tnm(i,j)    = 0.d0
         tnminv(i,j) = 0.d0
      end do
      end do

      do i = 1, nbead
         tnm(i,i)    = 1.d0
         tnminv(i,i) = 1.d0
      end do

!-----------------------------------------------------------------------
!     /*   atom position                                              */
!-----------------------------------------------------------------------

      if      ( ipos_start .eq. 0 ) then

!        /*   initialize position   */
         call init_position_cart

!        /*   ux --> x   */
         call nm_trans_cart( 2 )

      else if ( ipos_start .eq. 1 ) then

!        /*   restart position   */
         call restart_position( 2 )

      else

!        /*   error handling   */
         call error_handling( 1, 'subroutine setup_rehmc', 22 )

      end if

!-----------------------------------------------------------------------
!     /*   set up simulation box                                      */
!-----------------------------------------------------------------------

!     /*   fresh starts   */
      if      ( ibox_start .eq. 0 ) then

!        /*   initialize box position   */
         call init_box_position_rehmc

!     /*   restarts   */
      else if ( ibox_start .eq. 1 ) then

!        /*   restart box   */
         call restart_box_rehmc( 1 )

!     /*   otherwise   */
      else

!        /*   error termination   */
         call error_handling( 1, 'subroutine setup_rehmc', 22 )

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     //   exchanged atoms
!-----------------------------------------------------------------------

      call setup_ax

      return
      end





!***********************************************************************
      subroutine setup_ax
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, natom, nbead, ipotential

      use rehmc_variables, only : &
     &   pot_low_save, x_sub_save, y_sub_save, z_sub_save, pot_sub_save, &
     &   nsubstep_ax, iatom_mc, jatom_mc

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr

!-----------------------------------------------------------------------
!     //   exchanged atoms
!-----------------------------------------------------------------------

!     //   error flag
      ierr = 0

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<ax>', 4, iounit, ierr )

!     //   read range of atom exchange
      read( iounit, *, iostat=ierr ) iatom_mc, jatom_mc

!     /*   file close   */
      close( iounit )

!     /*   error handling   */
      call error_handling( ierr, 'subroutine setup_ax', 19 )

!     //    default: all atoms
      if( ( ierr .ne. 0 ) .or. ( iatom_mc .le. 0 ) ) iatom_mc = 1
      if( ( ierr .ne. 0 ) .or. ( jatom_mc .le. 0 ) ) jatom_mc = natom

!-----------------------------------------------------------------------
!     //    after this only for dual potential
!-----------------------------------------------------------------------

      if ( ipotential(1:5) .ne. 'DUAL ' ) return

!-----------------------------------------------------------------------
!     //    dual: inner steps for dual level method
!-----------------------------------------------------------------------

!     //    read step interval of atom exchange
      call read_int1 ( nsubstep_ax, '<nsubstep_ax>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   dual: memory allocation                                    */
!-----------------------------------------------------------------------

!     //   saved coordinates
      if ( .not. allocated(x_sub_save) ) &
     &    allocate( x_sub_save(natom,nbead) )
      if ( .not. allocated(y_sub_save) ) &
     &    allocate( y_sub_save(natom,nbead) )
      if ( .not. allocated(z_sub_save) ) &
     &    allocate( z_sub_save(natom,nbead) )

!     //   saved potentials
      if ( .not. allocated(pot_sub_save) ) &
     &    allocate( pot_sub_save(nbead) )

!     //   saved low level potentials
      if ( .not. allocated(pot_low_save) ) &
     &   allocate( pot_low_save(nbead) )

      return
      end
