!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     set up replica exchange hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_rehmc_npt
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, beta, boltz, box_ref, volume_ref, volmass, boxmass, dt, &
     &   boxinv_ref, temperature, boltz, tnm, tnminv, au_length, &
     &   box_bead, vbox_bead, vlog_bead, vvol_bead, volume_bead, &
     &   dt_ref, vir_bead, omega_baro, boxdot_bead, boxinv_bead, &
     &   pres_bead, pres_bead_iso, natom, ibox_start, iounit, nbead, &
     &   ipos_start, ivel_start, npt_type, irem_type, iounit, &
     &   au_length, char_boundary, nref, istep_hmc, box_anis

      use rehmc_variables, only : &
     &   temperature_bead, beta_bead, bfactor_bead, bfactor_rem, &
     &   bfactor_rex, temp_bead, ekin_bead, hamiltonian_bead, &
     &   ebaro_bead, x_save, y_save, z_save, fx_save, fy_save, fz_save, &
     &   ux_save, uy_save, uz_save, vx_save, vy_save, vz_save, &
     &   pot_save, hamiltonian_bead_save, pot_alchem_save, &
     &   fx_alchem_save, fy_alchem_save, fz_alchem_save, &
     &   box_bead_save, boxinv_bead_save, volume_bead_save, &
     &   vbox_bead_save, vir_bead_save, x_rehmc_last, y_rehmc_last, &
     &   z_rehmc_last, pot_rehmc_last, fx_rehmc_last, fy_rehmc_last, &
     &   fz_rehmc_last, box_rehmc_last, ratio_min_hmc, ratio_max_hmc, &
     &   istep_min_hmc, istep_max_hmc, istep_mul_hmc, istep_adjust_hmc, &
     &   mbox_save, istep_ax_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   unit conversion factor   */
      real(8) :: bohr2ang = au_length * 1.d+10

!     /*   error flag   */
      integer :: ierr = 0

!     /*   integers   */
      integer :: i, j

!     /*   real numbers   */
      real(8) :: t1, tn, alpha, det3
      real(8) :: huge = 1.d+30

!-----------------------------------------------------------------------
!     /*   time step for volume change                                */
!-----------------------------------------------------------------------

      dt_ref = dt/dble(nref)

!-----------------------------------------------------------------------
!     /*   istep_hmc:   molecular dynamics steps                      */
!-----------------------------------------------------------------------

      call read_int1 ( istep_hmc, '<istep_hmc>', 11, iounit )

!-----------------------------------------------------------------------
!     /*   istep_hmc:   molecular dynamics steps                      */
!-----------------------------------------------------------------------

!     //   interval of adjusting trial steps
      call read_int1 &
     &   ( istep_adjust_hmc, '<istep_adjust_hmc>', 18, iounit )

!     //   minimum trial steps
      istep_min_hmc = istep_hmc

!     //   maximum trial steps
      call read_int1 &
     &   ( istep_max_hmc, '<istep_max_hmc>', 15, iounit )

!     //   magnifying factor
      call read_int1 &
     &   ( istep_mul_hmc, '<istep_mul_hmc>', 15, iounit )

!     //   lower bound of target acceptance ratio
      call read_real1 &
     &   ( ratio_min_hmc, '<ratio_min_hmc>', 15, iounit )

!     //   upper bound of target acceptance ratio
      call read_real1 &
     &   ( ratio_max_hmc, '<ratio_max_hmc>', 15, iounit )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   box of each bead   */
      if ( .not. allocated(box_bead) ) &
     &   allocate( box_bead(3,3,nbead) )

!     /*   box of each bead   */
      if ( .not. allocated(boxinv_bead) ) &
     &   allocate( boxinv_bead(3,3,nbead) )

!     /*   boxdot of each bead   */
      if ( .not. allocated(boxdot_bead) ) &
     &   allocate( boxdot_bead(3,3,nbead) )

!     /*   virial of each bead   */
      if ( .not. allocated(vir_bead) ) &
     &   allocate( vir_bead(3,3,nbead) )

!     /*   volume of each bead   */
      if ( .not. allocated(volume_bead) ) &
     &   allocate( volume_bead(nbead) )

!     /*   temperature of each bead  */
      if ( .not. allocated( temperature_bead ) ) &
     &   allocate( temperature_bead(nbead) )

!     /*   beta values of each bead  */
      if ( .not. allocated( beta_bead ) ) &
     &   allocate( beta_bead(nbead) )

      if ( .not. allocated(vvol_bead) ) &
     &   allocate( vvol_bead(nbead) )

      if ( .not. allocated(vlog_bead) ) &
     &   allocate( vlog_bead(nbead) )

      if ( .not. allocated(vbox_bead) ) &
     &   allocate( vbox_bead(3,3,nbead) )

      if ( .not. allocated( pres_bead ) ) &
     &   allocate( pres_bead(3,3,nbead) )

      if ( .not. allocated( pres_bead_iso ) ) &
     &   allocate( pres_bead_iso(nbead) )

!-----------------------------------------------------------------------
!     /*   temperature                                                */
!-----------------------------------------------------------------------

!     /*   option:  no exchange   */
      if      ( irem_type(1:3) .eq. 'T  ' ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<temprange_rem>', 15, iounit, ierr )

!        /*   temperature of first and last replica   */
         read ( iounit, *, iostat=ierr ) t1, tn

!        /*   file close   */
         close( iounit )

!        /*   exponent   */
         if ( ierr .eq. 0 ) then
            if ( nbead .eq. 1 ) then
               alpha = 0.d0
            else
               alpha = log ( tn / t1 ) / dble(nbead-1)
            end if
         end if

!        /*   temperature of each replica   */
         if ( ierr .eq. 0 ) then
            do i = 1, nbead
               temperature_bead(i) = exp( alpha * dble(i-1) ) * t1
            end do
         end if

!     /*   option:  temperature exchange   */
      else if ( irem_type(1:3) .eq. 'TX ' ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<temprange_rem>', 15, iounit, ierr )

!        /*   temperature of first and last replica   */
         read ( iounit, *, iostat=ierr ) t1, tn

!        /*   file close   */
         close( iounit )

!        /*   exponent   */
         if ( ierr .eq. 0 ) then
            if ( nbead .eq. 1 ) then
               alpha = 0.d0
            else
               alpha = log ( tn / t1 ) / dble(nbead-1)
            end if
         end if

!        /*   temperature of each replica   */
         if ( ierr .eq. 0 ) then
            do i = 1, nbead
               temperature_bead(i) = exp( alpha * dble(i-1) ) * t1
            end do
         end if

!     /*   option:  hamiltonian exchange   */
      else if ( irem_type(1:3) .eq. 'HX ' ) then

!        /*   temperature of each bead   */
         do i = 1, nbead
            temperature_bead(i) = temperature
         end do

         ierr = 0

!     /*   option:  otherwise   */
      else

!        /*   temperature of each bead   */
         do i = 1, nbead
            temperature_bead(i) = temperature
         end do

         ierr = 0

!     /*   option   */
      end if

!     /*   check error   */
      if ( ierr .ne. 0 ) then

!        /*   error message   */
         write( 6, '(a)' ) &
     &      'Error - <temprange_rem> read incorrectly.'
         write( 6, '(a)' )

!     /*   check error   */
      end if

!     /*   error termination   */
      call error_handling ( ierr, 'subroutine setup_rehmc_npt', 26 )

!     /*   beta value of each bead   */
      do i = 1, nbead
         beta_bead(i) = 1.d0 / ( boltz * temperature_bead(i) )
      end do

!-----------------------------------------------------------------------
!     /*   unit matrix (just for convenience)                         */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, nbead
         tnm(i,j)    = 0.d0
         tnminv(i,j) = 0.d0
      end do
      end do

      do i = 1, nbead
         tnm(i,i)    = 1.d0
         tnminv(i,i) = 1.d0
      end do

!-----------------------------------------------------------------------
!     /*   atom position                                              */
!-----------------------------------------------------------------------

!     /*   fresh starts   */
      if      ( ipos_start .eq. 0 ) then

!        /*   initialize position   */
         call init_position_cart

!        /*   ux --> x   */
         call nm_trans_cart( 2 )

!     /*   restarts   */
      else if ( ipos_start .eq. 1 ) then

!        /*   restart position   */
         call restart_position( 2 )

!     /*   otherwise   */
      else

!        /*   error handling   */
         call error_handling( 1, 'subroutine setup_rehmc_npt', 26 )

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   atom velocity                                              */
!-----------------------------------------------------------------------

!     /*   fresh starts   */
      if      ( ivel_start .eq. 0 ) then

!        /*   initialize velocity   */
         call init_velocity_real

!     /*   restarts   */
      else if ( ivel_start .eq. 1 ) then

!        /*   restart velocity   */
         call restart_velocity( 2 )

!     /*   otherwise   */
      else

!        /*   error handling   */
         call error_handling( 1, 'subroutine setup_rehmc_npt', 26 )

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   barostat type                                              */
!-----------------------------------------------------------------------

      call read_char ( npt_type, 8, '<npt_type>', 10, iounit )

!-----------------------------------------------------------------------
!     /*   reference box and volume                                   */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'input.dat' )

!     /*   tag   */
      call search_tag ( '<iboundary>', 11, iounit, ierr )

!     /*   skip a line   */
      read( iounit, *, iostat=ierr )

!     /*   box size   */
      read( iounit, *, iostat=ierr ) &
     &   box_ref(1,1), box_ref(1,2), box_ref(1,3)
      read( iounit, *, iostat=ierr ) &
     &   box_ref(2,1), box_ref(2,2), box_ref(2,3)
      read( iounit, *, iostat=ierr ) &
     &   box_ref(3,1), box_ref(3,2), box_ref(3,3)

!     /*   close file   */
      close( iounit )

      if ( ierr .ne. 0 ) write( 6, '(a)' ) &
     &   'Error - keyword <iboundary> is incorrect.'

!     /*   check error   */
      call error_handling( ierr, 'subroutine setup_rehmc_npt', 26 )

!     /*   unit conversion   */
      if ( char_boundary(1:9) .eq. 'ANGSTROM ' ) then
         box_ref(:,:) = box_ref(:,:) / bohr2ang
      end if

!     /*   reference volume   */
      volume_ref = det3( box_ref )

!     /*   inverse matrix of cell matrix   */
      call inv3 ( box_ref, boxinv_ref )

!-----------------------------------------------------------------------
!     /*   volume mass                                                */
!-----------------------------------------------------------------------

!     /*   martyna barostat   */
      if      ( npt_type(1:6) .eq. 'CUBIC1' ) then
         boxmass(:,:) = 3.d0 * dble(natom+1) / beta / omega_baro**2
      else if ( npt_type(1:6) .eq. 'CUBIC2' ) then
         boxmass(:,:) = 3.d0 * dble(natom+1) / beta / omega_baro**2
      else if ( npt_type(1:6) .eq. 'PPHEX ' ) then
         boxmass(:,:) = dble(natom+1) / beta / omega_baro**2
      end if

!     /*   andersen barostat   */
      volmass = boxmass(1,1) * 9.d0 * volume_ref**2

!     //   box constraint for non-diagonal elements
      call read_char ( box_anis, 8, '<box_anis>', 10, iounit )

      if ( box_anis(1:3) .eq. 'OFF' ) then
         boxmass(1,2) = boxmass(1,2) * huge
         boxmass(1,3) = boxmass(1,3) * huge
         boxmass(2,1) = boxmass(2,1) * huge
         boxmass(2,3) = boxmass(2,3) * huge
         boxmass(3,1) = boxmass(3,1) * huge
         boxmass(3,2) = boxmass(3,2) * huge
      end if

!-----------------------------------------------------------------------
!     /*   set up simulation box                                      */
!-----------------------------------------------------------------------

!     /*   fresh starts   */
      if      ( ibox_start .eq. 0 ) then

!        /*   initialize box position   */
         call init_box_position_rehmc

!        /*   initialize box velocity   */
         call init_box_velocity_rehmc

!     /*   restarts   */
      else if ( ibox_start .eq. 1 ) then

!        /*   restart box   */
         call restart_box_rehmc( 3 )

!     /*   otherwise   */
      else

!        /*   error termination   */
         call error_handling( 1, 'subroutine setup_rehmc_npt', 26 )

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   preserved coordinates   */
      if ( .not. allocated( x_rehmc_last ) ) &
     &   allocate( x_rehmc_last(natom,nbead) )
      if ( .not. allocated( y_rehmc_last ) ) &
     &   allocate( y_rehmc_last(natom,nbead) )
      if ( .not. allocated( z_rehmc_last ) ) &
     &   allocate( z_rehmc_last(natom,nbead) )

!     /*   preserved potential   */
      if ( .not. allocated( pot_rehmc_last ) ) &
     &   allocate( pot_rehmc_last(nbead) )

!     /*   preserved coordinates   */
      if ( .not. allocated( fx_rehmc_last ) ) &
     &   allocate( fx_rehmc_last(natom,nbead) )
      if ( .not. allocated( fy_rehmc_last ) ) &
     &   allocate( fy_rehmc_last(natom,nbead) )
      if ( .not. allocated( fz_rehmc_last ) ) &
     &   allocate( fz_rehmc_last(natom,nbead) )

!     /*   preserved box   */
      if ( .not. allocated( box_rehmc_last ) ) &
     &   allocate( box_rehmc_last(3,3,nbead) )

!     /*   beta times energy difference   */
      if ( .not. allocated( bfactor_bead ) ) &
     &   allocate( bfactor_bead(nbead) )

!     /*   beta times energy difference   */
      if ( .not. allocated( bfactor_rem ) ) &
     &   allocate( bfactor_rem(nbead) )

!     /*   beta times energy difference   */
      if ( .not. allocated( bfactor_rex ) ) &
     &   allocate( bfactor_rex(nbead) )

!     /*   kinetic energy of each bead   */
      if ( .not. allocated( ekin_bead ) ) &
     &   allocate( ekin_bead(nbead) )

!     /*   barostat of each bead   */
      if ( .not. allocated( ebaro_bead ) ) &
     &   allocate( ebaro_bead(nbead) )

!     /*   instantaneous temperature of each bead  */
      if ( .not. allocated( temp_bead ) ) &
     &   allocate( temp_bead(nbead) )

!     /*   hamiltonian of each bead   */
      if ( .not. allocated( hamiltonian_bead ) ) &
     &   allocate( hamiltonian_bead(nbead) )

!     /*   saved coordinates   */
      if ( .not. allocated( x_save ) ) &
     &   allocate( x_save(natom,nbead) )
      if ( .not. allocated( y_save ) ) &
     &   allocate( y_save(natom,nbead) )
      if ( .not. allocated( z_save ) ) &
     &   allocate( z_save(natom,nbead) )

!     /*   saved coordinates   */
      if ( .not. allocated( ux_save ) ) &
     &   allocate( ux_save(natom,nbead) )
      if ( .not. allocated( uy_save ) ) &
     &   allocate( uy_save(natom,nbead) )
      if ( .not. allocated( uz_save ) ) &
     &   allocate( uz_save(natom,nbead) )

!     /*   saved velocities   */
      if ( .not. allocated( vx_save ) ) &
     &   allocate( vx_save(natom,nbead) )
      if ( .not. allocated( vy_save ) ) &
     &   allocate( vy_save(natom,nbead) )
      if ( .not. allocated( vz_save ) ) &
     &   allocate( vz_save(natom,nbead) )

!     /*   saved forces   */
      if ( .not. allocated( fx_save ) ) &
     &   allocate( fx_save(natom,nbead) )
      if ( .not. allocated( fy_save ) ) &
     &   allocate( fy_save(natom,nbead) )
      if ( .not. allocated( fz_save ) ) &
     &   allocate( fz_save(natom,nbead) )

!     /*   saved potential   */
      if ( .not. allocated( pot_save ) ) &
     &   allocate( pot_save(nbead) )

!     /*   saved potential   */
      if ( .not. allocated( hamiltonian_bead_save ) ) &
     &   allocate( hamiltonian_bead_save(nbead) )

!     /*   saved potential   */
      if ( .not. allocated( pot_alchem_save ) ) &
     &   allocate( pot_alchem_save(nbead,nbead) )

!     /*   saved force   */
      if ( .not. allocated( fx_alchem_save ) ) &
     &   allocate( fx_alchem_save(natom,nbead,nbead) )
      if ( .not. allocated( fy_alchem_save ) ) &
     &   allocate( fy_alchem_save(natom,nbead,nbead) )
      if ( .not. allocated( fz_alchem_save ) ) &
     &   allocate( fz_alchem_save(natom,nbead,nbead) )

!     /*   saved pbc index   */
      if ( .not. allocated( mbox_save ) ) &
     &   allocate( mbox_save(3,natom,nbead) )

!     /*   saved box position   */
      if ( .not. allocated( box_bead_save ) ) &
     &   allocate( box_bead_save(3,3,nbead) )

!     /*   saved box inverse  */
      if ( .not. allocated( boxinv_bead_save ) ) &
     &   allocate( boxinv_bead_save(3,3,nbead) )

!     /*   saved box volume   */
      if ( .not. allocated( volume_bead_save ) ) &
     &   allocate( volume_bead_save(nbead) )

!     /*   saved box velocity   */
      if ( .not. allocated( vbox_bead_save ) ) &
     &   allocate( vbox_bead_save(3,3,nbead) )

!     /*   saved virial   */
      if ( .not. allocated( vir_bead_save ) ) &
     &   allocate( vir_bead_save(3,3,nbead) )

!-----------------------------------------------------------------------
!     /*   istep_ax_hmc:   exchange interval                          */
!-----------------------------------------------------------------------

      call read_int1 ( istep_ax_hmc, '<istep_ax_hmc>', 10, iounit )

!-----------------------------------------------------------------------
!     /*   setup exchange                                             */
!-----------------------------------------------------------------------

      if ( istep_ax_hmc .le. 0 ) return

      call setup_ax

      return
      end
