!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Feb 12, 2022 by M. Shiga
!      Description:     thermostat fictitious masses
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_qmass_pimd
!***********************************************************************
!=======================================================================
!
!     set up thermostat fictitious masses
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, qmass_cent, qmass, beta, omega_p2, omega_bath, gamma_p2, &
     &   ncolor, nnhc, nbead, natom, bath_type

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, imode
      real(8) :: factor_color, omega, omega2
      real(8) :: scale_color = 0.1d0

!-----------------------------------------------------------------------
!     /*   thermostat attached to centroids                           */
!-----------------------------------------------------------------------

      if ( bath_type(1:5) .eq. 'NHC  ' ) then

         do m = 1, ncolor
            factor_color = scale_color**(m-1)
            omega = omega_bath * factor_color
            omega2 = omega * omega
               qmass_cent(1,m) = 3.d0*dble(natom)/beta/omega2
            do i = 2, nnhc
               qmass_cent(i,m) = 1.d0/beta/omega2
            end do
         end do

      else if ( bath_type(1:5) .eq. 'NHCS ' ) then

         do m = 1, ncolor
            factor_color = scale_color**(m-1)
            omega = omega_bath * factor_color
            omega2 = omega * omega
               qmass_cent(1,m) = dble(natom)/beta/omega2
            do i = 2, nnhc
               qmass_cent(i,m) = 1.d0/beta/omega2
            end do
         end do

      else if ( bath_type(1:5) .eq. 'MNHC ' ) then

         do m = 1, ncolor
            factor_color = scale_color**(m-1)
            omega = omega_bath * factor_color
            omega2 = omega * omega
               qmass_cent(1,m) = 1.d0/beta/omega2
            do i = 2, nnhc
               qmass_cent(i,m) = 1.d0/beta/omega2
            end do
         end do

      else

         do m = 1, ncolor
            factor_color = scale_color**(m-1)
            omega = omega_bath * factor_color
            omega2 = omega * omega
               qmass_cent(1,m) = 1.d0/beta/omega2
            do i = 2, nnhc
               qmass_cent(i,m) = 1.d0/beta/omega2
            end do
         end do

      end if

      if ( ncolor .ge. 2 ) then
         do i = 1, nnhc
            qmass_cent(i,ncolor) = 1.d0/beta/omega_p2
         end do
      end if

!-----------------------------------------------------------------------
!     /*   thermostat attached to non-centroid modes                  */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         qmass(imode) = 1.d0/beta/omega_p2 * gamma_p2
      end do

      return
      end





!***********************************************************************
      subroutine setup_qmass_box
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, qmass_box, beta, omega_baro, ncolor, nnhc, npt_type

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m
      real(8) :: factor_color, omega, omega2
      real(8) :: scale_color = 0.1d0

!-----------------------------------------------------------------------
!     /*   thermostat attached to box   */
!-----------------------------------------------------------------------

      if      ( npt_type(1:6) .eq. 'CUBIC1' ) then

         do m = 1, ncolor
            factor_color = scale_color**(m-1)
            omega = omega_baro * factor_color
            omega2 = omega * omega
            qmass_box(1,m) = 1.d0/beta/omega2
            do i = 2, nnhc
               qmass_box(i,m) = 1.d0/beta/omega2
            end do

         end do

      else if ( npt_type(1:6) .eq. 'CUBIC2' ) then

         do m = 1, ncolor
            factor_color = scale_color**(m-1)
            omega = omega_baro * factor_color
            omega2 = omega * omega
            qmass_box(1,m) = 1.d0/beta/omega2
            do i = 2, nnhc
               qmass_box(i,m) = 1.d0/beta/omega2
            end do
         end do

      else if ( npt_type(1:6) .eq. 'PPHEX' ) then

         do m = 1, ncolor
            factor_color = scale_color**(m-1)
            omega = omega_baro * factor_color
            omega2 = omega * omega
            qmass_box(1,m) = dble(3*(3+1)/2)/beta/omega2
            do i = 2, nnhc
               qmass_box(i,m) = 1.d0/beta/omega2
            end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine setup_qmass_cmd
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, qmass_cent, qmass, beta, omega_p2, omega_bath, gamma2, &
     &   gamma_p2, ncolor, nnhc, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, imode
      real(8) :: factor_color, omega, omega2
      real(8) :: scale_color = 0.1d0

!-----------------------------------------------------------------------
!     /*   thermostat attached to centroids                           */
!-----------------------------------------------------------------------

!     /*   thermostat attached to centroids   */

      do m = 1, ncolor
         factor_color = scale_color**(m-1)
         omega = omega_bath * factor_color
         omega2 = omega * omega
         qmass_cent(1,m) = 3.d0*dble(natom)/beta/omega2
         do i = 2, nnhc
            qmass_cent(i,m) = 1.d0/beta/omega2
         end do
      end do

      if ( ncolor .ge. 2 ) then
         do i = 1, nnhc
            qmass_cent(i,ncolor) = 1.d0/beta/omega_p2
         end do
      end if

!-----------------------------------------------------------------------
!     /*   thermostat attached to non-centroid modes                  */
!-----------------------------------------------------------------------

!     /*   NOTE:  qmass should be scaled by gamma2, since natural  */
!     /*   angular frequencies of modes are omega_p/gamma1         */

      do imode = 1, nbead
         qmass(imode) = 1.d0/beta/omega_p2 * gamma_p2 * gamma2
      end do

      return
      end





!***********************************************************************
      subroutine setup_qmass_rpmd_nvt
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, qmass_cent, qmass, beta, omega_p2, omega_bath, dnmmass, &
     &   physmass, ncolor, nnhc, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, imode
      real(8) :: factor_color, omega, omega2
      real(8) :: scale_color = 0.1d0

      do m = 1, ncolor
         factor_color = scale_color**(m-1)
         omega = omega_bath * factor_color
         omega2 = omega * omega
         qmass_cent(1,m) = 3.d0*dble(natom)/beta/omega2
         do i = 2, nnhc
            qmass_cent(i,m) = 1.d0/beta/omega2
         end do
      end do

!         if ( ncolor .ge. 2 ) then
!            do i = 1, nnhc
!               qmass_cent(i,ncolor) = 1.d0/beta/omega_p2
!            end do
!         end if
!
!      if ( ncolor .le. nbead ) then
!
!         omega = omega_bath
!         omega2 = omega * omega
!         qmass_cent(1,1) = 3.d0*dble(natom)/beta/omega2
!         do i = 2, nnhc
!            qmass_cent(i,1) = 1.d0/beta/omega2
!         end do
!
!         do m = 2, ncolor
!            omega2 = dnmmass(1,m) / physmass(1) * omega_p2
!            qmass_cent(1,m) = 1.d0/beta/omega2
!            do i = 2, nnhc
!               qmass_cent(i,m) = 1.d0/beta/omega2
!            end do
!         end do
!
!      end if

!-----------------------------------------------------------------------
!     /*   thermostat attached to non-centroid modes                  */
!-----------------------------------------------------------------------

!     /*   NOTE:  qmass should be scaled by gamma2, since natural  */
!     /*   angular frequencies of modes are omega_p/gamma1         */

      qmass(1) = 1.d0/beta/omega_p2

      do imode = 2, nbead
         omega2 = dnmmass(1,imode) / physmass(1) * omega_p2
         qmass(imode) = 1.d0/beta/omega2
      end do

      return
      end





!***********************************************************************
      subroutine setup_qmass_md
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, qmass_cart, beta, omega_bath, ncolor, nnhc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m
      real(8) :: factor_color, omega, omega2
      real(8) :: scale_color = 0.1d0

!-----------------------------------------------------------------------
!     /*   REVISED: thermostat masses attached to system              */
!-----------------------------------------------------------------------

      do m = 1, ncolor
         factor_color = scale_color**(m-1)
         omega = omega_bath * factor_color
         omega2 = omega * omega
         qmass_cart(1,m) = 1.d0/beta/omega2
         do i = 2, nnhc
            qmass_cart(i,m) = 1.d0/beta/omega2
         end do
      end do

      return
      end





!***********************************************************************
      subroutine setup_qmass_meta
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, beta, omega_bath, qmass, ncolor, nbead

      use meta_variables, only : &
     &   qmass_meta, omega_cv_bath

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m
      real(8) :: factor_color, omega, omega2
      real(8) :: scale_color = 0.1d0

!-----------------------------------------------------------------------
!     /*   thermostat mass attached to system                         */
!-----------------------------------------------------------------------

      do i = 1, nbead
         qmass(i) = 1.d0/beta/(omega_bath*omega_bath)
      end do

!     /*   thermostat mass attached to cv   */

      do m = 1, ncolor
         factor_color = scale_color**(m-1)
         omega = omega_cv_bath * factor_color
         omega2 = omega * omega
         qmass_meta(m) = 1.d0/beta/omega2
      end do

      return
      end





!***********************************************************************
      subroutine setup_qmass_tass
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, beta, omega_bath, qmass, ncolor, nbead

      use tass_variables, only : &
     &   qmass_tass, omega_cv_bath

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m
      real(8) :: factor_color, omega, omega2
      real(8) :: scale_color = 0.1d0

!-----------------------------------------------------------------------
!     /*   thermostat mass attached to system                         */
!-----------------------------------------------------------------------

      do i = 1, nbead
         qmass(i) = 1.d0/beta/(omega_bath*omega_bath)
      end do

!     /*   thermostat mass attached to cv   */

      do m = 1, ncolor
         factor_color = scale_color**(m-1)
         omega = omega_cv_bath * factor_color
         omega2 = omega * omega
         qmass_tass(m) = 1.d0/beta/omega2
      end do

      return
      end






!***********************************************************************
      subroutine setup_qmass_multi
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, beta, omega_bath, omega_p2, ncolor, nbead, nnhc

      use qmmm_variables, only : &
     &   qmass_cent_multi_a, qmass_cent_multi_b, qmass_multi_b, &
     &   qmass_multi_a, nmulti

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, imode
      real(8) :: factor_color, omega, omega2, scale
      real(8) :: scale_color = 0.1d0

!-----------------------------------------------------------------------
!     /*   thermostat attached to centroids p                         */
!-----------------------------------------------------------------------

      do m = 1, ncolor
         factor_color = scale_color**(m-1)
         omega = omega_bath * factor_color
         omega2 = omega * omega
         qmass_cent_multi_a(1,m) = 1.d0/beta/omega2
         do i = 2, nnhc
            qmass_cent_multi_a(i,m) = 1.d0/beta/omega2
         end do
      end do

      if ( ncolor .ge. 2 ) then
         do i = 1, nnhc
            qmass_cent_multi_a(i,ncolor) = 1.d0/beta/omega_p2
         end do
      end if

!-----------------------------------------------------------------------
!     /*   thermostat attached to non-centroid modes p                */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         qmass_multi_a(imode) = 1.d0/beta/omega_p2
      end do

!-----------------------------------------------------------------------
!     /*   thermostat attached to centroids s                         */
!-----------------------------------------------------------------------

      scale = 1.d0/dble(nmulti*nmulti)

      do m = 1, ncolor
         factor_color = scale_color**(m-1)
         omega = omega_bath * factor_color
         omega2 = omega * omega
         qmass_cent_multi_b(1,m) = 1.d0/beta/omega2 * scale
         do i = 2, nnhc
            qmass_cent_multi_b(i,m) = 1.d0/beta/omega2 * scale
         end do
      end do

      if ( ncolor .ge. 2 ) then
         do i = 1, nnhc
            qmass_cent_multi_b(i,ncolor) = 1.d0/beta/omega_p2
         end do
      end if

!-----------------------------------------------------------------------
!     /*   thermostat attached to non-centroid modes p                */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         qmass_multi_b(imode) = 1.d0/beta/omega_p2
      end do

      return
      end
