!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jun 25, 2021 by M. Shiga
!      Description:     polymers setup
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_polymers_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit, ipotential, myrank_world

      use cons_variables, only : ncons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr = 0

!-----------------------------------------------------------------------
!     /*   check error in input.dat                                   */
!-----------------------------------------------------------------------

!cc      if ( ( ipotential(1:10) .ne. 'ABINIT-MP ' ) .and.
!cc     &     ( ipotential(1:6)  .ne. 'AENET '     ) .and.
!cc     &     ( ipotential(1:8)  .ne. 'CP2KLIB '   ) .and.
!cc     &     ( ipotential(1:8)  .ne. 'DFTBLIB '   ) .and.
!cc     &     ( ipotential(1:4)  .ne. 'EAM '       ) .and.
!cc     &     ( ipotential(1:3)  .ne. 'MM '        ) .and.
!cc     &     ( ipotential(1:3)  .ne. 'QE '        ) .and.
!cc     &     ( ipotential(1:6)  .ne. 'SMASH '     ) .and.
!cc     &     ( ipotential(1:5)  .ne. 'VASP '      ) ) ierr = 1

      if ( ierr .ne. 0 ) then

         if ( myrank_world .eq. 0 ) then
            write( 6, '(a)' ) &
     &         'Error - potential ' // trim(ipotential) // &
     &         ' not supported for polymers.x.'
            write( 6, '(a)' )
         end if

      end if

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_polymers_MPI', 29 )

!-----------------------------------------------------------------------
!     /*   read number of constraints                                 */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( ncons, '<ncons>', 7, iounit )

!-----------------------------------------------------------------------
!     /*   option                                                     */
!-----------------------------------------------------------------------

      if ( ncons .eq. 0 ) then

         call setup_polymers_atom_MPI

      else

         call setup_polymers_cons_MPI

      end if

      return
      end





!***********************************************************************
      subroutine setup_polymers_atom_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!
!     set up polymers
!
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, au_time, ibath_start, ipos_start, ivel_start,istep_start, &
     &   iounit, natom, nstep, myrank_world

      use polymers_variables, only : &
     &   xc_poly, yc_poly, zc_poly, fxc_poly, fyc_poly, fzc_poly, &
     &   txc_poly, tyc_poly, tzc_poly, pmfc_poly, epot_poly, &
     &   entropy_poly, s_ref, r1_ref, r2_ref, f1_ref, f2_ref, arc_ref, &
     &   s1_ref, s2_ref, dt_poly, ekinvir_poly, pmfc_arc, rc_arc, npoly, &
     &   icycle_poly_start, ends_poly, narc, jpoly, nprocs_top, &
     &   ncycle_poly, ncycle_poly, ngrid_poly, guess_poly, projcmf_poly, &
     &   iprint_poly, istep_poly, myrank_top

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, itest

      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   read parameters                                            */
!-----------------------------------------------------------------------

!ccc     /*   number of polymers   */
!cc      call read_int1_MPI( npoly, '<npoly>', 7, iounit )

!     /*   step size   */
      call read_real1_MPI( dt_poly, '<dt_poly>', 9, iounit )

!     /*   number of cycles   */
      call read_int1_MPI( ncycle_poly, '<ncycle_poly>', 13, iounit )

!     /*   arc length grids per npoly  */
      call read_int1_MPI( ngrid_poly, '<ngrid_poly>', 12, iounit )

!     /*   guess option: LINE or SPLINE   */
      call read_char_MPI( guess_poly, 10, '<guess_poly>', 12, iounit )

!     /*   ends option: FIXED or FREE    */
      call read_char_MPI( ends_poly, 10, '<ends_poly>', 11, iounit )

!     /*   guess option: LINE or SPLINE   */
      call read_char_MPI &
     &   ( projcmf_poly, 10, '<projcmf_poly>', 14, iounit )

!     /*   print interval of average centroid force   */
      call read_int1_MPI( iprint_poly, '<iprint_poly>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

!     /*   total number of arc length grids   */
      narc = ( ngrid_poly - 1 ) * ( npoly - 1 ) + 1

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated(xc_poly) ) allocate( xc_poly(natom,npoly) )
      if ( .not. allocated(yc_poly) ) allocate( yc_poly(natom,npoly) )
      if ( .not. allocated(zc_poly) ) allocate( zc_poly(natom,npoly) )

      if ( .not. allocated(fxc_poly) ) allocate( fxc_poly(natom,npoly) )
      if ( .not. allocated(fyc_poly) ) allocate( fyc_poly(natom,npoly) )
      if ( .not. allocated(fzc_poly) ) allocate( fzc_poly(natom,npoly) )

      if ( .not. allocated(txc_poly) ) allocate( txc_poly(natom,npoly) )
      if ( .not. allocated(tyc_poly) ) allocate( tyc_poly(natom,npoly) )
      if ( .not. allocated(tzc_poly) ) allocate( tzc_poly(natom,npoly) )

      if ( .not. allocated(pmfc_poly) ) allocate( pmfc_poly(npoly) )
      if ( .not. allocated(epot_poly) ) allocate( epot_poly(npoly) )
      if ( .not. allocated(ekinvir_poly) ) allocate(ekinvir_poly(npoly))
      if ( .not. allocated(entropy_poly) ) allocate(entropy_poly(npoly))

      if ( .not. allocated(s_ref) ) allocate( s_ref(npoly,3*natom) )
      if ( .not. allocated(r1_ref) ) allocate( r1_ref(npoly,3*natom) )
      if ( .not. allocated(r2_ref) ) allocate( r2_ref(npoly,3*natom) )

      if ( .not. allocated(f1_ref) ) allocate( f1_ref(npoly,3*natom) )
      if ( .not. allocated(f2_ref) ) allocate( f2_ref(npoly,3*natom) )

      if ( .not. allocated(arc_ref) ) allocate( arc_ref(narc,1) )
      if ( .not. allocated(s1_ref) ) allocate( s1_ref(narc,1) )
      if ( .not. allocated(s2_ref) ) allocate( s2_ref(narc,1) )

      if ( .not. allocated( istep_poly ) ) allocate( istep_poly(npoly) )

      if ( .not. allocated(pmfc_arc) ) allocate( pmfc_arc(narc) )
      if ( .not. allocated(rc_arc) ) allocate( rc_arc(narc) )

!-----------------------------------------------------------------------
!     /*   read starting cycle number                                 */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!        /*   test existence of restart file   */
         call testfile ( 'polycycle.ini', 13, itest, iounit )

!        /*   if exists read cycle number   */
         if ( itest .eq. 0 ) then

!           /*   open file   */
            open ( iounit, file = 'polycycle.ini' )

!           /*   set cycle number   */
            read ( iounit, * ) icycle_poly_start

!           /*   close file   */
            close( iounit )

!        /*   if not reset cycle number to zero   */
         else

!           /*   set cycle number   */
            icycle_poly_start = -1

!        /*   end of if statement   */
         end if

!     /*   only master process   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0_world( icycle_poly_start )

!-----------------------------------------------------------------------
!     /*   copy files in parent directory to new subdirectories       */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!        /*   loop of polymers   */
         do m = 1, npoly

!           /*   get directory   */
            call int3_to_char( m, char_num )

!           /*   make directory   */
            call system( 'mkdir -p poly.' // char_num )

!        /*   loop of polymers   */
        end do

!     /*   only master process   */
      end if

!-----------------------------------------------------------------------
!     /*   wait for all processes                                     */
!-----------------------------------------------------------------------

      call my_mpi_barrier_world

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams_MPI

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix_MPI

!-----------------------------------------------------------------------
!     /*   thermostat masses for path integral MD                     */
!-----------------------------------------------------------------------

      call setup_qmass_pimd

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_polymers_atom_MPI

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   initial setup   */
      if ( ( icycle_poly_start .lt. 0 )  ) then

!        /*   starting step   */
         istep_start = 0

!        /*   starting step   */
         istep_poly(:) = istep_start

!        /*   read initial geometries from structure.dat   */
         call init_polymers_atom_MPI

!        /*   loop of polymers   */
         do jpoly = 1, npoly

!           /*   select my polymer   */
            if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle

!           /*   initial centroid positions   */
            call save_pos_polymers_atom_MPI( 0 )

!           /*   initial non-centroid positions   */
            call init_mode_MPI

!           /*   initial velocities   */
            call init_velocity_MPI

!           /*   initial thermostat positions and velocities   */
            call init_bath_mnhc_MPI

!           /*   write to each directory   */
            call restart_polymers_atom_MPI( 2 )

!        /*   loop of polymers   */
         end do

!        /*   reset cycle number   */
         icycle_poly_start = 0

!     /*   restart   */
      else

!        /*   initialize   */
         xc_poly(:,:) = 0.d0
         yc_poly(:,:) = 0.d0
         zc_poly(:,:) = 0.d0

!        /*   starting step   */
         istep_start = 0

!        /*   starting step   */
         istep_poly(:) = 0

!        /*   loop of polymers   */
         do jpoly = 1, npoly

!           /*   select my polymer   */
            if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle

!           /*   read from each directory   */
            call restart_polymers_atom_MPI( 1 )

!           /*   initial centroid positions   */
            call save_pos_polymers_atom_MPI( 2 )

!        /*   loop of polymers   */
         end do

!        /*   communication   */
         call my_mpi_allreduce_real_2_top( xc_poly, natom, npoly )
         call my_mpi_allreduce_real_2_top( yc_poly, natom, npoly )
         call my_mpi_allreduce_real_2_top( zc_poly, natom, npoly )

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   restart atomic positions and velocities                    */
!-----------------------------------------------------------------------

      ipos_start = 1
      ivel_start = 1
      ibath_start = 1

!-----------------------------------------------------------------------
!     /*   terminate if the calculation is over                       */
!-----------------------------------------------------------------------

      if ( icycle_poly_start .lt. ncycle_poly ) return

      do i = 1, npoly
         if ( istep_poly(i) .lt. nstep ) return
      end do

      call my_mpi_finalize_3
      stop

      return
      end





!***********************************************************************
      subroutine setup_polymers_cons_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!
!     set up polymers
!
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, au_time, natom, nstep, ibath_start,  ipos_start, &
     &   ivel_start, istep_start, iounit, myrank_world

      use cons_variables, only : &
     &   itype_cons, ncons, ntype_cons

      use analysis_variables, only : &
     &   iprint_cons

      use polymers_variables, only : &
     &   xc_poly, yc_poly, zc_poly, fcons_poly, pmfc_poly, scons_poly, &
     &   entropy_poly, s_ref, r1_ref, r2_ref, f1_ref, f2_ref, arc_ref, &
     &   s1_ref, s2_ref, dt_poly, ekinvir_poly, rcons_poly, &
     &   icycle_poly_start, ends_poly, narc, jpoly, nprocs_top, &
     &   ncycle_poly, ngrid_poly, guess_poly, projcmf_poly, epot_poly, &
     &   iprint_poly, istart_cons_poly, npoly, ncycle_poly, &
     &   istep_poly, myrank_top

      use afed_variables, only : &
     &   params_afed, fictmass_afed

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ierr, k, m, itest

      character(len=3) :: char_num

      character(len=8) :: params_char

!-----------------------------------------------------------------------
!     /*   read parameters                                            */
!-----------------------------------------------------------------------

!     /*   number of polymers   */
      call read_int1_MPI( npoly, '<npoly>', 7, iounit )

!     /*   step size   */
      call read_real1_MPI( dt_poly, '<dt_poly>', 9, iounit )

!     /*   number of cycles   */
      call read_int1_MPI( ncycle_poly, '<ncycle_poly>', 13, iounit )

!     /*   arc length grids per npoly  */
      call read_int1_MPI( ngrid_poly, '<ngrid_poly>', 12, iounit )

!     /*   guess option: LINE or SPLINE   */
      call read_char_MPI( guess_poly, 10, '<guess_poly>', 12, iounit )

!     /*   ends option: FIXED or FREE    */
      call read_char_MPI( ends_poly, 10, '<ends_poly>', 11, iounit )

!     /*   guess option: LINE or SPLINE   */
      call read_char_MPI &
     &   ( projcmf_poly, 10, '<projcmf_poly>', 14, iounit )

!     /*   print interval of average centroid force   */
      call read_int1_MPI( iprint_poly, '<iprint_poly>', 13, iounit )

!     /*   print interval of constraints   */
      call read_int1_MPI( iprint_cons, '<iprint_cons>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

!     /*   total number of arc length grids   */
      narc = ( ngrid_poly - 1 ) * ( npoly - 1 ) + 1

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated(xc_poly) ) allocate( xc_poly(natom,npoly) )
      if ( .not. allocated(yc_poly) ) allocate( yc_poly(natom,npoly) )
      if ( .not. allocated(zc_poly) ) allocate( zc_poly(natom,npoly) )

      if ( .not. allocated(rcons_poly) ) &
     &   allocate( rcons_poly(ncons,npoly) )

      if ( .not. allocated(scons_poly) ) &
     &   allocate( scons_poly(ncons,npoly) )

      if ( .not. allocated(fcons_poly) ) &
     &   allocate( fcons_poly(ncons,npoly) )

      if ( .not. allocated(pmfc_poly) ) allocate( pmfc_poly(npoly) )
      if ( .not. allocated(epot_poly) ) allocate( epot_poly(npoly) )
      if ( .not. allocated(ekinvir_poly) ) allocate(ekinvir_poly(npoly))
      if ( .not. allocated(entropy_poly) ) allocate(entropy_poly(npoly))

      if ( .not. allocated(s_ref) ) allocate( s_ref(npoly,ncons) )
      if ( .not. allocated(r1_ref) ) allocate( r1_ref(npoly,ncons) )
      if ( .not. allocated(r2_ref) ) allocate( r2_ref(npoly,ncons) )

      if ( .not. allocated(f1_ref) ) allocate( f1_ref(npoly,ncons) )
      if ( .not. allocated(f2_ref) ) allocate( f2_ref(npoly,ncons) )

      if ( .not. allocated(arc_ref) ) allocate( arc_ref(narc,1) )
      if ( .not. allocated(s1_ref) ) allocate( s1_ref(narc,1) )
      if ( .not. allocated(s2_ref) ) allocate( s2_ref(narc,1) )

      if ( .not. allocated( params_afed ) ) &
     &   allocate( params_afed(ntype_cons,3) )

      if ( .not. allocated( fictmass_afed ) ) &
     &   allocate( fictmass_afed(ncons) )

      if ( .not. allocated( istep_poly ) ) allocate( istep_poly(npoly) )

!-----------------------------------------------------------------------
!     /*   read starting cycle number                                 */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!        /*   test existence of restart file   */
         call testfile ( 'polycycle.ini', 13, itest, iounit )

!        /*   if exists read cycle number   */
         if ( itest .eq. 0 ) then

!           /*   open file   */
            open ( iounit, file = 'polycycle.ini' )

!           /*   set cycle number   */
            read ( iounit, * ) icycle_poly_start

!           /*   close file   */
            close( iounit )

!        /*   if not reset cycle number to zero   */
         else

!           /*   set cycle number   */
            icycle_poly_start = -1

!        /*   end of if statement   */
         end if

!     /*   only master process   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0_world( icycle_poly_start )

!-----------------------------------------------------------------------
!     /*   check if cons.ini exists                                   */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!        /*   test existence of restart file   */
         call testfile ( 'cons.ini', 8, itest, iounit )

!        /*   flag   */
         if ( itest .eq. 0 ) then
            istart_cons_poly = 1
         else
            istart_cons_poly = 0
         end if

!     /*   only master process   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0_world( istart_cons_poly )

!-----------------------------------------------------------------------
!     /*   copy files in parent directory to new subdirectories       */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!        /*   loop of polymers   */
         do m = 1, npoly

!           /*   get directory   */
            call int3_to_char( m, char_num )

!           /*   make directory   */
            call system( 'mkdir -p poly.' // char_num )

!        /*   loop of polymers   */
        end do

!     /*   only master process   */
      end if

!-----------------------------------------------------------------------
!     /*   wait for all processes                                     */
!-----------------------------------------------------------------------

      call my_mpi_barrier_world

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams_MPI

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix_MPI

!-----------------------------------------------------------------------
!     /*   thermostat masses for path integral MD                     */
!-----------------------------------------------------------------------

      call setup_qmass_pimd

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes, constraints     */
!-----------------------------------------------------------------------

      call init_mass_polymers_cons_MPI

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   initial setup   */
      if ( ( icycle_poly_start .lt. 0 )  ) then

!        /*   starting step   */
         istep_start = 0

!        /*   starting step   */
         istep_poly(:) = istep_start

!        /*   read initial geometries from structure.dat   */
         call init_polymers_atom_MPI

!        /*   loop of polymers   */
         do jpoly = 1, npoly

!           /*   select my polymer   */
            if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle

!           /*   initial centroid positions   */
            call save_pos_polymers_atom_MPI( 0 )

!           /*   initial non-centroid positions   */
            call init_mode_MPI

!           /*   initial velocities   */
            call init_velocity_MPI

!           /*   initial thermostat positions and velocities   */
            call init_bath_mnhc_MPI

!           /*   write beads to each polymer directory   */
            call restart_polymers_atom_MPI( 2 )

!        /*   loop of polymers   */
         end do

!        /*   reset cycle number   */
         icycle_poly_start = 0

!     /*   restart   */
      else

!        /*   loop of polymers   */
         do jpoly = 1, npoly

!           /*   select my polymer   */
            if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle

!           /*   read beads from each polymer directory   */
            call restart_polymers_atom_MPI( 1 )

!        /*   loop of polymers   */
         end do

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   restart atomic positions and velocities                    */
!-----------------------------------------------------------------------

      ipos_start = 1
      ivel_start = 1
      ibath_start = 1

!-----------------------------------------------------------------------
!     /*   look for constraints                                       */
!-----------------------------------------------------------------------

      call setup_cons_MPI

!-----------------------------------------------------------------------
!     /*   read parameters for constraint                             */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank_world .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<params_afed>', 13, iounit, ierr )

!     /*   parameters of constraint type   */

      if ( ierr .eq. 0 ) then

         do i = 1, ntype_cons

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_afed(k,1), params_afed(k,2), &
     &                      params_afed(k,3)

            if ( ierr .ne. 0 ) exit

         end do

      end if

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */

      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<params_afed>', 13, iounit, ierr )

!        /*   parameters of constraint type   */

         do i = 1, ntype_cons

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, params_afed(k,1), params_afed(k,2), &
     &                      params_afed(k,3)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <params_afed> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0_world ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_polymers_cons_MPI', 34 )

!     /*   broadcast   */
      call my_mpi_bcast_real_2_world ( params_afed, ntype_cons, 3 )

!-----------------------------------------------------------------------
!     /*   fictitious mass                                            */
!-----------------------------------------------------------------------

      do i = 1, ncons

         k = itype_cons(i)

         fictmass_afed(i) = 1.d0 / params_afed(k,1)**2

      end do

!-----------------------------------------------------------------------
!     /*   set up constraints                                         */
!-----------------------------------------------------------------------

!     /*   if cons.ini exists, overwrite constraint values   */
      if ( istart_cons_poly .eq. 1 ) then

!        /*   read constraints from main directory   */
         call restart_polymers_cons_MPI( 1 )

!     /*   if cons.ini does not exist, keep values from input.dat   */
      else

!        /*   write constraints to main directory   */
         call restart_polymers_cons_MPI( 2 )

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   initialize constraints                                     */
!-----------------------------------------------------------------------

!     /*   loop of polymers   */
      do jpoly = 1, npoly

!        /*   select my polymer   */
         if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle

!        /*   rcons_poly -> rcons   */
         call save_pos_polymers_cons_MPI( 0 )

!     /*   loop of polymers   */
      end do

!-----------------------------------------------------------------------
!     /*   terminate if the calculation is over                       */
!-----------------------------------------------------------------------

      if ( icycle_poly_start .lt. ncycle_poly ) return

      do i = 1, npoly
         if ( istep_poly(i) .lt. nstep ) exit
      end do

      call my_mpi_finalize_3
      stop

      return
      end
