!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jun 25, 2021 by M. Shiga
!      Description:     set up path integral molecular dynamics (NPH)
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_pimd_nph
!***********************************************************************
!=======================================================================
!
!     set up pimd in nph ensemble
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   beta, pi, box_ref, omega_baro, au_length, volume_ref, volmass, &
     &   boxmass, boltz, boxinv_ref, ipos_start, ivel_start, natom, &
     &   ibox_start, nstep, iounit, nph_type, box_anis, char_boundary

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: det3

      integer :: ierr

      real(8) :: bohr2ang = au_length * 1.d+10

      real(8) :: huge = 1.d+30

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_pimd

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file, and               */
!     /*   non-centroid modes are in gaussian distribution            */
      if     ( ipos_start .eq. 0 ) then
         call init_centroid
         call init_mode
!     /*   read normal mode position                                  */
      else if ( ipos_start .eq. 1 ) then
         call restart_position( 1 )
!     /*   read cartesian position                                    */
      else if ( ipos_start .eq. 2 ) then
         call restart_position( 2 )
      else
         call error_handling( 1, 'subroutine setup_pimd_nph', 25 )
      end if

!     /*   Maxwell distribution of velocities                         */
      if     ( ivel_start .eq. 0 ) then
         call init_velocity
      else if ( ivel_start .eq. 1 ) then
!        /*   read normal mode momentum (scaled)                      */
         call restart_velocity( 1 )
      else if ( ivel_start .eq. 2 ) then
!        /*   read cartesian momentum                                 */
         call restart_velocity( 2 )
      else
         call error_handling( 1, 'subroutine setup_pimd_nph', 25 )
      end if

!-----------------------------------------------------------------------
!     /*   subtract translational and rotational part of velocities   */
!-----------------------------------------------------------------------

      call correct_velocity

!-----------------------------------------------------------------------
!     /*   barostat type                                              */
!-----------------------------------------------------------------------

      call read_char ( nph_type, 8, '<nph_type>', 10, iounit )

!-----------------------------------------------------------------------
!     /*   reference box and volume                                   */
!-----------------------------------------------------------------------

      open (iounit,file = 'input.dat')

!        /*   tag   */
         call search_tag ( '<iboundary>', 11, iounit, ierr )

!        /*   skip a line   */
         read(iounit,*,iostat=ierr)

!        /*   box size   */
         read(iounit,*,iostat=ierr) &
     &      box_ref(1,1), box_ref(1,2), box_ref(1,3)
         read(iounit,*,iostat=ierr) &
     &      box_ref(2,1), box_ref(2,2), box_ref(2,3)
         read(iounit,*,iostat=ierr) &
     &      box_ref(3,1), box_ref(3,2), box_ref(3,3)

      close(iounit)

!     /*   unit conversion   */
      if ( ierr .eq. 0 ) then
      if ( char_boundary(1:9) .eq. 'ANGSTROM ' ) then
         box_ref(:,:) = box_ref(:,:) / bohr2ang
      end if
      end if

!     /*   error termination   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - Simulation box read incorrectly.'
         write( 6, '(a)' )
      end if

!     /*   error termination   */
      call error_handling( ierr, 'subroutine setup_pimd_nph', 25 )

!     /*   reference volume   */
      volume_ref = det3( box_ref )

!     /*   inverse matrix of cell matrix   */
      call inv3 ( box_ref, boxinv_ref )

!-----------------------------------------------------------------------
!     /*   volume mass                                                */
!-----------------------------------------------------------------------

!     /*   martyna barostat   */
      if      ( nph_type(1:6) .eq. 'CUBIC1' ) then
         boxmass(:,:) = 3.d0 * dble(natom+1) / beta / omega_baro**2
      else if ( nph_type(1:6) .eq. 'CUBIC2' ) then
         boxmass(:,:) = 3.d0 * dble(natom+1) / beta / omega_baro**2
      else if ( nph_type(1:6) .eq. 'PPHEX ' ) then
         boxmass(:,:) = dble(natom+1) / beta / omega_baro**2
      end if

!     /*   andersen barostat   */
      volmass = boxmass(1,1) * 9.d0 * volume_ref**2

!     //   box constraint for non-diagonal elements
      call read_char ( box_anis, 8, '<box_anis>', 10, iounit )

      if ( box_anis(1:3) .eq. 'OFF' ) then
         boxmass(1,2) = boxmass(1,2) * huge
         boxmass(1,3) = boxmass(1,3) * huge
         boxmass(2,1) = boxmass(2,1) * huge
         boxmass(2,3) = boxmass(2,3) * huge
         boxmass(3,1) = boxmass(3,1) * huge
         boxmass(3,2) = boxmass(3,2) * huge
      end if

!-----------------------------------------------------------------------
!     /*   set up box velocities                                      */
!-----------------------------------------------------------------------

      if      ( ibox_start .eq. 0 ) then
         call init_box_position
         call init_box_velocity
      else if ( ibox_start .eq. 1 ) then
         call restart_box( 3 )
      else
         call error_handling( 1, 'subroutine setup_pimd_nph', 25 )
      end if

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         call backup_pimd_nph

         stop

      end if

      return
      end

