!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jun 25, 2021 by M. Shiga
!      Description:     set up path integral hybrid Monte Carlo (NVT)
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_pihmc_nvt_MPI
!***********************************************************************
!=======================================================================
!
!     set up pihmc in nvt ensemble
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, nstep, ipos_start, iounit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams_MPI

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix_MPI

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_pihmc_MPI

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file, and               */
!     /*   non-centroid modes are in Gaussian distribution            */
      if     ( ipos_start .eq. 0 ) then
         call init_centroid_MPI
         call init_mode_MPI
!     /*   read normal mode position                                  */
      else if ( ipos_start .eq. 1 ) then
         call restart_position_MPI( 1 )
!     /*   read Cartesian position                                    */
      else if ( ipos_start .eq. 2 ) then
         call restart_position_MPI( 2 )
      else
         call error_handling_MPI( 1, 'subroutine setup_pimd_MPI', 25 )
      end if

!     /*   Maxwell distribution of velocities                         */
      if     ( ivel_start .eq. 0 ) then
         call init_velocity_MPI
      else if ( ivel_start .eq. 1 ) then
!        /*   read normal mode momentum (scaled)                      */
         call restart_velocity_MPI( 1 )
      else if ( ivel_start .eq. 2 ) then
!        /*   read Cartesian momentum                                 */
         call restart_velocity_MPI( 2 )
      else
         call error_handling_MPI( 1, 'subroutine setup_pimd_MPI', 25 )
      end if
 
!-----------------------------------------------------------------------
!     /*   initialize hmc paramters                                   */
!-----------------------------------------------------------------------

      call init_pihmc_MPI

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         call backup_pimd_nve_MPI

         call my_mpi_finalize_2

         stop

      end if

      return
      end





!***********************************************************************
      subroutine init_pihmc_MPI
!***********************************************************************
!=======================================================================
!
!     initialize hybrid monte carlo
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead, iorder_hmc, istep_hmc, iounit, myrank

      use hmc_variables, only : &
     &   pot_hmc, pot_second, pot_fourth, &
     &   fx_second, fy_second, fz_second, ux_save, uy_save, uz_save, &
     &   vux_save, vuy_save, vuz_save, fux_save, fuy_save, fuz_save, &
     &   fux_ref_save, fuy_ref_save, fuz_ref_save, pot_save, &
     &   pot_second_save, pot_fourth_save, pot_hmc_save, &
     &   x_hmc_last, y_hmc_last, z_hmc_last, pot_hmc_last, &
     &   fx_hmc_last, fy_hmc_last, fz_hmc_last, box_hmc_last, &
     &   ratio_min_hmc, ratio_max_hmc, mbox_save, istep_min_hmc, &
     &   istep_max_hmc, istep_mul_hmc, istep_adjust_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ierr

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( x_hmc_last ) ) &
     &   allocate( x_hmc_last(natom,nbead) )
      if ( .not. allocated( y_hmc_last ) ) &
     &   allocate( y_hmc_last(natom,nbead) )
      if ( .not. allocated( z_hmc_last ) ) &
     &   allocate( z_hmc_last(natom,nbead) )

      if ( .not. allocated( pot_hmc_last ) ) &
     &   allocate( pot_hmc_last(nbead) )

      if ( .not. allocated( fx_hmc_last ) ) &
     &   allocate( fx_hmc_last(natom,nbead) )
      if ( .not. allocated( fy_hmc_last ) ) &
     &   allocate( fy_hmc_last(natom,nbead) )
      if ( .not. allocated( fz_hmc_last ) ) &
     &   allocate( fz_hmc_last(natom,nbead) )

      if ( .not. allocated( box_hmc_last ) ) &
     &   allocate( box_hmc_last(3,3) )

      if ( .not. allocated( pot_hmc ) ) &
     &   allocate( pot_hmc(nbead) )
      if ( .not. allocated( pot_second ) ) &
     &   allocate( pot_second(nbead) )
      if ( .not. allocated( pot_fourth ) ) &
     &   allocate( pot_fourth(nbead) )

      if ( .not. allocated( fx_second ) ) &
     &   allocate( fx_second(natom,nbead) )
      if ( .not. allocated( fy_second ) ) &
     &   allocate( fy_second(natom,nbead) )
      if ( .not. allocated( fz_second ) ) &
     &   allocate( fz_second(natom,nbead) )

      if ( .not. allocated( ux_save ) ) &
     &   allocate( ux_save(natom,nbead) )
      if ( .not. allocated( uy_save ) ) &
     &   allocate( uy_save(natom,nbead) )
      if ( .not. allocated( uz_save ) ) &
     &   allocate( uz_save(natom,nbead) )

      if ( .not. allocated( vux_save ) ) &
     &   allocate( vux_save(natom,nbead) )
      if ( .not. allocated( vuy_save ) ) &
     &   allocate( vuy_save(natom,nbead) )
      if ( .not. allocated( vuz_save ) ) &
     &   allocate( vuz_save(natom,nbead) )

      if ( .not. allocated( fux_save ) ) &
     &   allocate( fux_save(natom,nbead) )
      if ( .not. allocated( fuy_save ) ) &
     &   allocate( fuy_save(natom,nbead) )
      if ( .not. allocated( fuz_save ) ) &
     &   allocate( fuz_save(natom,nbead) )

      if ( .not. allocated( fux_ref_save ) ) &
     &   allocate( fux_ref_save(natom,nbead) )
      if ( .not. allocated( fuy_ref_save ) ) &
     &   allocate( fuy_ref_save(natom,nbead) )
      if ( .not. allocated( fuz_ref_save ) ) &
     &   allocate( fuz_ref_save(natom,nbead) )

      if ( .not. allocated( pot_save ) ) &
     &   allocate( pot_save(nbead) )
      if ( .not. allocated( pot_second_save ) ) &
     &   allocate( pot_second_save(nbead) )
      if ( .not. allocated( pot_fourth_save ) ) &
     &   allocate( pot_fourth_save(nbead) )
      if ( .not. allocated( pot_hmc_save ) ) &
     &   allocate( pot_hmc_save(nbead) )

      if ( .not. allocated( mbox_save ) ) &
     &   allocate( mbox_save(3,natom,nbead) )

!-----------------------------------------------------------------------
!     /*   iorder_hmc:  trotter expansion order:  2 or 4              */
!-----------------------------------------------------------------------

      call read_int1_MPI ( iorder_hmc, '<iorder_hmc>', 12, iounit )

!-----------------------------------------------------------------------
!     /*   istep_hmc:   molecular dynamics steps                      */
!-----------------------------------------------------------------------

      call read_int1_MPI ( istep_hmc, '<istep_hmc>', 11, iounit )

!     //   minimum trial steps
      istep_min_hmc = istep_hmc

!     //   adjust istep_hmc if we are restarting
      if ( myrank .eq. 0 ) then
         call testfile ( 'step.ini', 8, itest )
      end if

      call my_mpi_bcast_int_0( itest )

      if ( itest .eq. 0 ) then
         ierr = 0
         if ( myrank .eq. 0 ) then
            open ( iounit, file = 'step.ini' )
            read ( iounit, *, iostat=ierr )
            read ( iounit, *, iostat=ierr ) itest
            close( iounit )
            if ( ierr .eq. 0 ) istep_hmc = itest
         end if
         call my_mpi_bcast_int_0( ierr )
         if ( ierr .eq. 0 ) then
            call my_mpi_bcast_int_0( istep_hmc )
         end if
      end if

!-----------------------------------------------------------------------
!     /*   istep_hmc:   molecular dynamics steps                      */
!-----------------------------------------------------------------------

!     //   interval of adjusting trial steps
      call read_int1_MPI &
     &   ( istep_adjust_hmc, '<istep_adjust_hmc>', 18, iounit )

!     //   maximum trial steps
      call read_int1_MPI &
     &   ( istep_max_hmc, '<istep_max_hmc>', 15, iounit )

!     //   magnifying factor
      call read_int1_MPI &
     &   ( istep_mul_hmc, '<istep_mul_hmc>', 15, iounit )

!     //   lower bound of target acceptance ratio
      call read_real1_MPI &
     &   ( ratio_min_hmc, '<ratio_min_hmc>', 15, iounit )

!     //   upper bound of target acceptance ratio
      call read_real1_MPI &
     &   ( ratio_max_hmc, '<ratio_max_hmc>', 15, iounit )

      return
      end

