!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 17, 2022 by M. Shiga
!      Description:     set up Onsager-Machlup action
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_omopt_MPI
!***********************************************************************

!     /*   shared variables   */
      use common_variables, only : &
     &   au_time, omega_p, omega_p2, natom, nbead, fdiff, iounit, &
     &   ipos_start, myrank

      use om_variables, only : &
     &   gamma_om, fx_om, fy_om, fz_om, v_om, t_om, pot_0, dt_om, &
     &   fx_p, fy_p, fz_p, fx_m, fy_m, fz_m, fx_0, fy_0, fz_0, &
     &   fux_om, fuy_om, fuz_om, fdscale_om, fux_om, fuy_om, fuz_om, &
     &   fx_ref_om, fy_ref_om, fz_ref_om, fux_ref_om, fuy_ref_om, &
     &   fuz_ref_om

      use lbfgs_variables, only : &
     &    pos, pos0, grad, dm, ws, drms_tol, dmax_tol, &
     &    fmax_tol, frms_tol, iflag, nwork, ndim, nup

      use string_variables, only : &
     &    pot_string, s_ref, r1_ref, r2_ref, f1_ref, f2_ref, arc_ref, &
     &    s1_ref, s2_ref, rc_arc, pot_arc, pot1_ref, &
     &    xmax_string, ymax_string, zmax_string, dmax_string, &
     &    pot2_ref, ngrid_string, narc

!     /*   local variables   */
      implicit none

!     /*   integer   */
      integer :: ierr

!-----------------------------------------------------------------------
!     /*   read increment parameter                                   */
!-----------------------------------------------------------------------

!     /*   friction constant   */
      call read_real1_MPI ( gamma_om, '<gamma_om>', 10, iounit )

!     /*   [1/fs] -> [a.u.]   */
      gamma_om = gamma_om /(1.d-15/au_time)

!     /*   finite difference parameter   */
      call read_real1_MPI ( fdiff, '<fdiff>', 7, iounit )

!     /*   om time step   */
      call read_real1_MPI ( dt_om, '<dt_om>', 7, iounit )

!     /*   [fs] -> [a.u.]   */
      dt_om = dt_om *(1.d-15/au_time)

!     /*   om total time   */
      t_om = dt_om * dble(nbead-1)

!-----------------------------------------------------------------------
!     /*   parameters for path integral simulation                    */
!-----------------------------------------------------------------------

!     /*   spring constant of harmonic forces   */
      omega_p = sqrt( 0.5d0*gamma_om/dt_om )
      omega_p2 = omega_p*omega_p

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   potential part of om action   */
      if ( .not. allocated( v_om ) ) &
     &   allocate( v_om(nbead) )

!     /*   forces   */
      if ( .not. allocated( fx_om ) ) &
     &   allocate( fx_om(natom,nbead) )
      if ( .not. allocated( fy_om ) ) &
     &   allocate( fy_om(natom,nbead) )
      if ( .not. allocated( fz_om ) ) &
     &   allocate( fz_om(natom,nbead) )

!     /*   forces at plus position   */
      if ( .not. allocated( fx_p ) ) &
     &   allocate( fx_p(natom,nbead) )
      if ( .not. allocated( fy_p ) ) &
     &   allocate( fy_p(natom,nbead) )
      if ( .not. allocated( fz_p ) ) &
     &   allocate( fz_p(natom,nbead) )

!     /*   forces at minus position   */
      if ( .not. allocated( fx_m ) ) &
     &   allocate( fx_m(natom,nbead) )
      if ( .not. allocated( fy_m ) ) &
     &   allocate( fy_m(natom,nbead) )
      if ( .not. allocated( fz_m ) ) &
     &   allocate( fz_m(natom,nbead) )

!     /*   forces at original position   */
      if ( .not. allocated( fx_0 ) ) &
     &   allocate( fx_0(natom,nbead) )
      if ( .not. allocated( fy_0 ) ) &
     &   allocate( fy_0(natom,nbead) )
      if ( .not. allocated( fz_0 ) ) &
     &   allocate( fz_0(natom,nbead) )

!     /*   potential at original position   */
      if ( .not. allocated( pot_0 ) ) &
     &   allocate( pot_0(nbead) )

!     /*   scaling factor   */
      if ( .not. allocated( fdscale_om ) ) &
     &   allocate( fdscale_om(nbead) )

!     /*   normal mode forces   */
      if ( .not. allocated( fux_om ) ) &
     &   allocate( fux_om(natom,nbead) )
      if ( .not. allocated( fuy_om ) ) &
     &   allocate( fuy_om(natom,nbead) )
      if ( .not. allocated( fuz_om ) ) &
     &   allocate( fuz_om(natom,nbead) )

!     /*   reference forces   */
      if ( .not. allocated( fx_ref_om ) ) &
     &   allocate( fx_ref_om(natom,nbead) )
      if ( .not. allocated( fy_ref_om ) ) &
     &   allocate( fy_ref_om(natom,nbead) )
      if ( .not. allocated( fz_ref_om ) ) &
     &   allocate( fz_ref_om(natom,nbead) )

!     /*   normal mode reference forces   */
      if ( .not. allocated( fux_ref_om ) ) &
     &   allocate( fux_ref_om(natom,nbead) )
      if ( .not. allocated( fuy_ref_om ) ) &
     &   allocate( fuy_ref_om(natom,nbead) )
      if ( .not. allocated( fuz_ref_om ) ) &
     &   allocate( fuz_ref_om(natom,nbead) )

!-----------------------------------------------------------------------
!     /*   setup masses                                               */
!-----------------------------------------------------------------------

      call init_mass_string_MPI

!-----------------------------------------------------------------------
!     /*   read parameters                                            */
!-----------------------------------------------------------------------

!     /*   arc length grids per nbead  */
      call read_int1_MPI( ngrid_string, '<ngrid_string>', 14, iounit )

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

!     /*   total number of arc length grids   */
      narc = (ngrid_string-1)*(nbead-1) + 1

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( pot_string ) ) &
     &   allocate( pot_string(nbead) )

      if ( .not. allocated( s_ref ) ) &
     &   allocate( s_ref(nbead,3*natom) )
      if ( .not. allocated( r1_ref ) ) &
     &   allocate( r1_ref(nbead,3*natom) )
      if ( .not. allocated( r2_ref ) ) &
     &   allocate( r2_ref(nbead,3*natom) )

      if ( .not. allocated( f1_ref ) ) &
     &   allocate( f1_ref(nbead,3*natom) )
      if ( .not. allocated( f2_ref ) ) &
     &   allocate( f2_ref(nbead,3*natom) )

      if ( .not. allocated( pot1_ref ) ) &
     &   allocate( pot1_ref(nbead,1) )
      if ( .not. allocated( pot2_ref ) ) &
     &   allocate( pot2_ref(nbead,1) )

      if ( .not. allocated( arc_ref ) ) &
     &   allocate( arc_ref(narc,1) )
      if ( .not. allocated( s1_ref ) ) &
     &   allocate( s1_ref(narc,1) )
      if ( .not. allocated( s2_ref ) ) &
     &   allocate( s2_ref(narc,1) )

      if ( .not. allocated( rc_arc ) ) &
     &   allocate( rc_arc(narc) )
      if ( .not. allocated( pot_arc ) ) &
     &   allocate( pot_arc(narc) )

      if ( .not. allocated( xmax_string ) ) &
     &   allocate( xmax_string(natom) )
      if ( .not. allocated( ymax_string ) ) &
     &   allocate( ymax_string(natom) )
      if ( .not. allocated( zmax_string ) ) &
     &   allocate( zmax_string(natom) )
      if ( .not. allocated( dmax_string ) ) &
     &   allocate( dmax_string(3*natom) )

!-----------------------------------------------------------------------
!     /*   setup normal modes                                         */
!-----------------------------------------------------------------------

!     /*   normal mode matrix   */
      call nm_matrix_om_MPI

!-----------------------------------------------------------------------
!     /*   setup initial coordinates                                  */
!-----------------------------------------------------------------------

      if ( ipos_start .eq. 1 ) then

!        /*   restart from geometry.ini   */
         call restart_position_MPI( 1 )

      else if ( ipos_start .eq. 3 ) then

!        /*   restart from string.ini   */
         call restart_string_MPI( 1 )

      else

!        /*   initialize path   */
         call init_string_MPI

      end if

!     /*   convert position: cartesian to normal modes   */
      call nm_trans_om( 1 )

!-----------------------------------------------------------------------
!     //   initialize:  read lbfgs parameters
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

         call search_tag ( '<params_lbfgs>', 14, iounit, ierr )

!        /*   read data   */
         read ( iounit, *, iostat=ierr ) &
     &      dmax_tol, drms_tol, fmax_tol, frms_tol

         close( iounit )

         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

            call search_tag ( '<params_lbfgs>', 14, iounit, ierr )

!           /*   read data   */
            read ( iounit, *, iostat=ierr ) &
     &         dmax_tol, drms_tol, fmax_tol, frms_tol

            close( iounit )

         end if

      end if

      call my_mpi_bcast_real_0 ( dmax_tol )
      call my_mpi_bcast_real_0 ( drms_tol )
      call my_mpi_bcast_real_0 ( fmax_tol )
      call my_mpi_bcast_real_0 ( frms_tol )

!-----------------------------------------------------------------------
!     //   allocation
!-----------------------------------------------------------------------

!     //   number of degrees of freedom
      ndim  = 3*natom*(nbead-2)

!     //   dimension of the workspace in lbfgs
      nwork = ndim*(2*nup+1)+2*nup

!     //   coordinates
      if ( .not. allocated( pos ) ) &
     &   allocate( pos(ndim) )

!     //   old coordinates
      if ( .not. allocated( pos0 ) ) &
     &   allocate( pos0(ndim) )

!     //   gradient
      if ( .not. allocated( grad ) ) &
     &   allocate( grad(ndim) )

!     //   diagonal matrix in lbfgs
      if ( .not. allocated( dm ) ) &
     &   allocate( dm(ndim) )

!     //   workspace in lbfgs
      if ( .not. allocated( ws ) ) &
     &   allocate( ws(nwork) )

!     //   initialize: convergence flag
      iflag = 0

      return
      end
