!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up Ehrenfest mean field dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_mfe
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, vx, vy, vz, ux, uy, uz, dt, dt_ref, nref, iounit, &
     &   natom, nbead, istep_start, mbox

      use multistate_variables, only : &
     &   vstate, dxstate, dystate, dzstate, gxstate, gystate, gzstate, &
     &   dipxstate, dipystate, dipzstate, cstate, pop_state, pop_sum, &
     &   branch_ratio, dxstate_old, dystate_old, dzstate_old, &
     &   istate_tfs, nstate, istate_init

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: i, j, itest, ierr

      integer:: idummy

!-----------------------------------------------------------------------
!     /*   number of states                                           */
!-----------------------------------------------------------------------

      call read_int1 ( nstate, '<nstate>', 8, iounit )

!-----------------------------------------------------------------------
!     /*   memory allocation for multi-state calculations             */
!-----------------------------------------------------------------------

!     /*   potential   */
      if ( .not. allocated( vstate ) ) &
     &   allocate( vstate(nstate,nstate,nbead) )

!     /*   gradients   */
      if ( .not. allocated( gxstate ) ) &
     &   allocate( gxstate(nstate,nstate,natom,nbead) )
      if ( .not. allocated( gystate ) ) &
     &   allocate( gystate(nstate,nstate,natom,nbead) )
      if ( .not. allocated( gzstate ) ) &
     &   allocate( gzstate(nstate,nstate,natom,nbead) )

!     /*   nonadiabatic coupling   */
      if ( .not. allocated( dxstate ) ) &
     &   allocate( dxstate(nstate,nstate,natom,nbead) )
      if ( .not. allocated( dystate ) ) &
     &   allocate( dystate(nstate,nstate,natom,nbead) )
      if ( .not. allocated( dzstate ) ) &
     &   allocate( dzstate(nstate,nstate,natom,nbead) )

!     /*   dipole moment   */
      if ( .not. allocated( dipxstate ) ) &
     &   allocate( dipxstate(nstate,nbead) )
      if ( .not. allocated( dipystate ) ) &
     &   allocate( dipystate(nstate,nbead) )
      if ( .not. allocated( dipzstate ) ) &
     &   allocate( dipzstate(nstate,nbead) )

!     /*   state coefficients   */
      if ( .not. allocated( cstate ) ) &
     &   allocate( cstate(nstate,nbead) )

!     /*   state population   */
      if ( .not. allocated( pop_state ) ) &
     &   allocate( pop_state(nstate,nbead) )

!     /*   sum of state population   */
      if ( .not. allocated( pop_sum ) ) &
     &   allocate( pop_sum(nbead) )

!     /*   occupied state   */
      if ( .not. allocated( istate_tfs ) ) &
     &   allocate( istate_tfs(nbead) )

!     /*   branching ratio   */
      if ( .not. allocated( branch_ratio ) ) &
     &   allocate( branch_ratio(nstate) )

!     /*   nonadiabatic coupling   */
      if ( .not. allocated( dxstate_old ) ) &
     &   allocate( dxstate_old(nstate,nstate,natom,nbead) )
      if ( .not. allocated( dystate_old ) ) &
     &   allocate( dystate_old(nstate,nstate,natom,nbead) )
      if ( .not. allocated( dzstate_old ) ) &
     &   allocate( dzstate_old(nstate,nstate,natom,nbead) )

!-----------------------------------------------------------------------
!     /*   time step for state coefficients                           */
!-----------------------------------------------------------------------

      dt_ref = dt/dble(nref)

!-----------------------------------------------------------------------
!     /*   test existence of step.ini                                 */
!-----------------------------------------------------------------------

      call testfile ( 'step.ini', 8, itest, iounit )

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( itest .eq. 1 ) then

!        /*   read initial state   */
         call read_int1 ( istate_init, '<istate_init>', 13, iounit )

!        /*   occupied state initialized for all the beads   */

         do j = 1, nbead
            istate_tfs(j) = istate_init
         end do

!        /*   state coefficients initialized for all the beads   */

         do j = 1, nbead
         do i = 1, nstate

            if ( i .eq. istate_tfs(j) ) then

               cstate(i,j) = dcmplx( 1.d0, 0.d0 )

            else

               cstate(i,j) = dcmplx( 0.d0, 0.d0 )

            end if

         end do
         end do

!        /*   initial geometry read from geometry.ini   */

         open ( iounit, file = 'geometry.ini', status = 'unknown' )

            do j = 1, nbead
            do i = 1, natom
               read ( iounit, *, iostat=ierr ) &
     &            idummy, x(i,j), y(i,j), z(i,j), &
     &                    vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
            end do
            end do

         close( iounit )

!        /*   alternatively read from centroid.dat   */

         if ( ierr .ne. 0 ) then

!           /*   read ux, uy, uz from centroid.dat   */
            call init_centroid

!           /*   the centroid position used for all the beads   */

            do j = 1, nbead
            do i = 1, natom
               x(i,j) = ux(i,1)
               y(i,j) = uy(i,1)
               z(i,j) = uz(i,1)
            end do
            end do

!           /*   generate velocity be maxwell distribution   */
            call init_velocity_cart

         end if

      end if

!-----------------------------------------------------------------------
!     /*   restart                                                    */
!-----------------------------------------------------------------------

      if ( itest .ne. 1 ) then

!        /*   read geometry   */

         open ( iounit, file = 'geometry.ini', status = 'unknown' )

            do j = 1, nbead
            do i = 1, natom
               read ( iounit, *, iostat=ierr ) &
     &            idummy, x(i,j), y(i,j), z(i,j), &
     &                    vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
            end do
            end do

         close( iounit )

!        /*   error handling   */
         call error_handling( ierr, 'subroutine setup_mfe', 20 )

!        /*   read occupied state and state coefficients   */
         call restart_cstate ( 1 )

!        /*   read step number          */

         open ( iounit, file = 'step.ini' )
            read ( iounit, *, iostat=ierr ) istep_start
         close( iounit )

!        /*   error handling  */
         call error_handling( ierr, 'subroutine setup_mfe', 20 )

      end if

      return
      end





!***********************************************************************
      subroutine backup_mfe
!***********************************************************************
!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, vx, vy, vz, nstep, natom, nbead, istep_end, &
     &   iprint_rest, iounit, mbox

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: i, j

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( mod(istep_end,iprint_rest) .eq. 0 ) then
         continue
      else if ( istep_end .eq. nstep ) then
         continue
      else
         return
      end if

!-----------------------------------------------------------------------
!     /*   write geometry to geometry.ini                             */
!-----------------------------------------------------------------------

      open ( iounit, file = 'geometry.ini', status = 'unknown' )

         do j = 1, nbead
         do i = 1, natom
            write(iounit,'(i8,6e24.16,3i4)') istep_end, &
     &         x(i,j), y(i,j), z(i,j), &
     &         vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
         end do
         end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   write state coefficients to cstate.ini                     */
!-----------------------------------------------------------------------

      call restart_cstate ( 2 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
      close( iounit )

      return
      end
