!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, ipos_start, ibath_start

      use meta_variables, only : &
     &   icv_start, ihills_start

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   read input file and memory allocation                      */
!-----------------------------------------------------------------------

      call setmdparams_MPI

      call setup_meta_param_MPI

!-----------------------------------------------------------------------
!     /*   atomic mass                                                */
!-----------------------------------------------------------------------

      call init_mass_md_MPI

!-----------------------------------------------------------------------
!     /*   atom position                                              */
!-----------------------------------------------------------------------

      if      ( ipos_start .eq. 0 ) then

!        /*   initialize position   */
         call init_position_cart_MPI

!        /*   ux --> x   */
         call nm_trans_cart( 2 )

      else if ( ipos_start .eq. 1 ) then

!        /*   restart position   */
         call restart_position_MPI( 2 )

      else

!        /*   check error   */
         call error_handling_MPI( 1, 'subroutine setup_meta_MPI', 25 )

      end if

!-----------------------------------------------------------------------
!     /*   atom velocity                                              */
!-----------------------------------------------------------------------

      if      ( ivel_start .eq. 0 ) then

!        /*   initialize velocity   */
         call init_velocity_MPI

      else if ( ivel_start .eq. 1 ) then

!        /*   restart velocity   */
         call restart_velocity_MPI( 2 )

      else

!        /*   check error   */
         call error_handling_MPI( 1, 'subroutine setup_meta_MPI', 25 )

      end if

!-----------------------------------------------------------------------
!     /*   thermostat masses                                          */
!-----------------------------------------------------------------------

      call setup_qmass_meta

!-----------------------------------------------------------------------
!     /*   position and velocity of all thermostats                   */
!-----------------------------------------------------------------------

      if ( ibath_start .le. 0 ) then

!        /*   initialize thermostats   */
         call init_bath_meta_MPI

      else if ( ibath_start .eq. 1 ) then

!        /*   restart thermostats   */
         call restart_bath_meta_MPI( 1 )

      end if

!-----------------------------------------------------------------------
!     /*   actual position of cv                                      */
!-----------------------------------------------------------------------

      call get_cv_meta_MPI

!-----------------------------------------------------------------------
!     /*   initialize/restart cv                                      */
!-----------------------------------------------------------------------

      if      ( icv_start .le. 0 ) then

!        /*   initialize cv   */
         call init_cv_meta_MPI

      else if ( icv_start .eq. 1 ) then

!        /*   restart cv   */
         call restart_cv_meta_MPI( 1 )

      end if

!-----------------------------------------------------------------------
!     /*   initialize/restart hills                                   */
!-----------------------------------------------------------------------

      if      ( ihills_start .le. 0 ) then

!        /*   initialize hills   */
         call init_hills_meta_MPI

      else if ( ihills_start .eq. 1 ) then

!        /*   initialize hills   */
         call restart_hills_meta_MPI( 1 )

      end if

      return
      end





!***********************************************************************
      subroutine setup_meta_param_MPI
!***********************************************************************
!=======================================================================
!
!     read parameters for metadynamics
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_time, dt, beta, pi, boltz, iounit, natom, nbead, &
     &   ncolor, nnhc, istep_start, myrank

      use meta_variables, only : &
     &   rmeta, vmeta, smeta, fictmass_meta, rbath_meta, fbath_meta, &
     &   vbath_meta, fref_meta, fc_meta, pot_meta, fmeta, gc_meta, &
     &   gh_meta, gw_meta, gw_meta_type, gw_meta_save, gc_meta_save, &
     &   gh_meta_save, time_limit_meta, pot_ref_meta, time_meta_save, &
     &   dmeta, qmass_meta, req_meta, params_rec_meta, dtemp_meta, &
     &   fx_meta, fy_meta, fz_meta, fux_meta, fuy_meta, fuz_meta, &
     &   omega_cv_bath, time_cv_bath, time_cv_meta, time_fc_meta, &
     &   dt_ref_meta, time_cv_meta, dmeta_max, params_meta, &
     &   itype_meta, i_meta, j_meta, k_meta, l_meta, mg_meta, &
     &   mu_meta, nu_meta, idim_rec_meta, nref_meta, itype_meta, &
     &   iprint_rec_meta, ihills_start, icv_start, ntype_meta, &
     &   ioption_meta, joption_meta, ipbc_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, ierr, itest

      real(8) :: f1, f2

      real(8) :: huge = 1.d38

      character(len=8) :: params_char, char_meta

      character(len=120) :: char_line

!-----------------------------------------------------------------------
!     /*   default                                                    */
!-----------------------------------------------------------------------

      ierr = 0

!-----------------------------------------------------------------------
!     /*   read number of collective variables                        */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( nmeta, '<nmeta>', 7, iounit )

!-----------------------------------------------------------------------
!     /*   read maximum number of gaussians                           */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( mg_meta, '<mg_meta>', 9, iounit )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   cv position   */
      if ( .not. allocated( rmeta ) ) &
     &   allocate( rmeta(nmeta,nbead) )

!     /*   cv velocity   */
      if ( .not. allocated( vmeta ) ) &
     &   allocate( vmeta(nmeta,nbead) )

!     /*   actual cv position   */
      if ( .not. allocated( smeta ) ) &
     &   allocate( smeta(nmeta,nbead) )

!     /*   cv mass   */
      if ( .not. allocated( fictmass_meta ) ) &
     &   allocate( fictmass_meta(nmeta) )

!     /*   thermostat position attached to cv   */
      if ( .not. allocated( rbath_meta ) ) &
     &   allocate( rbath_meta(nmeta,nnhc,ncolor,nbead) )

!     /*   thermostat velocity attached to cv   */
      if ( .not. allocated( vbath_meta ) ) &
     &   allocate( vbath_meta(nmeta,nnhc,ncolor,nbead) )

!     /*   thermostat force attached to cv   */
      if ( .not. allocated( fbath_meta ) ) &
     &   allocate( fbath_meta(nmeta,nnhc,ncolor,nbead) )

!     /*   thermostat mass attached to cv   */
      if ( .not. allocated( qmass_meta ) ) &
     &   allocate( qmass_meta(ncolor) )

!     /*   type of cv   */
      if ( .not. allocated( itype_meta ) ) &
     &   allocate( itype_meta(nmeta) )

!     /*   boundary condition of cv   */
      if ( .not. allocated( ipbc_meta ) ) &
     &   allocate( ipbc_meta(nmeta) )

!     /*   harmonic potential, each bead    */
      if ( .not. allocated( pot_ref_meta ) ) &
     &   allocate( pot_ref_meta(nbead) )

!     /*   cv force: harmonic   */
      if ( .not. allocated( fref_meta ) ) &
     &   allocate( fref_meta(nmeta,nbead) )

!     /*   force constant of harmonic potential   */
      if ( .not. allocated( fc_meta ) ) &
     &   allocate( fc_meta(nmeta) )

!     /*   gaussian potential, each bead   */
      if ( .not. allocated( pot_meta ) ) &
     &   allocate( pot_meta(nbead) )

!     /*   cv force: gaussian   */
      if ( .not. allocated( fmeta ) ) &
     &   allocate( fmeta(nmeta,nbead) )

!     /*   gaussian center   */
      if ( .not. allocated( gc_meta ) ) &
     &   allocate( gc_meta(nmeta,mg_meta) )

!     /*   gaussian height   */
      if ( .not. allocated( gh_meta ) ) &
     &   allocate( gh_meta(mg_meta) )

!     /*   gaussian width   */
      if ( .not. allocated( gw_meta ) ) &
     &   allocate( gw_meta(nmeta,mg_meta) )

!     /*   gaussian width   */
      if ( .not. allocated( gw_meta_type ) ) &
     &   allocate( gw_meta_type(ntype_meta) )

!     /*   current value of gaussian width   */
      if ( .not. allocated( gw_meta_save ) ) &
     &   allocate( gw_meta_save(nmeta) )

!     /*   latest gaussian center   */
      if ( .not. allocated( gc_meta_save ) ) &
     &   allocate( gc_meta_save(nmeta,nbead) )

!     /*   time since last gaussian has been added   */
      if ( .not. allocated( time_meta_save ) ) &
     &   allocate( time_meta_save(nbead) )

!     /*   maximum displacement of cv   */
      if ( .not. allocated( dmeta ) ) &
     &   allocate( dmeta(nbead) )

!     /*   atoms i, j, k, l of cv   */
      if ( .not. allocated( i_meta ) ) &
     &   allocate( i_meta(nmeta) )
      if ( .not. allocated( j_meta ) ) &
     &   allocate( j_meta(nmeta) )
      if ( .not. allocated( k_meta ) ) &
     &   allocate( k_meta(nmeta) )
      if ( .not. allocated( l_meta ) ) &
     &   allocate( l_meta(nmeta) )

!     /*   rational function parameters for coordination number   */
      if ( .not. allocated( nu_meta ) ) &
     &   allocate( nu_meta(nmeta,2) )
      if ( .not. allocated( mu_meta ) ) &
     &   allocate( mu_meta(nmeta,2) )
      if ( .not. allocated( req_meta ) ) &
     &   allocate( req_meta(nmeta,2) )

!     /*   parameters for reconstruction of pes   */
      if ( .not. allocated( params_rec_meta ) ) &
     &   allocate( params_rec_meta(3,ntype_meta) )

!     /*   axes of hills reconstruction   */
      if ( .not. allocated( idim_rec_meta ) ) &
     &   allocate( idim_rec_meta(nmeta) )

!     /*   forces    */
      if ( .not. allocated( fx_meta ) ) &
     &   allocate( fx_meta(natom,nbead) )
      if ( .not. allocated( fy_meta ) ) &
     &   allocate( fy_meta(natom,nbead) )
      if ( .not. allocated( fz_meta ) ) &
     &   allocate( fz_meta(natom,nbead) )

!     /*   forces    */
      if ( .not. allocated( fux_meta ) ) &
     &   allocate( fux_meta(natom,nbead) )
      if ( .not. allocated( fuy_meta ) ) &
     &   allocate( fuy_meta(natom,nbead) )
      if ( .not. allocated( fuz_meta ) ) &
     &   allocate( fuz_meta(natom,nbead) )

!     /*   minimum and maximum   */
      if ( .not. allocated( params_meta ) ) &
     &   allocate( params_meta(2,nmeta) )

!-----------------------------------------------------------------------
!     /*   number of reference steps for harmonic potential           */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( nref_meta, '<nref_meta>', 11, iounit )

!-----------------------------------------------------------------------
!     /*   step size for harmonic potential                           */
!-----------------------------------------------------------------------

      dt_ref_meta = dt/dble(nref_meta)

!-----------------------------------------------------------------------
!     /*   read type of collective variables                          */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<nmeta>', 7, iounit, ierr )

!     /*   read integer   */
      read ( iounit, *, iostat=ierr )

      do k = 1, nmeta

         read ( iounit, *, iostat=ierr ) char_meta

         backspace( iounit )

         if      ( ( char_meta(1:6) .eq. '1     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIST  ' ) ) then
            itype_meta(k) = 1
         else if ( ( char_meta(1:6) .eq. '2     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'ANGL  ' ) ) then
            itype_meta(k) = 2
         else if ( ( char_meta(1:6) .eq. '3     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIH   ' ) ) then
            itype_meta(k) = 3
         else if ( ( char_meta(1:6) .eq. '4     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIFF  ' ) ) then
            itype_meta(k) = 4
         else if ( ( char_meta(1:6) .eq. '5     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'CN    ' ) ) then
            itype_meta(k) = 5
         else if ( ( char_meta(1:6) .eq. '6     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DCN   ' ) ) then
            itype_meta(k) = 6
         else if ( ( char_meta(1:6) .eq. '7     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'XYZ   ' ) ) then
            itype_meta(k) = 7
         else if ( ( char_meta(1:6) .eq. '8     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DXYZ  ' ) ) then
            itype_meta(k) = 8
         else
            ierr = 1
            exit
         end if

         read ( iounit, '(a)', iostat=ierr ) char_line

         params_meta(1,k) = - huge
         params_meta(2,k) = + huge

         if      ( itype_meta(k) .eq. 1 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 2 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 3 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), l_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), l_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 4 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 5 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 6 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1), &
     &                    k_meta(k), l_meta(k), &
     &         nu_meta(k,2), mu_meta(k,2), req_meta(k,2), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1), &
     &                    k_meta(k), l_meta(k), &
     &         nu_meta(k,2), mu_meta(k,2), req_meta(k,2)
            end if
         else if ( itype_meta(k) .eq. 7 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 8 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else
            ierr = 1
            exit
         end if

      end do

!     /*   file close   */
      close( iounit )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <nmeta> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_meta_param_MPI', 31 )

!     /*   broadcast   */
      call my_mpi_bcast_int_1 ( i_meta, nmeta )
      call my_mpi_bcast_int_1 ( j_meta, nmeta )
      call my_mpi_bcast_int_1 ( k_meta, nmeta )
      call my_mpi_bcast_int_1 ( l_meta, nmeta )
      call my_mpi_bcast_int_2 ( nu_meta, nmeta, 2 )
      call my_mpi_bcast_int_2 ( mu_meta, nmeta, 2 )
      call my_mpi_bcast_real_2( req_meta, nmeta, 2 )
      call my_mpi_bcast_int_1 ( itype_meta, nmeta )
      call my_mpi_bcast_real_2( params_meta, 2, nmeta )

!-----------------------------------------------------------------------
!     /*   read gaussian height                                       */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_real1_MPI ( gh_meta_save, '<gh_meta>', 9, iounit )

!     /*   kelvin ->  au   */
      gh_meta_save = gh_meta_save * boltz

!-----------------------------------------------------------------------
!     /*   read gaussian width                                        */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<gw_meta>', 9, iounit, ierr )

!     /*   width of each cv type   */

      do i = 1, ntype_meta

         read ( iounit, *, iostat=ierr ) char_meta

         backspace( iounit )

         if      ( ( char_meta(1:6) .eq. '1     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIST  ' ) ) then
            k = 1
         else if ( ( char_meta(1:6) .eq. '2     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'ANGL  ' ) ) then
            k = 2
         else if ( ( char_meta(1:6) .eq. '3     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIH   ' ) ) then
            k = 3
         else if ( ( char_meta(1:6) .eq. '4     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIFF  ' ) ) then
            k = 4
         else if ( ( char_meta(1:6) .eq. '5     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'CN    ' ) ) then
            k = 5
         else if ( ( char_meta(1:6) .eq. '6     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DCN   ' ) ) then
            k = 6
         else if ( ( char_meta(1:6) .eq. '7     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'XYZ   ' ) ) then
            k = 7
         else if ( ( char_meta(1:6) .eq. '8     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DXYZ  ' ) ) then
            k = 8
         else
            ierr = 1
            exit
         end if

         if ( ierr .ne. 0 ) exit

         read ( iounit, *, iostat=ierr ) char_meta, gw_meta_type(k)

         if ( ierr .ne. 0 ) exit

      end do

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<gw_meta>', 9, iounit, ierr )

!     /*   width of each cv type   */

         do i = 1, ntype_meta

            read ( iounit, *, iostat=ierr ) char_meta

            backspace( iounit )

            if      ( ( char_meta(1:6) .eq. '1     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( char_meta(1:6) .eq. '2     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( char_meta(1:6) .eq. '3     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( char_meta(1:6) .eq. '4     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( char_meta(1:6) .eq. '5     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( char_meta(1:6) .eq. '6     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( char_meta(1:6) .eq. '7     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( char_meta(1:6) .eq. '8     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
               exit
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) char_meta, gw_meta_type(k)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <gw_meta> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_meta_param_MPI', 31 )

!     /*   broadcast   */
      call my_mpi_bcast_real_1 ( gw_meta_type, ntype_meta )

!-----------------------------------------------------------------------
!     /*   gaussian width                                             */
!-----------------------------------------------------------------------

      do k = 1, nmeta
         gw_meta_save(k) = gw_meta_type(itype_meta(k))
      end do

!-----------------------------------------------------------------------
!     /*   read characteristic time scale for cvs                     */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI ( time_cv_meta, '<time_cv_meta>', 14, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_cv_meta = time_cv_meta *(1.d-15/au_time)

!-----------------------------------------------------------------------
!     /*   fictitious mass of cv                                      */
!-----------------------------------------------------------------------

      do k = 1, nmeta

!        /*   distance   */
         f1 = dmeta_max*gw_meta_save(k)

!        /*   velocity   */
         f2 = f1 / time_cv_meta

!        /*   mass   */
         fictmass_meta(k) = 1.d0 / beta / (f2*f2)

      end do

!-----------------------------------------------------------------------
!     /*   read characteristic time scale for force constant          */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI ( time_fc_meta, '<time_fc_meta>', 14, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_fc_meta = time_fc_meta *(1.d-15/au_time)

!-----------------------------------------------------------------------
!     /*   force constant of harmonic term                            */
!-----------------------------------------------------------------------

      do k = 1, nmeta

!        /*   mass   */
         f1 = fictmass_meta(k)

!        /*   angular frequency   */
         f2 = 2.d0*pi / time_fc_meta

         fc_meta(k) = f1 * f2*f2

      end do

!-----------------------------------------------------------------------
!     /*   option for metadynamics                                    */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_int1_MPI &
     &   ( ioption_meta, '<ioption_meta>', 14, iounit )

!     /*   read value   */
      call read_int1_MPI &
     &   ( joption_meta, '<joption_meta>', 14, iounit )

!-----------------------------------------------------------------------
!     /*   read print inverval for reconstruction of pes              */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI &
     &   ( iprint_rec_meta, '<iprint_rec_meta>', 17, iounit )

!-----------------------------------------------------------------------
!     /*   read parameters for reconstruction of pes                  */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<params_rec_meta>', 17, iounit, ierr )

!     /*   min, max, mesh of each cv type   */

      do i = 1, ntype_meta

         read ( iounit, *, iostat=ierr ) params_char

         backspace( iounit )

         if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIST  ' ) ) then
            k = 1
         else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &             ( params_char(1:6) .eq. 'ANGL  ' ) ) then
            k = 2
         else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIH   ' ) ) then
            k = 3
         else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIFF  ' ) ) then
            k = 4
         else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &             ( params_char(1:6) .eq. 'CN    ' ) ) then
            k = 5
         else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DCN   ' ) ) then
            k = 6
         else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &             ( params_char(1:6) .eq. 'XYZ   ' ) ) then
            k = 7
         else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
            k = 8
         else
            ierr = 1
         end if

         if ( ierr .ne. 0 ) exit

         read ( iounit, *, iostat=ierr ) &
     &      params_char, &
     &      params_rec_meta(1,k), &
     &      params_rec_meta(2,k), &
     &      params_rec_meta(3,k)

         if ( ierr .ne. 0 ) exit

      end do

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<params_rec_meta>', 17, iounit, ierr )

!        /*   min, max, mesh of each cv type   */

         do i = 1, ntype_meta

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, &
     &         params_rec_meta(1,k), &
     &         params_rec_meta(2,k), &
     &         params_rec_meta(3,k)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <params_rec_meta> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_meta_param_MPI', 31 )

!     /*   broadcast   */
      call my_mpi_bcast_real_2 ( params_rec_meta, 3, ntype_meta )

!-----------------------------------------------------------------------
!     /*   time since last gaussian has been added                    */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI &
     &   ( time_limit_meta, '<time_limit_meta>', 17, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_limit_meta = time_limit_meta *(1.d-15/au_time)

!     /*   initialize   */
      time_meta_save(:) = istep_start*dt

!-----------------------------------------------------------------------
!     /*   time scale of thermostat attached to cv                    */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI( time_cv_bath, '<time_cv_bath>', 14, iounit )

!     /*   [fs] -> [a.u.]   */
      time_cv_bath  = time_cv_bath * (1.d-15/au_time)

!     /*   change unit  [fs]  -->  [au]   */
      omega_cv_bath = 2.d0*pi/time_cv_bath

!-----------------------------------------------------------------------
!     /*   initial/restart conditions:  hills                         */
!-----------------------------------------------------------------------

!     /*   default:  new collective variables with random numbers     */
      ihills_start = 0

!     /*   check if file called `hills.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'hills.ini', 9, itest )
      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( itest )

!     /*   if the file exists, restart    */
      if ( itest .eq. 0 )  ihills_start = 1

!-----------------------------------------------------------------------
!     /*   initial/restart conditions:  metadynamics                  */
!-----------------------------------------------------------------------

!     /*   default:  new collective variables with random numbers     */
      icv_start = 0

!     /*   check if file called `cv.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'cv.ini', 6, itest )
      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( itest )

!     /*   if the file exists, restart    */
      if ( itest .eq. 0 )  icv_start = 1

!-----------------------------------------------------------------------
!     /*   scaling factor                                             */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_real1_MPI ( dtemp_meta, '<dtemp_meta>', 12, iounit )

      return
      end
