!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up QM/MM electronic embedding
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_ee_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, natom, nbead, myrank

      use qmmm_variables, only : &
     &   pot_a, pot_b, fx_a, fy_a, fz_a, fx_b, fy_b, fz_b, &
     &   vir_a, vir_b, dipx_a, dipy_a, dipz_a, dipx_b, dipy_b, dipz_b, &
     &   natom_l, natom_p, natom_s, &
     &   iprint_qmmm, layer, species_link, int_spec_link, i_link, &
     &   j_link, r_link, ivar_qmmm

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, ierr, jerr, iatom_l, j1, j2, j3, j4, k, l

!     /*   characters   */
      character(len=80)  :: char_1, char_2, char_3, char_4, char_5
      character(len=8)   :: char_6, a1, a2, a3, a4
      character(len=200) :: char_line

!     /*   real numbers   */
      real(8) :: r1, r2, r3, r4

!     /*   number of atomic symbols   */
      integer :: nsymbol

!     /*   atomic symbols   */
      character(len=8), dimension(:), allocatable :: symbol

!     /*   atomic numbers   */
      integer, dimension(:), allocatable :: num_symbol

!-----------------------------------------------------------------------
!     /*   print interval                                             */
!-----------------------------------------------------------------------

      call read_int1_MPI (iprint_qmmm, '<iprint_qmmm>', 13, iounit)

!-----------------------------------------------------------------------
!     /*   variable charge                                            */
!-----------------------------------------------------------------------

      call read_int1_MPI (ivar_qmmm, '<ivar_qmmm>', 11, iounit)

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   potential   */
      if ( .not. allocated( pot_a ) ) &
     &   allocate( pot_a(nbead) )
      if ( .not. allocated( pot_b ) ) &
     &   allocate( pot_b(nbead) )

!     /*   forces   */
      if ( .not. allocated( fx_a ) ) &
     &   allocate( fx_a(natom,nbead) )
      if ( .not. allocated( fy_a ) ) &
     &   allocate( fy_a(natom,nbead) )
      if ( .not. allocated( fz_a ) ) &
     &   allocate( fz_a(natom,nbead) )
      if ( .not. allocated( fx_b ) ) &
     &   allocate( fx_b(natom,nbead) )
      if ( .not. allocated( fy_b ) ) &
     &   allocate( fy_b(natom,nbead) )
      if ( .not. allocated( fz_b ) ) &
     &   allocate( fz_b(natom,nbead) )

!     /*   dipole moment   */
      if ( .not. allocated( dipx_a ) ) &
     &   allocate( dipx_a(nbead) )
      if ( .not. allocated( dipy_a ) ) &
     &   allocate( dipy_a(nbead) )
      if ( .not. allocated( dipz_a ) ) &
     &   allocate( dipz_a(nbead) )
      if ( .not. allocated( dipx_b ) ) &
     &   allocate( dipx_b(nbead) )
      if ( .not. allocated( dipy_b ) ) &
     &   allocate( dipy_b(nbead) )
      if ( .not. allocated( dipz_b ) ) &
     &   allocate( dipz_b(nbead) )

!     /*   virial   */
      if ( .not. allocated( vir_a ) ) &
     &   allocate( vir_a(3,3) )
      if ( .not. allocated( vir_b ) ) &
     &   allocate( vir_b(3,3) )

!     /*   layer   */
      if ( .not. allocated( layer ) ) &
     &   allocate( layer(natom) )

!-----------------------------------------------------------------------
!     /*   read structure.dat                                         */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!     /*   number of atoms in the primary region   */
      natom_p = 0

!     /*   number of atoms in the secondary region   */
      natom_s = 0

!     /*   number of atoms link atoms   */
      natom_l = 0

!     /*   reset error flag   */
      jerr = 0

!     /*   file open   */
      open ( iounit, file = 'structure.dat' )

!     /*   read line   */
      read ( iounit, * )

!     /*   read line   */
      read ( iounit, * )

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   read line   */
         read ( iounit, '(a)', iostat=ierr ) char_line

!        /*   if read error is found   */
         if ( ierr .ne. 0 ) then

!           /*   error flag   */
            jerr = 1

!           /*   stop   */
            exit

!        /*   if read error is found   */
         end if

!        /*   read six data   */
         read ( char_line, *, iostat=ierr ) &
     &      char_1, char_2, char_3, char_4, char_5, char_6

!        /*   if only four data is found   */
         if ( ierr .ne. 0 ) then

!           /*   read four data   */
            read ( char_line, *, iostat=ierr ) &
     &         char_1, char_2, char_3, char_4

!           /*   if read error is found   */
            if ( ierr .ne. 0 ) then

!              /*   error flag   */
               jerr = 1

!              /*   stop   */
               exit

!           /*   if read error is found   */
            end if

!           /*   secondary region chosen by default   */
            layer(i) = 'B'

!        /*   if six data is found   */
         else

!           /*   region is given by sixth data   */
            layer(i) = char_6

!        /*   end of if statement   */
         end if

!        /*   for secondary region   */
         if ( layer(i)(1:1) .eq. 'B' ) then

!           /*   count number of atoms   */
            natom_s = natom_s + 1

!        /*   for primary region   */
         else if ( layer(i)(1:1) .eq. 'A' ) then

!           /*   count number of atoms   */
            natom_p = natom_p + 1

!           /*   for atoms with a link atom   */
            if      ( char_6(1:3) .eq. 'AL ' ) then

!              /*   count number of link atoms   */
               natom_l = natom_l + 1

!           /*   for atoms with one link atom   */
            else if ( char_6(1:3) .eq. 'AL1' ) then

!              /*   count number of link atoms   */
               natom_l = natom_l + 1

!           /*   for atoms with two link atoms   */
            else if ( char_6(1:3) .eq. 'AL2' ) then

!              /*   count number of link atoms   */
               natom_l = natom_l + 2

!           /*   for atoms with three link atom   */
            else if ( char_6(1:3) .eq. 'AL3' ) then

!              /*   count number of link atoms   */
               natom_l = natom_l + 3

!           /*   for atoms with four link atom   */
            else if ( char_6(1:3) .eq. 'AL4' ) then

!              /*   count number of link atoms   */
               natom_l = natom_l + 4

!           /*   end of if statement   */
            end if

!        /*   otherwise error   */
         else

!           /*   error flag   */
            jerr = 1

!           /*   stop   */
            exit

!        /*   end of if statement   */
         end if

!     /*   loop of atoms   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   master rank only   */
      end if

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( jerr )

!-----------------------------------------------------------------------
!     /*   error check                                                */
!-----------------------------------------------------------------------

!     /*   error termination   */
      call error_handling_MPI( jerr, 'subroutine setup_ee_MPI', 23 )

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( natom_p )
      call my_mpi_bcast_int_0 ( natom_s )
      call my_mpi_bcast_int_0 ( natom_l )

      call my_mpi_bcast_char_1 ( layer, len(layer), natom )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   if link atom exists   */
      if ( natom_l .gt. 0 ) then

!        /*   bonded atom in primary region   */
         if ( .not. allocated( i_link ) ) &
     &      allocate( i_link(natom_l) )

!        /*   bonded atom in secondary region   */
         if ( .not. allocated( j_link ) ) &
     &      allocate( j_link(natom_l) )

!        /*   atom in bond distance ratio   */
         if ( .not. allocated( r_link ) ) &
     &      allocate( r_link(natom_l) )

!        /*   atomic species of link atom   */
         if ( .not. allocated( species_link ) ) &
     &      allocate( species_link(natom_l) )

!        /*   atomic number of link atom   */
         if ( .not. allocated( int_spec_link ) ) &
     &      allocate( int_spec_link(natom_l) )

!     /*   if link atom exists   */
      end if

!-----------------------------------------------------------------------
!     /*   read structure.dat                                         */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!     /*   reset error flag   */
      jerr = 0

!     /*   link atom counter   */
      iatom_l = 0

!     /*   file open   */
      open ( iounit, file = 'structure.dat' )

!     /*   read line   */
      read ( iounit, * )

!     /*   read line   */
      read ( iounit, * )

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   read line   */
         read ( iounit, '(a)', iostat=ierr ) char_line

!        /*   line with a link atom  */
         if      ( layer(i)(1:3) .eq. 'AL ' ) then

!           /*   read line   */
            read ( char_line, *, iostat=ierr ) &
     &         char_1, char_2, char_3, char_4, char_5, char_6, &
     &         j1, r1, a1

!           /*   if read error found   */
            if ( ierr .ne. 0 ) then

!              /*   set error flag   */
               jerr = 2

!              /*   stop   */
               exit

!           /*   if read error found   */
            end if

!           /*   bonded atom in primary region   */
            i_link(iatom_l+1) = i

!           /*   bonded atom in secondary region   */
            j_link(iatom_l+1) = j1

!           /*   bond ratio   */
            r_link(iatom_l+1) = r1

!           /*   atom species   */
            species_link(iatom_l+1) = a1

!           /*   update link atom counter   */
            iatom_l = iatom_l + 1

!           /*   bonded atom in secondary region   */
            layer(j1) = 'BL'

!        /*   line with one link atom  */
         else if ( layer(i)(1:3) .eq. 'AL1' ) then

!           /*   read line   */
            read ( char_line, *, iostat=ierr ) &
     &         char_1, char_2, char_3, char_4, char_5, char_6, &
     &         j1, r1, a1

!           /*   if read error found   */
            if ( ierr .ne. 0 ) then

!              /*   set error flag   */
               jerr = 2

!              /*   stop   */
               exit

!           /*   if read error found   */
            end if

!           /*   bonded atom in primary region   */
            i_link(iatom_l+1) = i

!           /*   bonded atom in secondary region   */
            j_link(iatom_l+1) = j1

!           /*   bond ratio   */
            r_link(iatom_l+1) = r1

!           /*   atom species   */
            species_link(iatom_l+1) = a1

!           /*   update link atom counter   */
            iatom_l = iatom_l + 1

!           /*   bonded atom in secondary region   */
            layer(j1) = 'BL'

!        /*   line with two link atoms   */
         else if ( layer(i)(1:3) .eq. 'AL2' ) then

!           /*   read line   */
            read ( char_line, *, iostat=ierr ) &
     &         char_1, char_2, char_3, char_4, char_5, char_6, &
     &         j1, r1, a1, j2, r2, a2

!           /*   if read error found   */
            if ( ierr .ne. 0 ) then

!              /*   set error flag   */
               jerr = 2

!              /*   stop   */
               exit

!           /*   if read error found   */
            end if

!           /*   bonded atom in primary region   */
            i_link(iatom_l+1) = i
            i_link(iatom_l+2) = i

!           /*   bonded atom in secondary region   */
            j_link(iatom_l+1) = j1
            j_link(iatom_l+2) = j2

!           /*   bond ratio   */
            r_link(iatom_l+1) = r1
            r_link(iatom_l+2) = r2

!           /*   atom species   */
            species_link(iatom_l+1) = a1
            species_link(iatom_l+2) = a2

!           /*   update link atom counter   */
            iatom_l = iatom_l + 2

!           /*   bonded atom in secondary region   */
            layer(j1) = 'BL'
            layer(j2) = 'BL'

!        /*   line with three link atoms   */
         else if ( layer(i)(1:3) .eq. 'AL3' ) then

!           /*   read line   */
            read ( char_line, *, iostat=ierr ) &
     &         char_1, char_2, char_3, char_4, char_5, char_6, &
     &         j1, r1, a1, j2, r2, a2, j3, r3, a3

!           /*   if read error found   */
            if ( ierr .ne. 0 ) then

!              /*   set error flag   */
               jerr = 2

!              /*   stop   */
               exit

!           /*   if read error found   */
            end if

!           /*   bonded atom in primary region   */
            i_link(iatom_l+1) = i
            i_link(iatom_l+2) = i
            i_link(iatom_l+3) = i

!           /*   bonded atom in secondary region   */
            j_link(iatom_l+1) = j1
            j_link(iatom_l+2) = j2
            j_link(iatom_l+3) = j3

!           /*   bond ratio   */
            r_link(iatom_l+1) = r1
            r_link(iatom_l+2) = r2
            r_link(iatom_l+3) = r3

!           /*   atom species   */
            species_link(iatom_l+1) = a1
            species_link(iatom_l+2) = a2
            species_link(iatom_l+3) = a3

!           /*   update link atom counter   */
            iatom_l = iatom_l + 3

!           /*   bonded atom in secondary region   */
            layer(j1) = 'BL'
            layer(j2) = 'BL'
            layer(j3) = 'BL'

!        /*   line with four link atoms   */
         else if ( layer(i)(1:3) .eq. 'AL4' ) then

!           /*   read line   */
            read ( char_line, *, iostat=ierr ) &
     &         char_1, char_2, char_3, char_4, char_5, char_6, &
     &         j1, r1, a1, j2, r2, a2, j3, r3, a3, j4, r4, a4

!           /*   if read error found   */
            if ( ierr .ne. 0 ) then

!              /*   set error flag   */
               jerr = 2

!              /*   stop   */
               exit

!           /*   if read error found   */
            end if

!           /*   bonded atom in primary region   */
            i_link(iatom_l+1) = i
            i_link(iatom_l+2) = i
            i_link(iatom_l+3) = i
            i_link(iatom_l+4) = i

!           /*   bonded atom in secondary region   */
            j_link(iatom_l+1) = j1
            j_link(iatom_l+2) = j2
            j_link(iatom_l+3) = j3
            j_link(iatom_l+4) = j4

!           /*   bond ratio   */
            r_link(iatom_l+1) = r1
            r_link(iatom_l+2) = r2
            r_link(iatom_l+3) = r3
            r_link(iatom_l+4) = r4

!           /*   atom species   */
            species_link(iatom_l+1) = a1
            species_link(iatom_l+2) = a2
            species_link(iatom_l+3) = a3
            species_link(iatom_l+4) = a4

!           /*   update link atom counter   */
            iatom_l = iatom_l + 4

!           /*   bonded atom in secondary region   */
            layer(j1) = 'BL'
            layer(j2) = 'BL'
            layer(j3) = 'BL'
            layer(j4) = 'BL'

!        /*   end of if statement   */
         end if

!     /*   loop of atoms   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   master rank only   */
      end if

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( jerr )

!-----------------------------------------------------------------------
!     /*   error check                                                */
!-----------------------------------------------------------------------

!     /*   error termination   */
      call error_handling_MPI( jerr, 'subroutine setup_ee_MPI', 23 )

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      j = len(species_link)

      call my_mpi_bcast_int_1 ( i_link, natom_l )
      call my_mpi_bcast_int_1 ( j_link, natom_l )
      call my_mpi_bcast_real_1( r_link, natom_l )
      call my_mpi_bcast_char_1( species_link, j, natom_l )
      call my_mpi_bcast_char_1( layer, len(layer), natom )

!-----------------------------------------------------------------------
!     /*   read atomic symbols                                        */
!-----------------------------------------------------------------------

!     /*   number of atoms   */
      call read_int1_MPI ( nsymbol, '<nsymbol>', 9, iounit )

!     /*   memory allocation: atomic symbols   */
      if ( .not. allocated( symbol ) ) &
     &   allocate( symbol(nsymbol) )

!     /*   memory allocation: atomic symbols   */
      if ( .not. allocated( num_symbol ) ) &
     &   allocate( num_symbol(nsymbol) )

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<nsymbol>', 9, iounit, ierr )

!     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr )

!     /*   loop of symbols   */
      do i = 1, nsymbol

!        /*   read symbol, atomic number, atomic mass   */
         read ( iounit, *, iostat=ierr ) symbol(i), num_symbol(i)

!     /*   loop of symbols   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   on error, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<nsymbol>', 9, iounit, ierr )

!        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr )

!        /*   loop of symbols   */
         do i = 1, nsymbol

!           /*   read symbol, atomic number, atomic mass   */
            read ( iounit, *, iostat=ierr ) symbol(i), num_symbol(i)

!        /*   loop of symbols   */
         end do

!        /*   file close   */
         close( iounit )

!     /*   on error, read default values   */
      end if

!     /*   master rank only   */
      end if

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_char_1( symbol, len(symbol), nsymbol )
      call my_mpi_bcast_int_1( num_symbol, nsymbol )

!-----------------------------------------------------------------------
!     /*   physmass: atomic mass of each atom                         */
!-----------------------------------------------------------------------

!     /*   flag   */
      ierr = 0

!     /*   loop of atoms   */
      do i = 1, natom_l

!        /*   flag   */
         k = 0

!        /*   loop of symbols   */
         do l = 1, nsymbol

!           /*   if symbol matched   */
            if ( species_link(i) .eq. symbol(l) ) then

!              /*   flag   */
               k = 1

!              /*   substitute atomic number   */
               int_spec_link(i) = num_symbol(l)

!              /*   go to next loop   */
               exit

!           /*   if symbol matched   */
            end if

!        /*   loop of symbols   */
         end do

!        /*   error flag   */
         if ( k .eq. 0 ) then

!           /*   error flag   */
            ierr = 1

!           /*   exit from the loop  */
            exit

!        /*   error flag   */
         end if

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   error check                                                */
!-----------------------------------------------------------------------

!     /*   error termination   */
      call error_handling_MPI( ierr, 'subroutine setup_ee_MPI', 23 )

      return
      end

