!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     dual level path integral hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_dual
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead, iounit, iounit_dual

      use dual_variables, only : &
     &   x_lo, y_lo, z_lo, e_lo, fx_lo, fy_lo, fz_lo, dipx_lo, dipy_lo, &
     &   dipz_lo, pot_low, fx_low, fy_low, fz_low, dipx_low, dipy_low, &
     &   dipz_low, vir_low, dipy_high, dipz_high, vir_high, &
     &   x_hi, y_hi, z_hi, e_hi, fx_hi, fy_hi, fz_hi, dipx_hi, dipy_hi, &
     &   dipz_hi, pot_high, fx_high, fy_high, fz_high, dipx_high, &
     &   dat_dir_hi, dat_dir_lo, scr_dir_hi, scr_dir_lo, idual_hi, &
     &   idual_lo, iprint_dual, x_trial, y_trial, z_trial, params_dual

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<dual_potential>', 16, iounit, ierr )

!        /*   dual:  high and low levels   */
         read ( iounit, *, iostat=ierr ) idual_hi, idual_lo

!        /*   on error read default   */
         if ( ierr .ne. 0 ) then

!           /*   file close   */
            close( iounit )

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<dual_potential>', 16, iounit, ierr )

!           /*   dual:  high and low levels   */
            read ( iounit, *, iostat=ierr ) idual_hi, idual_lo

!        /*   on error read default   */
         end if

!        /*   file close   */
         close( iounit )

!        /*   stop on error   */
         call error_handling( ierr, 'subroutine setup_dual', 21)

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<dual_dat_dir>', 14, iounit, ierr )

!        /*   data directory:  high and low levels   */
         read ( iounit, *, iostat=ierr ) dat_dir_hi, dat_dir_lo

!        /*   on error read default   */
         if ( ierr .ne. 0 ) then

!           /*   file close   */
            close( iounit )

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<dual_dat_dir>', 14, iounit, ierr )

!           /*   data directory:  high and low levels   */
            read ( iounit, *, iostat=ierr ) dat_dir_hi, dat_dir_lo

!        /*   on error read default   */
         end if

!        /*   file close   */
         close( iounit )

!        /*   stop on error   */
         call error_handling( ierr, 'subroutine setup_dual', 21)

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<dual_scr_dir>', 14, iounit, ierr )

!        /*   data directory:  high and low levels   */
         read ( iounit, *, iostat=ierr ) scr_dir_hi, scr_dir_lo

!        /*   on error read default   */
         if ( ierr .ne. 0 ) then

!           /*   file close   */
            close( iounit )

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<dual_scr_dir>', 14, iounit, ierr )

!           /*   data directory:  high and low levels   */
            read ( iounit, *, iostat=ierr ) scr_dir_hi, scr_dir_lo

!        /*   on error read default   */
         end if

!        /*   file close   */
         close( iounit )

!        /*   stop on error   */
         call error_handling( ierr, 'subroutine setup_dual', 21)

!-----------------------------------------------------------------------
!        /*   memory allocation                                       */
!-----------------------------------------------------------------------

         if ( .not. allocated( x_lo ) ) &
     &      allocate( x_lo(natom) )
         if ( .not. allocated( y_lo ) ) &
     &      allocate( y_lo(natom) )
         if ( .not. allocated( z_lo ) ) &
     &      allocate( z_lo(natom) )

         if ( .not. allocated( e_lo ) ) &
     &      allocate( e_lo(1) )

         if ( .not. allocated( fx_lo ) ) &
     &      allocate( fx_lo(natom) )
         if ( .not. allocated( fy_lo ) ) &
     &      allocate( fy_lo(natom) )
         if ( .not. allocated( fz_lo ) ) &
     &      allocate( fz_lo(natom) )

         if ( .not. allocated( dipx_lo ) ) &
     &      allocate( dipx_lo(nbead) )
         if ( .not. allocated( dipy_lo ) ) &
     &      allocate( dipy_lo(nbead) )
         if ( .not. allocated( dipz_lo ) ) &
     &      allocate( dipz_lo(nbead) )

         if ( .not. allocated( pot_low ) ) &
     &      allocate( pot_low(nbead) )

         if ( .not. allocated( fx_low ) ) &
     &      allocate( fx_low(natom,nbead) )
         if ( .not. allocated( fy_low ) ) &
     &      allocate( fy_low(natom,nbead) )
         if ( .not. allocated( fz_low ) ) &
     &      allocate( fz_low(natom,nbead) )

         if ( .not. allocated( vir_low ) ) &
     &      allocate( vir_low(3,3) )

         if ( .not. allocated( dipx_low ) ) &
     &      allocate( dipx_low(nbead) )
         if ( .not. allocated( dipy_low ) ) &
     &      allocate( dipy_low(nbead) )
         if ( .not. allocated( dipz_low ) ) &
     &      allocate( dipz_low(nbead) )

         if ( .not. allocated( x_hi ) ) &
     &      allocate( x_hi(natom) )
         if ( .not. allocated( y_hi ) ) &
     &      allocate( y_hi(natom) )
         if ( .not. allocated( z_hi ) ) &
     &      allocate( z_hi(natom) )

         if ( .not. allocated( e_hi ) ) &
     &      allocate( e_hi(1) )

         if ( .not. allocated( x_trial ) ) &
     &      allocate( x_trial(natom, nbead) )
         if ( .not. allocated( y_trial ) ) &
     &      allocate( y_trial(natom, nbead) )
         if ( .not. allocated( z_trial ) ) &
     &      allocate( z_trial(natom, nbead) )

         if ( .not. allocated( fx_hi ) ) &
     &      allocate( fx_hi(natom) )
         if ( .not. allocated( fy_hi ) ) &
     &      allocate( fy_hi(natom) )
         if ( .not. allocated( fz_hi ) ) &
     &      allocate( fz_hi(natom) )

         if ( .not. allocated( dipx_hi ) ) &
     &      allocate( dipx_hi(nbead) )
         if ( .not. allocated( dipy_hi ) ) &
     &      allocate( dipy_hi(nbead) )
         if ( .not. allocated( dipz_hi ) ) &
     &      allocate( dipz_hi(nbead) )

         if ( .not. allocated( pot_high ) ) &
     &      allocate( pot_high(nbead) )

         if ( .not. allocated( fx_high ) ) &
     &      allocate( fx_high(natom,nbead) )
         if ( .not. allocated( fy_high ) ) &
     &      allocate( fy_high(natom,nbead) )
         if ( .not. allocated( fz_high ) ) &
     &      allocate( fz_high(natom,nbead) )

         if ( .not. allocated( vir_high ) ) &
     &      allocate( vir_high(3,3) )

         if ( .not. allocated( dipx_high ) ) &
     &      allocate( dipx_high(nbead) )
         if ( .not. allocated( dipy_high ) ) &
     &      allocate( dipy_high(nbead) )
         if ( .not. allocated( dipz_high ) ) &
     &      allocate( dipz_high(nbead) )

         if ( .not. allocated(params_dual) ) &
     &      allocate(params_dual(2))

!-----------------------------------------------------------------------
!        /*   parameters                                              */
!-----------------------------------------------------------------------

         call read_realn &
     &      ( params_dual, 2, '<params_dual>', 13, iounit )

!-----------------------------------------------------------------------
!        /*   set up dual-level calculation                           */
!-----------------------------------------------------------------------

!        /*   same potentials   */
         if ( idual_hi(1:len_trim(idual_hi)) .eq. &
     &        idual_lo(1:len_trim(idual_lo)) ) then

!           /*   low level   */
            call setup_dual_lo

!           /*   high level   */
            call setup_dual_hi

!        /*   same potentials   */
         end if

!-----------------------------------------------------------------------
!        /*   prepare for output                                      */
!-----------------------------------------------------------------------

         call read_int1 ( iprint_dual, '<iprint_dual>', 13, iounit )

         call testfile ( 'dual.out', 8, itest, iounit )

         if ( itest .eq. 1 ) then

            open ( iounit_dual, file = 'dual.out' )

            write(iounit_dual,'(a)') &
     &         '----------------------------------------' // &
     &         '--------------------------------'
            write(iounit_dual,'(a)') &
     &         '    step          pot_hi          pot_lo' // &
     &         '      average_hi      average_lo'
            write(iounit_dual,'(a)') &
     &         '----------------------------------------' // &
     &         '--------------------------------'

         else

            open ( iounit_dual, file = 'dual.out', access = 'append' )

         end if

         iset = 1

      end if

      return
      end





!***********************************************************************
      subroutine setup_dual_hi
!***********************************************************************

#ifndef aenet

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   amu_mass, au_mass, x, y, z, box, nbead, iounit, iboundary, &
     &   natom, species, ikind

      use dual_variables, only : &
     &   idual_hi, dat_dir_hi, scr_dir_hi

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      character(len=80) :: char_file
      character(len=3)  :: char_num

!-----------------------------------------------------------------------
!     /*   make directory                                             */
!-----------------------------------------------------------------------

!      call system ( 'mkdir -p ' // trim(scr_dir_hi) )

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do j = 1, nbead

!-----------------------------------------------------------------------
!        /*   make directory                                          */
!-----------------------------------------------------------------------

!        /*   integer to character   */
         call int3_to_char( j, char_num )

!        /*   make directory   */
         call system('mkdir -p ' // trim(scr_dir_hi) // '/' // char_num)

!-----------------------------------------------------------------------
!        /*   copy files                                              */
!-----------------------------------------------------------------------

!        /*   copy input_default.dat   */
         call system ( 'cp -f input_default.dat ' // &
     &                 trim(scr_dir_hi) // '/' // char_num )

!        /*   copy *.dat   */
         call system ( 'cp -f ' // trim(dat_dir_hi) // '/* ' // &
     &                 trim(scr_dir_hi) // '/' // char_num )

!-----------------------------------------------------------------------
!        /*   make input.dat                                          */
!-----------------------------------------------------------------------

         char_file = trim(scr_dir_hi) // '/' // char_num // &
     &               '/input.dat'

         open ( iounit, file = char_file )

         write(iounit,'(a)')   '<method>'
         write(iounit,'(a)')   'STATIC'
         write(iounit,*)

         write(iounit,'(a)')   '<input_style>'
         write(iounit,'(a)')   'NEW'
         write(iounit,*)

         write(iounit,'(a)')   '<ipotential>'
         write(iounit,'(a)')   idual_hi
         write(iounit,*)

         write(iounit,'(a)')   '<iboundary>'
         write(iounit,'(i8)')  iboundary
         if      ( iboundary .eq. 1 ) then
            write(iounit,'(3e24.16)') box(1,1), box(1,2), box(1,3)
            write(iounit,'(3e24.16)') box(2,1), box(2,2), box(2,3)
            write(iounit,'(3e24.16)') box(3,1), box(3,2), box(3,3)
         else if ( iboundary .eq. 2 ) then
            write(iounit,'(3e24.16)') box(1,1), box(1,2), box(1,3)
            write(iounit,'(3e24.16)') box(2,1), box(2,2), box(2,3)
            write(iounit,'(3e24.16)') box(3,1), box(3,2), box(3,3)
         end if
         write(iounit,*)

         write( iounit, '(a)' ) '<iprint_dip>'
         write( iounit, '(i8)' ) 1

         close(iounit)

         char_file = trim(scr_dir_hi) // '/' // char_num // &
     &               '/structure.dat'

         open ( iounit, file = char_file )

         write(iounit,'(i8)')  natom
         write(iounit,'(a)') 'BOHR'
         do i = 1, natom
            write(iounit,'(a,3f24.16,i4)') &
     &         species(i), x(i,j), y(i,j), z(i,j), ikind(i)
         end do

         close(iounit)

      end do

#endif

      return
      end





!***********************************************************************
      subroutine setup_dual_lo
!***********************************************************************

#ifndef aenet

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   amu_mass, au_mass, x, y, z, box, nbead, iounit, iboundary, &
     &   natom, species, ikind

      use dual_variables, only : &
     &   idual_lo, dat_dir_lo, scr_dir_lo

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      character(len=80) :: char_file
      character(len=3)  :: char_num

!-----------------------------------------------------------------------
!     /*   make directory                                             */
!-----------------------------------------------------------------------

!      call system ( 'mkdir -p ' // trim(scr_dir_hi) )

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do j = 1, nbead

!-----------------------------------------------------------------------
!        /*   make directory                                          */
!-----------------------------------------------------------------------

!        /*   integer to character   */
         call int3_to_char( j, char_num )

!        /*   make directory   */
         call system('mkdir -p ' // trim(scr_dir_lo) // '/' // char_num)

!-----------------------------------------------------------------------
!        /*   copy files                                              */
!-----------------------------------------------------------------------

!        /*   copy input_default.dat   */
         call system ( 'cp -f input_default.dat ' // &
     &                 trim(scr_dir_lo) // '/' // char_num )

!        /*   copy *.dat   */
         call system ( 'cp -f ' // trim(dat_dir_lo) // '/* ' // &
     &                 trim(scr_dir_lo) // '/' // char_num )

!-----------------------------------------------------------------------
!        /*   make input.dat                                          */
!-----------------------------------------------------------------------

         char_file = trim(scr_dir_lo) // '/' // char_num // &
     &               '/input.dat'

         open ( iounit, file = char_file )

         write(iounit,'(a)')   '<method>'
         write(iounit,'(a)')   'STATIC'
         write(iounit,*)

         write(iounit,'(a)')   '<input_style>'
         write(iounit,'(a)')   'NEW'
         write(iounit,*)

         write(iounit,'(a)')   '<ipotential>'
         write(iounit,'(a)')   idual_lo
         write(iounit,*)

         write(iounit,'(a)')   '<iboundary>'
         write(iounit,'(i8)')  iboundary
         if      ( iboundary .eq. 1 ) then
            write(iounit,'(3e24.16)') box(1,1), box(1,2), box(1,3)
            write(iounit,'(3e24.16)') box(2,1), box(2,2), box(2,3)
            write(iounit,'(3e24.16)') box(3,1), box(3,2), box(3,3)
         else if ( iboundary .eq. 2 ) then
            write(iounit,'(3e24.16)') box(1,1), box(1,2), box(1,3)
            write(iounit,'(3e24.16)') box(2,1), box(2,2), box(2,3)
            write(iounit,'(3e24.16)') box(3,1), box(3,2), box(3,3)
         end if
         write(iounit,*)

         write( iounit, '(a)' ) '<iprint_dip>'
         write( iounit, '(i8)' ) 1

         close(iounit)

         char_file = trim(scr_dir_lo) // '/' // char_num // &
     &               '/structure.dat'

         open ( iounit, file = char_file )

         write(iounit,'(i8)')  natom
         write(iounit,'(a)') 'BOHR'
         do i = 1, natom
            write(iounit,'(a,3f24.16,i4)') &
     &         species(i), x(i,j), y(i,j), z(i,j), ikind(i)
         end do

         close(iounit)

      end do

#endif

      return
      end





!***********************************************************************
      subroutine setup_dual_rem
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead, iounit, iounit_dual

      use dual_variables, only : &
     &   x_lo, y_lo, z_lo, e_lo, fx_lo, fy_lo, fz_lo, dipx_lo, dipy_lo, &
     &   dipz_lo, pot_low, fx_low, fy_low, fz_low, dipx_low, dipy_low, &
     &   dipz_low, vir_low, dipy_high, dipz_high, vir_bead_high, &
     &   x_hi, y_hi, z_hi, e_hi, fx_hi, fy_hi, fz_hi, dipx_hi, dipy_hi, &
     &   dipz_hi, pot_high, fx_high, fy_high, fz_high, dipx_high, &
     &   dat_dir_hi, dat_dir_lo, scr_dir_hi, scr_dir_lo, idual_hi, &
     &   idual_lo, iprint_dual

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag   */
         call search_tag ( '<dual_potential>', 16, iounit, ierr )

!        /*   dual:  high and low levels   */
         read ( iounit, *, iostat=ierr ) idual_hi, idual_lo

!        /*   on error read default   */
         if ( ierr .ne. 0 ) then

!           /*   file close   */
            close( iounit )

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<dual_potential>', 16, iounit, ierr )

!           /*   dual:  high and low levels   */
            read ( iounit, *, iostat=ierr ) idual_hi, idual_lo

!        /*   on error read default   */
         end if

!        /*   file close   */
         close( iounit )

!        /*   stop on error   */
         call error_handling &
     &      ( ierr, 'subroutine setup_dual_rem', 25 )

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<dual_dat_dir>', 14, iounit, ierr )

!        /*   data directory:  high and low levels   */
         read ( iounit, *, iostat=ierr ) dat_dir_hi, dat_dir_lo

!        /*   on error read default   */
         if ( ierr .ne. 0 ) then

!           /*   file close   */
            close( iounit )

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<dual_dat_dir>', 14, iounit, ierr )

!           /*   data directory:  high and low levels   */
            read ( iounit, *, iostat=ierr ) dat_dir_hi, dat_dir_lo

!        /*   on error read default   */
         end if

!        /*   file close   */
         close( iounit )

!        /*   stop on error   */
         call error_handling &
     &      ( ierr, 'subroutine setup_dual_rem', 25 )

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<dual_scr_dir>', 14, iounit, ierr )

!        /*   data directory:  high and low levels   */
         read ( iounit, *, iostat=ierr ) scr_dir_hi, scr_dir_lo

!        /*   on error read default   */
         if ( ierr .ne. 0 ) then

!           /*   file close   */
            close( iounit )

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<dual_scr_dir>', 14, iounit, ierr )

!           /*   data directory:  high and low levels   */
            read ( iounit, *, iostat=ierr ) scr_dir_hi, scr_dir_lo

!        /*   on error read default   */
         end if

!        /*   file close   */
         close( iounit )

!        /*   stop on error   */
         call error_handling &
     &      ( ierr, 'subroutine setup_dual_rem', 25 )

!-----------------------------------------------------------------------
!        /*   memory allocation                                       */
!-----------------------------------------------------------------------

         if ( .not. allocated( x_lo ) ) &
     &      allocate( x_lo(natom) )
         if ( .not. allocated( y_lo ) ) &
     &      allocate( y_lo(natom) )
         if ( .not. allocated( z_lo ) ) &
     &      allocate( z_lo(natom) )

         if ( .not. allocated( e_lo ) ) &
     &      allocate( e_lo(1) )

         if ( .not. allocated( fx_lo ) ) &
     &      allocate( fx_lo(natom) )
         if ( .not. allocated( fy_lo ) ) &
     &      allocate( fy_lo(natom) )
         if ( .not. allocated( fz_lo ) ) &
     &      allocate( fz_lo(natom) )

         if ( .not. allocated( dipx_lo ) ) &
     &      allocate( dipx_lo(nbead) )
         if ( .not. allocated( dipy_lo ) ) &
     &      allocate( dipy_lo(nbead) )
         if ( .not. allocated( dipz_lo ) ) &
     &      allocate( dipz_lo(nbead) )

         if ( .not. allocated( pot_low ) ) &
     &      allocate( pot_low(nbead) )

         if ( .not. allocated( fx_low ) ) &
     &      allocate( fx_low(natom,nbead) )
         if ( .not. allocated( fy_low ) ) &
     &      allocate( fy_low(natom,nbead) )
         if ( .not. allocated( fz_low ) ) &
     &      allocate( fz_low(natom,nbead) )

         if ( .not. allocated( vir_low ) ) &
     &      allocate( vir_low(3,3) )

         if ( .not. allocated( dipx_low ) ) &
     &      allocate( dipx_low(nbead) )
         if ( .not. allocated( dipy_low ) ) &
     &      allocate( dipy_low(nbead) )
         if ( .not. allocated( dipz_low ) ) &
     &      allocate( dipz_low(nbead) )

         if ( .not. allocated( x_hi ) ) &
     &      allocate( x_hi(natom) )
         if ( .not. allocated( y_hi ) ) &
     &      allocate( y_hi(natom) )
         if ( .not. allocated( z_hi ) ) &
     &      allocate( z_hi(natom) )

         if ( .not. allocated( e_hi ) ) &
     &      allocate( e_hi(1) )

         if ( .not. allocated( fx_hi ) ) &
     &      allocate( fx_hi(natom) )
         if ( .not. allocated( fy_hi ) ) &
     &      allocate( fy_hi(natom) )
         if ( .not. allocated( fz_hi ) ) &
     &      allocate( fz_hi(natom) )

         if ( .not. allocated( dipx_hi ) ) &
     &      allocate( dipx_hi(nbead) )
         if ( .not. allocated( dipy_hi ) ) &
     &      allocate( dipy_hi(nbead) )
         if ( .not. allocated( dipz_hi ) ) &
     &      allocate( dipz_hi(nbead) )

         if ( .not. allocated( pot_high ) ) &
     &      allocate( pot_high(nbead) )

         if ( .not. allocated( fx_high ) ) &
     &      allocate( fx_high(natom,nbead) )
         if ( .not. allocated( fy_high ) ) &
     &      allocate( fy_high(natom,nbead) )
         if ( .not. allocated( fz_high ) ) &
     &      allocate( fz_high(natom,nbead) )

         if ( .not. allocated( vir_bead_high ) ) &
     &      allocate( vir_bead_high(3,3,nbead) )

         if ( .not. allocated( dipx_high ) ) &
     &      allocate( dipx_high(nbead) )
         if ( .not. allocated( dipy_high ) ) &
     &      allocate( dipy_high(nbead) )
         if ( .not. allocated( dipz_high ) ) &
     &      allocate( dipz_high(nbead) )

!-----------------------------------------------------------------------
!        /*   set up dual-level calculation                           */
!-----------------------------------------------------------------------

!        /*   same potentials   */
         if ( idual_hi(1:len_trim(idual_hi)) .eq. &
     &        idual_lo(1:len_trim(idual_lo)) ) then

!           /*   error termination   */
            call error_handling &
     &         ( 1, 'subroutine setup_dual_rem', 25 )

!        /*   same potentials   */
         end if

!-----------------------------------------------------------------------
!        /*   prepare for output                                      */
!-----------------------------------------------------------------------

         call read_int1 ( iprint_dual, '<iprint_dual>', 13, iounit )

         call testfile ( 'dual.out', 8, itest, iounit )

         if ( itest .eq. 1 ) then

            open ( iounit_dual, file = 'dual.out' )

            write(iounit_dual,'(a)') &
     &      '----------------------------------------' // &
     &      '--------------------------------'
            write(iounit_dual,'(a)') &
     &      '    step          pot_hi          pot_lo' // &
     &      '      average_hi      average_lo'
            write(iounit_dual,'(a)') &
     &      '----------------------------------------' // &
     &      '--------------------------------'

         else

            open( iounit_dual, file = 'dual.out', access = 'append' )

         end if

         iset = 1

      end if

      return
      end





!***********************************************************************
      subroutine getforce_dual_lo_rem
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, dipx, dipy, dipz, pot, vir_bead, potential, &
     &   ipotential

      use dual_variables, only : &
     &   pot_low, potential_low, fx_low, fy_low, fz_low, vir_bead_low, &
     &   dipx_low, dipy_low, dipz_low, vir_bead_low, idual_lo

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      call setup_dual_rem

!-----------------------------------------------------------------------
!     /*   force calculation                                          */
!-----------------------------------------------------------------------

      ipotential = idual_lo

      call getforce_rem

!-----------------------------------------------------------------------
!     /*   save                                                       */
!-----------------------------------------------------------------------

      pot_low(:)        = pot(:)

      potential_low     = potential

      fx_low(:,:)       = fx(:,:)
      fy_low(:,:)       = fy(:,:)
      fz_low(:,:)       = fz(:,:)

      vir_bead_low(:,:,:) = vir_bead(:,:,:)

      dipx_low(:)       = dipx(:)
      dipy_low(:)       = dipy(:)
      dipz_low(:)       = dipz(:)

      return
      end





!***********************************************************************
      subroutine getforce_dual_hi_rem
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, dipx, dipy, dipz, pot, vir_bead, potential, &
     &   ipotential, istep, nbead, iounit_dual

      use dual_variables, only : &
     &   pot_high, potential_high, fx_high, fy_high, fz_high, &
     &   dipx_high, dipy_high, dipz_high, pot_low, potential_low, &
     &   vir_bead_high, iprint_dual, idual_hi

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      call setup_dual_rem

!-----------------------------------------------------------------------
!     /*   force calculation                                          */
!-----------------------------------------------------------------------

      ipotential = idual_hi

      call getforce_rem

!-----------------------------------------------------------------------
!     /*   save                                                       */
!-----------------------------------------------------------------------

      pot_high(:)    = pot(:)

      potential_high = potential

      fx_high(:,:)   = fx(:,:)
      fy_high(:,:)   = fy(:,:)
      fz_high(:,:)   = fz(:,:)

      vir_bead_high(:,:,:) = vir_bead(:,:,:)

      dipx_high(:)   = dipx(:)
      dipy_high(:)   = dipy(:)
      dipz_high(:)   = dipz(:)

!-----------------------------------------------------------------------
!     /*   print energies                                             */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_dual) .eq. 0 ) then

         do i = 1, nbead

            write ( iounit_dual, '(i8,4f16.8)' ) &
     &         istep, pot_high(i), pot_low(i), &
     &         potential_high, potential_low

         end do

      end if

      return
      end

