!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 26, 2023 by M. Shiga
!      Description:     set up constraints
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_cons_geoopt_MPI
!***********************************************************************
!=======================================================================
!
!     read parameters for constraints
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   dt, dt_ref, iounit, natom, nbead, nref, method, code, myrank

      use cons_variables, only : &
     &   rcons, scons, fc_cons, fref_cons, pot_ref_cons, req_cons, &
     &   params_cons, fx_ref_cons, fy_ref_cons, fz_ref_cons, &
     &   gcons, hcons, i_cons, j_cons, k_cons, l_cons, &
     &   ipbc_cons, mu_cons, nu_cons, ncons, itype_cons, ntype_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, ierr

      integer, save :: iset = 0

      character(len=8) :: params_char, char_cons

!-----------------------------------------------------------------------
!     /*   first time only                                            */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then
         iset = 1
      else
         return
      end if

!-----------------------------------------------------------------------
!     /*   default                                                    */
!-----------------------------------------------------------------------

      ierr = 0

!-----------------------------------------------------------------------
!     /*   read number of constraints                                 */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( ncons, '<ncons>', 7, iounit )

!-----------------------------------------------------------------------
!     /*   return if there are no constraints                         */
!-----------------------------------------------------------------------

      if ( ncons .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   position   */
      if ( .not. allocated( rcons ) ) &
     &   allocate( rcons(ncons,nbead) )

!     /*   actual position   */
      if ( .not. allocated( scons ) ) &
     &   allocate( scons(ncons,nbead) )

!     /*   type of constraint   */
      if ( .not. allocated( itype_cons ) ) &
     &   allocate( itype_cons(ncons) )

!     /*   potential from lagrangian contribution    */
      if ( .not. allocated( pot_ref_cons ) ) &
     &   allocate( pot_ref_cons(1) )

!     /*   lagrangian    */
      if ( .not. allocated( gcons ) ) &
     &   allocate( gcons(1) )

!     /*   lagrangian    */
      if ( .not. allocated( hcons ) ) &
     &   allocate( hcons(1) )

!     /*   force constant of harmonic potential   */
      if ( .not. allocated( fc_cons ) ) &
     &   allocate( fc_cons(ncons) )

!     /*   atoms i, j, k, l of constraint   */
      if ( .not. allocated( i_cons ) ) &
     &   allocate( i_cons(ncons) )
      if ( .not. allocated( j_cons ) ) &
     &   allocate( j_cons(ncons) )
      if ( .not. allocated( k_cons ) ) &
     &   allocate( k_cons(ncons) )
      if ( .not. allocated( l_cons ) ) &
     &   allocate( l_cons(ncons) )

!     /*   rational function parameters for coordination number   */
      if ( .not. allocated( nu_cons ) ) &
     &   allocate( nu_cons(ncons,2) )
      if ( .not. allocated( mu_cons ) ) &
     &   allocate( mu_cons(ncons,2) )
      if ( .not. allocated( req_cons ) ) &
     &   allocate( req_cons(ncons,2) )

!     /*   parameters for constraints   */
      if ( .not. allocated( params_cons ) ) &
     &   allocate( params_cons(ntype_cons) )

!     /*   boundary condition of cv   */
      if ( .not. allocated( ipbc_cons ) ) &
     &   allocate( ipbc_cons(ncons) )

!     /*   reference forces for PIMD  */
      if ( .not. allocated( fx_ref_cons ) ) &
     &   allocate( fx_ref_cons(natom,nbead) )
      if ( .not. allocated( fy_ref_cons ) ) &
     &   allocate( fy_ref_cons(natom,nbead) )
      if ( .not. allocated( fz_ref_cons ) ) &
     &   allocate( fz_ref_cons(natom,nbead) )

!-----------------------------------------------------------------------
!     /*   number of reference steps for harmonic potential           */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( nref, '<nref>', 6, iounit )

!-----------------------------------------------------------------------
!     /*   step size for harmonic potential                           */
!-----------------------------------------------------------------------

      dt_ref = dt/dble(nref)

!-----------------------------------------------------------------------
!     /*   read type of constraints                                   */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<ncons>', 7, iounit, ierr )

!     /*   read integer   */
      read ( iounit, *, iostat=ierr )

      do k = 1, ncons

         read ( iounit, *, iostat=ierr ) char_cons

         backspace( iounit )

         if      ( ( char_cons(1:6) .eq. '1     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIST  ' ) ) then
            itype_cons(k) = 1
         else if ( ( char_cons(1:6) .eq. '2     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'ANGL  ' ) ) then
            itype_cons(k) = 2
         else if ( ( char_cons(1:6) .eq. '3     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIH   ' ) ) then
            itype_cons(k) = 3
         else if ( ( char_cons(1:6) .eq. '4     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DIFF  ' ) ) then
            itype_cons(k) = 4
         else if ( ( char_cons(1:6) .eq. '5     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'CN    ' ) ) then
            itype_cons(k) = 5
         else if ( ( char_cons(1:6) .eq. '6     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DCN   ' ) ) then
            itype_cons(k) = 6
         else if ( ( char_cons(1:6) .eq. '7     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'XYZ   ' ) ) then
            itype_cons(k) = 7
         else if ( ( char_cons(1:6) .eq. '8     ' ) .or. &
     &             ( char_cons(1:6) .eq. 'DXYZ  ' ) ) then
            itype_cons(k) = 8
         else
            ierr = 1
            exit
         end if

         if ( itype_cons(k) .eq. 1 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_cons, i_cons(k), j_cons(k), &
     &         rcons(k,1)
            if ( ierr .ne. 0 ) exit
         else if ( itype_cons(k) .eq. 2 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_cons, i_cons(k), j_cons(k), k_cons(k), &
     &         rcons(k,1)
            if ( ierr .ne. 0 ) exit
         else if ( itype_cons(k) .eq. 3 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_cons, i_cons(k), j_cons(k), k_cons(k), &
     &         l_cons(k), rcons(k,1)
            if ( ierr .ne. 0 ) exit
         else if ( itype_cons(k) .eq. 4 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_cons, i_cons(k), j_cons(k), k_cons(k), &
     &         rcons(k,1)
            if ( ierr .ne. 0 ) exit
         else if ( itype_cons(k) .eq. 5 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_cons, i_cons(k), j_cons(k), &
     &         nu_cons(k,1), mu_cons(k,1), req_cons(k,1), &
     &         rcons(k,1)
            if ( ierr .ne. 0 ) exit
         else if ( itype_cons(k) .eq. 6 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_cons, i_cons(k), j_cons(k), &
     &         nu_cons(k,1), mu_cons(k,1), req_cons(k,1), &
     &                    k_cons(k), l_cons(k), &
     &         nu_cons(k,2), mu_cons(k,2), req_cons(k,2), &
     &         rcons(k,1)
            if ( ierr .ne. 0 ) exit
         else if ( itype_cons(k) .eq. 7 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_cons, i_cons(k), j_cons(k), rcons(k,1)
            if ( ierr .ne. 0 ) exit
         else if ( itype_cons(k) .eq. 8 ) then
            read ( iounit, *, iostat=ierr ) &
     &         char_cons, i_cons(k), j_cons(k), k_cons(k), rcons(k,1)
            if ( ierr .ne. 0 ) exit
         else
            ierr = 1
            exit
         end if

         do i = 2, nbead
            rcons(k,i) = rcons(k,1)
         end do

      end do

!     /*   file close   */
      close( iounit )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <ncons> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI ( ierr, 'subroutine setup_cons_MPI', 25 )

!     /*   broadcast   */
      call my_mpi_bcast_int_1 ( i_cons, ncons )
      call my_mpi_bcast_int_1 ( j_cons, ncons )
      call my_mpi_bcast_int_1 ( k_cons, ncons )
      call my_mpi_bcast_int_1 ( l_cons, ncons )
      call my_mpi_bcast_int_2 ( nu_cons, ncons, 2 )
      call my_mpi_bcast_int_2 ( mu_cons, ncons, 2 )
      call my_mpi_bcast_real_2 ( req_cons, ncons, 2 )
      call my_mpi_bcast_real_2 ( rcons, ncons, nbead )

!-----------------------------------------------------------------------
!     /*   read parameters for constraint                             */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<params_cons>', 13, iounit, ierr )

!     /*   rms displacement of constraint type   */

      do i = 1, ntype_cons

         read ( iounit, *, iostat=ierr ) params_char

         backspace( iounit )

         if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIST  ' ) ) then
            k = 1
         else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &             ( params_char(1:6) .eq. 'ANGL  ' ) ) then
            k = 2
         else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIH   ' ) ) then
            k = 3
         else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIFF  ' ) ) then
            k = 4
         else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &             ( params_char(1:6) .eq. 'CN    ' ) ) then
            k = 5
         else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DCN   ' ) ) then
            k = 6
         else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &             ( params_char(1:6) .eq. 'XYZ   ' ) ) then
            k = 7
         else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
            k = 8
         else
            ierr = 1
         end if

         if ( ierr .ne. 0 ) exit

         read ( iounit, *, iostat=ierr ) params_char, params_cons(k)

         if ( ierr .ne. 0 ) exit

      end do

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<params_cons>', 13, iounit, ierr )

!        /*   rms displacement of constraint type   */

         do i = 1, ntype_cons

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) params_char, params_cons(k)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <params_cons> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI ( ierr, 'subroutine setup_cons_MPI', 25 )

!     /*   broadcast   */
      call my_mpi_bcast_real_1 ( params_cons, ntype_cons )

!-----------------------------------------------------------------------
!     /*   force constant of harmonic term                            */
!-----------------------------------------------------------------------

      do k = 1, ncons
         fc_cons(k) = params_cons(itype_cons(k))
      end do

!-----------------------------------------------------------------------
!     /*   set boundary condition                                     */
!-----------------------------------------------------------------------

      call get_cv_cons_MPI

      return
      end
