!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up BEST method for solute solvent systems
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_best_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   beta, iounit, natom, nbead, ikind, myrank

      use qmmm_variables, only : &
     &   layer

      use best_variables, only : &
     &   r_best, pot_best, s_best, sx_best, sy_best, sz_best, dv_best, &
     &   pot_best, fx_best, fy_best, fz_best, fux_best, fuy_best, &
     &   fuz_best, vir_best, alpha_best, fc_best, eps_best, s_sort, &
     &   nobest, npbest, nsbest, nbest, ioption_best, iprint_best, &
     &   ikind_best, iobest, ibest, num_sort

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   real numbers   */
      integer :: i, j, ierr, itest

!-----------------------------------------------------------------------
!     /*   read best switch                                           */
!-----------------------------------------------------------------------

!     /*   option of best   */
      call read_int1_MPI ( ioption_best, '<ioption_best>', 14, iounit )

!-----------------------------------------------------------------------
!     /*   if best is off return                                      */
!-----------------------------------------------------------------------

      if ( ioption_best .ne. 1 ) return

!-----------------------------------------------------------------------
!     /*   read best parameters                                       */
!-----------------------------------------------------------------------

!     /*   print interval   */
      call read_int1_MPI ( iprint_best, '<iprint_best>', 13, iounit )

!     /*   solute atom   */
      call read_int1_MPI ( iobest, '<iobest>', 8, iounit )

!     /*   solvent species   */
      call read_int1_MPI ( ikind_best, '<ikind_best>', 12, iounit )

!     /*   force constant   */
      call read_real1_MPI ( fc_best, '<fc_best>', 9, iounit )

!     /*   cut off energy   */
      call read_real1_MPI ( eps_best, '<eps_best>', 10, iounit )

!-----------------------------------------------------------------------
!     /*   convert parameters                                         */
!-----------------------------------------------------------------------

!     /*   force constant to exponent of g function   */
      alpha_best = 0.5d0 * beta * fc_best

!     /*    energy cut off parameter to distance cutoff parameter   */
      r_best = sqrt( abs( - log( beta * eps_best ) / alpha_best ) )

!-----------------------------------------------------------------------
!     /*   set number of solute and solvent atoms                     */
!-----------------------------------------------------------------------

!     /*   number of solute atom   */
      nobest = 1

!     /*   number of solvent atoms in primary subsystem   */
      npbest = 0

!     /*   number of solvent atoms in secondary subsystem   */
      nsbest = 0

!     /*   loop of atoms  */
      do i = 1, natom

!        /*   skip solute atom   */
         if ( i .eq. iobest ) cycle

!        /*   only solvent atoms   */
         if ( ikind(i) .ne. ikind_best ) cycle

!        /*   count solvent atoms in primary region   */
         if ( layer(i)(1:1) .eq. 'A' ) npbest = npbest + 1

!        /*   count solvent atoms in secondary region   */
         if ( layer(i)(1:1) .eq. 'B' ) nsbest = nsbest + 1

!     /*   loop of atoms  */
      end do

!     /*   total number of solute and solvent atoms   */
      nbest  = npbest + nsbest

!-----------------------------------------------------------------------
!     /*   error check                                                */
!-----------------------------------------------------------------------

!     /*   initialize flag   */
      ierr = 0

!     /*   error if solvent atom do not exist in both regions   */
      if ( ( npbest .le. 0 ) .or. ( nsbest .le. 0 ) ) ierr = 1

!     /*   error termination   */
      call error_handling_MPI( ierr, 'subroutine setup_best_MPI', 25 )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   solvent atoms   */
      if ( .not. allocated( ibest ) ) &
     &   allocate( ibest(nbest) )

!     /*   derivative of bias potential   */
      if ( .not. allocated( dv_best ) ) &
     &   allocate( dv_best(nbest) )

!     /*   index s: solute-solvent distance   */
      if ( .not. allocated( s_best ) ) &
     &   allocate( s_best(nbest) )

!     /*   best index in sorted order   */
      if ( .not. allocated( s_sort ) ) &
     &   allocate( s_sort(nbest) )

!     /*   solute atom in sorted order   */
      if ( .not. allocated( num_sort ) ) &
     &   allocate( num_sort(nbest) )

!     /*   x, y, z components of solute-solvent distance   */
      if ( .not. allocated( sx_best ) ) &
     &   allocate( sx_best(nbest) )
      if ( .not. allocated( sy_best ) ) &
     &   allocate( sy_best(nbest) )
      if ( .not. allocated( sz_best ) ) &
     &   allocate( sz_best(nbest) )

!     /*   bias potential   */
      if ( .not. allocated( pot_best ) ) &
     &   allocate( pot_best(nbead) )

!     /*   bias force for bead based best method   */
      if ( .not. allocated( fx_best ) ) &
     &   allocate( fx_best(natom,nbead) )
      if ( .not. allocated( fy_best ) ) &
     &   allocate( fy_best(natom,nbead) )
      if ( .not. allocated( fz_best ) ) &
     &   allocate( fz_best(natom,nbead) )

!     /*   bias force for centroid based best method   */
      if ( .not. allocated( fux_best ) ) &
     &   allocate( fux_best(natom,nbead) )
      if ( .not. allocated( fuy_best ) ) &
     &   allocate( fuy_best(natom,nbead) )
      if ( .not. allocated( fuz_best ) ) &
     &   allocate( fuz_best(natom,nbead) )

!     /*   bias virial   */
      if ( .not. allocated( vir_best ) ) &
     &   allocate( vir_best(3,3) )

!-----------------------------------------------------------------------
!     /*   ibest: solvent atoms in primary subsystem                  */
!-----------------------------------------------------------------------

!     /*   reset counter   */
      j = 0

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   for solvent species   */
         if ( ikind(i) .ne. ikind_best ) cycle

!        /*   in primary region   */
         if ( layer(i)(1:1) .ne. 'A' ) cycle

!        /*   excluding solute atom   */
         if ( i .eq. iobest ) cycle

!        /*  update counter   */
         j = j + 1

!        /*   book solvent atoms in secondary region   */
         ibest(j) = i

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   ibest: solvent atoms in secondary subsystem                */
!-----------------------------------------------------------------------

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   for solvent species   */
         if ( ikind(i) .ne. ikind_best ) cycle

!        /*   in secondary region   */
         if ( layer(i)(1:1) .ne. 'B' ) cycle

!        /*   excluding solute atom   */
         if ( i .eq. iobest ) cycle

!        /*  update counter   */
         j = j + 1

!        /*   book solvent atoms in secondary region   */
         ibest(j) = i

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   print output                                               */
!-----------------------------------------------------------------------

      if ( ( myrank .eq. 0 ) .and. ( iprint_best .gt. 0 ) ) then

         call testfile ( 'best.out', 8, itest )

         if ( itest .eq. 1 ) then

            open ( iounit, file = 'best.out' )

            write( iounit, '(a)') '------------------------------------'
            write( iounit, '(a)') '    step all  qm  mm  bias potential'
            write( iounit, '(a)') '------------------------------------'

            close( iounit )

         end if

      end if

      return
      end
