!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up alchemical systems
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_alchem_MPI
!***********************************************************************
!=======================================================================
!
!     alchemical mixture
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead, iounit, myrank

      use alchem_variables, only : &
     &   fx_a, fy_a, fz_a, fx_b, fy_b, fz_b, vir_a, vir_b, &
     &   ratio_alchem, pot_alchem, fx_alchem, fy_alchem, fz_alchem, &
     &   pot_a, pot_b, alchem_potential_a, alchem_potential_b, &
     &   alchem_dat_dir_a, alchem_dat_dir_b, alchem_scr_dir_a, &
     &   alchem_scr_dir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: i, ierr

!     /*   flag for initial setting   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   read input data                                            */
!-----------------------------------------------------------------------

!     /*   initial setting   */
      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   read potentials                                            */
!-----------------------------------------------------------------------

!     /*   parent process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<alchem_potential>', 18, iounit, ierr )

!     /*   potentials A and B   */
      read( iounit, *, iostat=ierr ) &
     &   alchem_potential_a, alchem_potential_b

!     /*   default   */
      if ( ierr .ne. 0 ) then

!        /*   file close   */
         close( iounit )

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<alchem_potential>', 18, iounit, ierr )

!        /*   potentials A and B   */
         read ( iounit, *, iostat=ierr ) &
     &      alchem_potential_a, alchem_potential_b

!     /*   default   */
      end if

!     /*   file close   */
      close( iounit )

!     /*   parent process only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_alchem_MPI', 27 )

!-----------------------------------------------------------------------
!     /*   read data directories                                      */
!-----------------------------------------------------------------------

!     /*   parent process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<alchem_dat_dir>', 16, iounit, ierr )

!     /*   data directories for A and B   */
      read( iounit, *, iostat=ierr ) &
     &   alchem_dat_dir_a, alchem_dat_dir_b

!     /*   default   */
      if ( ierr .ne. 0 ) then

!        /*   file close   */
         close( iounit )

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<alchem_dat_dir>', 16, iounit, ierr )

!        /*   data directories for A and B   */
         read ( iounit, *, iostat=ierr ) &
     &      alchem_dat_dir_a, alchem_dat_dir_b

!     /*   default   */
      end if

!     /*   file close   */
      close( iounit )

!     /*   parent process only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_alchem_MPI', 27 )

!-----------------------------------------------------------------------
!        /*   read scratch directories                                */
!-----------------------------------------------------------------------

!     /*   parent process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<alchem_scr_dir>', 16, iounit, ierr )

!     /*   scratch directories for A and B    */
      read( iounit, *, iostat=ierr ) &
     &   alchem_scr_dir_a, alchem_scr_dir_b

!     /*   default   */
      if ( ierr .ne. 0 ) then

!        /*   file close   */
         close( iounit )

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<alchem_scr_dir>', 16, iounit, ierr )

!        /*   scratch directories for A and B    */
         read ( iounit, *, iostat=ierr ) &
     &      alchem_scr_dir_a, alchem_scr_dir_b

!     /*   default   */
      end if

!     /*   file close   */
      close( iounit )

!     /*   parent process only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_alchem_MPI', 27 )

!-----------------------------------------------------------------------
!     /*   communicate                                                */
!-----------------------------------------------------------------------

      call my_mpi_bcast_char_0( alchem_potential_a, 80 )
      call my_mpi_bcast_char_0( alchem_potential_b, 80 )

      call my_mpi_bcast_char_0( alchem_dat_dir_a, 80 )
      call my_mpi_bcast_char_0( alchem_dat_dir_b, 80 )

      call my_mpi_bcast_char_0( alchem_scr_dir_a, 80 )
      call my_mpi_bcast_char_0( alchem_scr_dir_b, 80 )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( pot_a ) ) &
     &   allocate( pot_a(nbead) )
      if ( .not. allocated( pot_b ) ) &
     &   allocate( pot_b(nbead) )

      if ( .not. allocated( fx_a ) ) &
     &   allocate( fx_a(natom,nbead) )
      if ( .not. allocated( fy_a ) ) &
     &   allocate( fy_a(natom,nbead) )
      if ( .not. allocated( fz_a ) ) &
     &   allocate( fz_a(natom,nbead) )

      if ( .not. allocated( fx_b ) ) &
     &   allocate( fx_b(natom,nbead) )
      if ( .not. allocated( fy_b ) ) &
     &   allocate( fy_b(natom,nbead) )
      if ( .not. allocated( fz_b ) ) &
     &   allocate( fz_b(natom,nbead) )

      if ( .not. allocated( vir_a ) ) &
     &   allocate( vir_a(3,3) )
      if ( .not. allocated( vir_b ) ) &
     &   allocate( vir_b(3,3) )

      if ( .not. allocated( ratio_alchem ) ) &
     &   allocate( ratio_alchem(2,nbead) )

      if ( .not. allocated( pot_alchem ) ) &
     &   allocate( pot_alchem(nbead,nbead) )

      if ( .not. allocated( fx_alchem ) ) &
     &   allocate( fx_alchem(natom,nbead,nbead) )
      if ( .not. allocated( fy_alchem ) ) &
     &   allocate( fy_alchem(natom,nbead,nbead) )
      if ( .not. allocated( fz_alchem ) ) &
     &   allocate( fz_alchem(natom,nbead,nbead) )

!-----------------------------------------------------------------------
!     /*   ratio of alchemical mixture                                */
!-----------------------------------------------------------------------

      if ( nbead .eq. 1 ) then

         ratio_alchem(1,1) = 1.d0
         ratio_alchem(2,1) = 0.d0

      else

         do i = 1, nbead
            ratio_alchem(1,i) = 1.d0 - dble(i-1)/dble(nbead-1)
            ratio_alchem(2,i) = dble(i-1)/dble(nbead-1)
         end do

      end if

!-----------------------------------------------------------------------
!     /*   make input.dat and mm.dat in subdirectories                */
!-----------------------------------------------------------------------

      call setup_alchem_a_MPI
      call setup_alchem_b_MPI

!-----------------------------------------------------------------------
!     /*   reset flag                                                 */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine setup_alchem_a_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   amu_mass, au_mass, box, iounit, iboundary, nbead, natom, &
     &   species, myrank, nprocs

      use alchem_variables, only : &
     &   alchem_potential_a, alchem_scr_dir_a, alchem_dat_dir_a

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: i, j

!     /*   characters   */
      character(len=80) :: char_file
      character(len=3)  :: char_num

!-----------------------------------------------------------------------
!     /*   make directory                                             */
!-----------------------------------------------------------------------

      call system ( 'mkdir -p ' // trim(alchem_scr_dir_a) )

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do j = 1, nbead

!-----------------------------------------------------------------------
!        /*   make directory                                          */
!-----------------------------------------------------------------------

         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

         call int3_to_char( j, char_num )

         call system('mkdir -p ' // &
     &               trim(alchem_scr_dir_a) // '/' // char_num )

!-----------------------------------------------------------------------
!        /*   copy files                                              */
!-----------------------------------------------------------------------

!        /*   copy input_default.dat   */
         call system( 'cp -f input_default.dat ' // &
     &                trim(alchem_scr_dir_a) // '/' // char_num )

!c        /*   copy structure.dat   */
!         call system( 'cp -f structure.dat ' //
!     &                trim(alchem_scr_dir_a) // '/' // char_num )

!        /*   copy *.dat   */
         call system ('cp -rf ' // trim(alchem_dat_dir_a) // '/* ' // &
     &                trim(alchem_scr_dir_a) // '/' // char_num )

!-----------------------------------------------------------------------
!        /*   make structure.dat                                      */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_a) // '/' // char_num // &
     &               '/structure.dat'

         open ( iounit, file = char_file )

         write( iounit, '(i8)' ) natom
         write( iounit, '(a)' ) 'BOHR'

         do i = 1, natom
            write( iounit, '(a,3e24.16,i2)' ) &
     &      species(i), 0.d0, 0.d0, 0.d0, 1
         end do

         close( iounit )

!-----------------------------------------------------------------------
!        /*   make input.dat                                          */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_a) // '/' // char_num // &
     &               '/input.dat'

         open ( iounit, file = char_file )

         write(iounit,'(a)')   '<input_style>'
         write(iounit,'(a)')   'NEW'
         write(iounit,*)

         write(iounit,'(a)')   '<method>'
         write(iounit,'(a)')   'STATIC'
         write(iounit,*)

         write(iounit,'(a)')   '<natom>'
         write(iounit,'(i8)')  natom
         write(iounit,*)

         write(iounit,'(a)')   '<ipotential>'
         write(iounit,'(a)')   alchem_potential_a
         write(iounit,*)

         write(iounit,'(a)')   '<iboundary>'
         write(iounit,'(i8)')  iboundary
         if ( iboundary .ne. 1 ) then
            write(iounit,'(3e24.16)') box(1,1), box(1,2), box(1,3)
            write(iounit,'(3e24.16)') box(2,1), box(2,2), box(2,3)
            write(iounit,'(3e24.16)') box(3,1), box(3,2), box(3,3)
         end if
         write(iounit,*)

         close(iounit)

      end do

      return
      end





!***********************************************************************
      subroutine setup_alchem_b_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   amu_mass, au_mass, box, iounit, iboundary, nbead, natom, &
     &   species, myrank, nprocs

      use alchem_variables, only : &
     &   alchem_potential_b, alchem_scr_dir_b, alchem_dat_dir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: i, j

!     /*   characters   */
      character(len=80) :: char_file
      character(len=3)  :: char_num

!-----------------------------------------------------------------------
!     /*   make directory                                             */
!-----------------------------------------------------------------------

      call system ( 'mkdir -p ' // trim(alchem_scr_dir_b) )

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do j = 1, nbead

!-----------------------------------------------------------------------
!        /*   make directory                                          */
!-----------------------------------------------------------------------

         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

         call int3_to_char( j, char_num )

         call system('mkdir -p ' // &
     &               trim(alchem_scr_dir_b) // '/' // char_num )

!-----------------------------------------------------------------------
!        /*   copy files                                              */
!-----------------------------------------------------------------------

!        /*   copy input_default.dat   */
         call system( 'cp -f input_default.dat ' // &
     &                trim(alchem_scr_dir_b) // '/' // char_num )

!c        /*   copy structure.dat   */
!         call system( 'cp -f structure.dat ' //
!     &                trim(alchem_scr_dir_b) // '/' // char_num )

!        /*   copy *.dat   */
         call system ('cp -rf ' // trim(alchem_dat_dir_b) // '/* ' // &
     &                trim(alchem_scr_dir_b) // '/' // char_num )

!-----------------------------------------------------------------------
!        /*   make structure.dat                                      */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_b) // '/' // char_num // &
     &               '/structure.dat'

         open ( iounit, file = char_file )

         write( iounit, '(i8)' ) natom
         write( iounit, '(a)' ) 'BOHR'

         do i = 1, natom
            write( iounit, '(a,3e24.16,i2)' ) &
     &      species(i), 0.d0, 0.d0, 0.d0, 1
         end do

         close( iounit )

!-----------------------------------------------------------------------
!        /*   make input.dat                                          */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_b) // '/' // char_num // &
     &               '/input.dat'

         open ( iounit, file = char_file )

         write(iounit,'(a)')   '<input_style>'
         write(iounit,'(a)')   'NEW'
         write(iounit,*)

         write(iounit,'(a)')   '<method>'
         write(iounit,'(a)')   'STATIC'
         write(iounit,*)

         write(iounit,'(a)')   '<natom>'
         write(iounit,'(i8)')  natom
         write(iounit,*)

         write(iounit,'(a)')   '<ipotential>'
         write(iounit,'(a)')   alchem_potential_b
         write(iounit,*)

         write(iounit,'(a)')   '<iboundary>'
         write(iounit,'(i8)')  iboundary
         if ( iboundary .ne. 1 ) then
            write(iounit,'(3e24.16)') box(1,1), box(1,2), box(1,3)
            write(iounit,'(3e24.16)') box(2,1), box(2,2), box(2,3)
            write(iounit,'(3e24.16)') box(3,1), box(3,2), box(3,3)
         end if
         write(iounit,*)

         close(iounit)

      end do

      return
      end
