!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jun 25, 2021 by M. Shiga
!      Description:     set up ring polymer molecular dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setpiparams_MPI
!***********************************************************************
!=======================================================================
!
!     set up path integral parameters
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, beta, gamma2, dt, dt_ref, omega_p, omega_p2, amu_mass, &
     &   gamma2_cent, gamma_p2, amu_mass_earth, time_mode, au_time, &
     &   gamma1, gamma2, gamma_cent, nbead, nref, igamma, iounit, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: dp, omega_mode, time_mode_default
      integer :: ierr
 
!-----------------------------------------------------------------------
!     /*   time step for harmonic forces                              */
!-----------------------------------------------------------------------

      dt_ref = dt / dble(nref)

!-----------------------------------------------------------------------
!     /*   parameters for path integral simulation                    */
!-----------------------------------------------------------------------

!     /*   number of beads   */
      dp = dble(nbead)

!     /*   spring constant of harmonic forces   */
      omega_p = sqrt(dp) / beta
      omega_p2 = omega_p * omega_p

!-----------------------------------------------------------------------
!     /*   gamma1 = adiabaticity parameter for centroid MD            */
!-----------------------------------------------------------------------

!     /*   read from file   */
      call read_int1_MPI ( igamma, '<igamma>', 8, iounit )

!     /*   define gamma2   */

      if ( igamma .ge. 1 ) then
         gamma1      = 1.d0 / dble(igamma)
         gamma_cent  = 1.d0
      else if ( igamma .eq. 0 ) then
         gamma1      = 1.d0
         gamma_cent  = amu_mass_earth
      else
         gamma1      = 1.d0
         gamma_cent  = 1.d0
      end if

!     /*   define gamma2   */

      gamma2 = gamma1 * gamma1
      gamma2_cent = gamma_cent * gamma_cent

!-----------------------------------------------------------------------
!     /*    read time_mode                                            */
!-----------------------------------------------------------------------

!cc      call read_real1_MPI ( time_mode, '<time_mode>', 11, iounit )

      time_mode_default = ( 2.d0 * pi / omega_p ) * ( au_time / 1.d-15 )

      if ( myrank .eq. 0 ) then

         open ( iounit, file = 'input.dat' )

         call search_tag ( '<time_mode>', 11, iounit, ierr )

         if ( ierr .eq. 0 ) then

            read ( iounit, *, iostat=ierr ) time_mode

            if ( ierr .ne. 0 ) then
               time_mode = time_mode_default
            end if

            close( iounit )

         else

            close( iounit )

            open ( iounit, file = 'input_default.dat' )

            call search_tag ( '<time_mode>', 11, iounit, ierr )

            if ( ierr .eq. 0 ) then

               read ( iounit, *, iostat=ierr ) time_mode

               if ( ierr .ne. 0 ) then
                  time_mode = time_mode_default
               end if

            else

               time_mode = time_mode_default

            end if

            close( iounit )

         end if

      end if

      call my_mpi_bcast_real_0( time_mode )

      if ( time_mode .le. 0.d0 ) time_mode = time_mode_default

!-----------------------------------------------------------------------
!     /*   new setting of non-centroid masses                         */
!-----------------------------------------------------------------------

      time_mode  = time_mode * ( 1.d-15 / au_time )
      omega_mode = 2.d0 * pi / time_mode

      if ( time_mode .le. 0.d0 ) then
         gamma_p2 = 1.d0
      else
         gamma_p2 = omega_p2 / ( omega_mode * omega_mode )
      end if

      return
      end





!***********************************************************************
      subroutine setmdparams_MPI
!***********************************************************************
!=======================================================================
!
!     set up MD parameters
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, nbead, tnm, tnminv

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   unit matrix (just for convenience)                         */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, nbead
         tnm(i,j)    = 0.d0
         tnminv(i,j) = 0.d0
      end do
      end do

      do i = 1, nbead
         tnm(i,i)    = 1.d0
         tnminv(i,i) = 1.d0
      end do

      return
      end





!***********************************************************************
      subroutine setatomchanges_MPI
!***********************************************************************
!=======================================================================
!
!     set up MD parameters
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, atom_change, natom, myrank_world

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, nline, ierr

      character(len=9) :: char

!-----------------------------------------------------------------------
!     /*    read mass scaling parameters                              */
!-----------------------------------------------------------------------

      if ( myrank_world .eq. 0 ) then

         atom_change(1:natom) = 'NONE'

         open ( iounit, file = 'input.dat' )

         call search_tag ( '<atom_change>', 13, iounit, ierr )

         read ( iounit, *, iostat=ierr ) char

         read ( char, *, iostat=ierr ) nline

         if ( ierr .eq. 0 ) then

            do k = 1, nline

               read ( iounit, *, iostat=ierr ) char, i, j

               if ( ierr .ne. 0 ) then

                  backspace( iounit )

                  read ( iounit, *, iostat=ierr ) char, i

                  j = i

               end if

               if ( ( i .le. 0 ) .or. ( i .gt. natom ) ) i = 1
               if ( ( j .le. 0 ) .or. ( j .gt. natom ) ) j = natom

               atom_change(i:j) = char

            end do

         else

            close( iounit )

            open ( iounit, file = 'input_default.dat' )

            call search_tag ( '<atom_change>', 13, iounit, ierr )

            read ( iounit, *, iostat=ierr ) char

         end if

         close( iounit )

      end if

      call my_mpi_bcast_int_0_world( ierr )

      call error_handling_MPI &
     &   ( ierr, 'subroutine setatomchanges_MPI', 29 )

      call my_mpi_bcast_char_1_world &
     &   ( atom_change, len(atom_change), natom )

      return
      end
