!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up simulation parameters
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setparams_MPI
!***********************************************************************
!=======================================================================
!
!     set up parameters
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   method, natom, nkind, nbead, nref, nys, bath_type, ncolor, &
     &   nnhc, ipotential, iounit, ensemble, input_style, mspec, &
     &   qmmm_potential, qmmm_embedding, oniom_hi_potential, &
     &   oniom_lo_potential, species, ikind, myrank_world, xmpi

      implicit none

      integer :: itest

!-----------------------------------------------------------------------
!     test if input file exist
!-----------------------------------------------------------------------

      itest = 0

      if ( myrank_world .eq. 0 ) then
         call testfile ( 'input.dat', 9, itest )
      end if

      call my_mpi_bcast_int_0_world( itest )

      if ( itest .eq. 1 ) then
         if ( myrank_world .eq. 0 ) then
            write ( 6, '(a)' ) 'Error - file missing: input.dat.'
            write ( 6, '(a)' )
         end if
         call error_handling_MPI( 1, 'subroutine setparams_MPI', 24 )
      end if

      if ( myrank_world .eq. 0 ) then
         call testfile ( 'input_default.dat', 17, itest )
      end if

      call my_mpi_bcast_int_0_world( itest )

      if ( itest .eq. 1 ) then
         if ( myrank_world .eq. 0 ) then
            write( 6, '(a)' ) 'Error - file missing: input_default.dat.'
            write( 6, '(a)' )
         end if
         call error_handling_MPI( 1, 'subroutine setparams_MPI', 24 )
      end if

      call my_mpi_barrier_world

!-----------------------------------------------------------------------
!     molecular dynamics
!-----------------------------------------------------------------------

!     /*   method   */
      call read_char_MPI ( method, 80, '<method>', 8, iounit )

!-----------------------------------------------------------------------
!     ensemble
!-----------------------------------------------------------------------

!     /*   method   */
      call read_char_MPI ( ensemble, 10, '<ensemble>', 10, iounit )

!-----------------------------------------------------------------------
!     system
!-----------------------------------------------------------------------

!     /*   option of initial geometry file   */
      call read_char_MPI &
     &   ( input_style, 10, '<input_style>', 13, iounit )

      if      ( input_style(1:4) .eq. 'OLD ' ) then

!        /*   number of atoms   */
         call read_int1_MPI ( natom, '<natom>', 7, iounit )

!        /*   number of species   */
         call read_int1_MPI ( nkind, '<nspec>', 7, iounit )

!        /*   number of species   */
         mspec = nkind

!        /*   atomic species   */
         if ( .not. allocated( species ) ) &
     &      allocate( species(natom) )

!        /*   species number   */
         if ( .not. allocated( ikind ) ) &
     &      allocate( ikind(natom) )

      else if ( input_style(1:4) .eq. 'NEW ' ) then

         if ( myrank_world .eq. 0 ) then
            call testfile ( 'structure.dat', 13, itest )
         end if

         if ( itest .eq. 1 ) then
            if ( myrank_world .eq. 0 ) then
               write( 6, '(a)' ) 'Error - file missing: structure.dat.'
               write( 6, '(a)' )
            end if
            call error_handling_MPI( 1, 'subroutine setparams_MPI', 24 )
         end if

!        /*   read atoms from structure.dat   */
         call read_init_MPI

      else

         if ( myrank_world .eq. 0 ) then

            write ( 6, '(a)' ) &
     &         'Error - input_style should be either OLD or NEW.'
            write ( 6, '(a)' )

         end if

         call error_handling_MPI( 1, 'subroutine setparams_MPI', 24 )

      end if

!     /*   number of beads:  only even number is valid here.   */
      call read_int1_MPI ( nbead, '<nbead>', 7, iounit )

!-----------------------------------------------------------------------
!     time step
!-----------------------------------------------------------------------

!     /*   time step decompsition for reference harmonic forces   */
      call read_int1_MPI ( nref, '<nref>', 6, iounit )

!     /*   Suzuki-Yoshida decomposition for thermostat   */
      call read_int1_MPI ( nys, '<nys>', 5, iounit )

!-----------------------------------------------------------------------
!     thermostat
!-----------------------------------------------------------------------

!     /*   type of thermostat   */
      call read_char_MPI ( bath_type, 10, '<bath_type>', 11, iounit )

!     /*   number of "colors" of the Nose-Hoover chains   */
      call read_int1_MPI ( ncolor, '<ncolor>', 8, iounit )

!     /*   length of Nose-Hoover chain   */
      call read_int1_MPI ( nnhc, '<nnhc>', 6, iounit )

!-----------------------------------------------------------------------
!     potential
!-----------------------------------------------------------------------

!     /*   type of potential model   */
      call read_char_MPI ( ipotential, 80, '<ipotential>', 12, iounit )

!-----------------------------------------------------------------------
!     QM/MM potential
!-----------------------------------------------------------------------

      call read_char_MPI &
     &   ( qmmm_potential, 80, '<qmmm_potential>', 16, iounit )

      call read_char_MPI &
     &   ( qmmm_embedding, 80, '<qmmm_embedding>', 16, iounit )

      if  ( ( qmmm_embedding(1:3) .ne. 'EE ' ) .and. &
     &      ( qmmm_embedding(1:3) .ne. 'ME ' ) ) then

         if ( myrank_world .eq. 0 ) then

            write ( 6, '(a)' ) &
     &         'Error - QM/MM embedding should be EE or ME.'
            write ( 6, '(a)' )

         end if

         call error_handling_MPI( 1, 'subroutine setparams_MPI', 24 )

      end if

!-----------------------------------------------------------------------
!     ONIOM potential
!-----------------------------------------------------------------------

      call read_char_MPI &
     &   ( oniom_hi_potential, 80, '<oniom_hi_potential>', 20, iounit )

      call read_char_MPI &
     &   ( oniom_lo_potential, 80, '<oniom_lo_potential>', 20, iounit )

!-----------------------------------------------------------------------
!     extensive message passing algorithm
!-----------------------------------------------------------------------

      call read_char_MPI ( xmpi, 5, '<xmpi>', 6, iounit )

      return
      end





!***********************************************************************
      subroutine read_init_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nkind, iounit, mspec, species, ikind, myrank_world

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, i, j, iflag

      real(8) :: x, y, z

      character(len=120) :: charline

!-----------------------------------------------------------------------
!     /*   read number of atoms                                       */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'structure.dat' )

!     /*   number of atoms   */
      read ( iounit, *, iostat=ierr ) natom

!     /*   file close   */
      close( iounit )

!     /*   only master process   */
      end if

!     /*   share with all other processes   */
      call my_mpi_bcast_int_0_world( ierr )

!-----------------------------------------------------------------------
!     /*   stop on error                                              */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!     /*   if error found   */
      if ( ierr .ne. 0 ) then

!        /*   print comments   */
         write( 6, '(a)' ) &
     &      'Error - structure.dat read incorrectly.'
         write( 6, '(a)' )

!     /*   if error found   */
      end if

!     /*   only master process   */
      end if

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine read_init_MPI', 24 )

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

!     /*   share with all other processes   */
      call my_mpi_bcast_int_0_world( natom )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   atomic species   */
      if ( .not. allocated( species ) ) &
     &   allocate( species(natom) )

!     /*   numbering of species   */
      if ( .not. allocated( ikind ) ) &
     &   allocate( ikind(natom) )

!-----------------------------------------------------------------------
!     /*   read atomic species                                        */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'structure.dat' )

!     /*   skip two lines   */
      read ( iounit, *, iostat=ierr )
      read ( iounit, *, iostat=ierr )

!     /*   read species   */

      do i = 1, natom
         read ( iounit, '(a)', iostat=ierr ) charline
         read ( charline, *, iostat=ierr ) species(i), x, y, z, ikind(i)
         if ( ierr .ne. 0 ) then
            read ( charline, *, iostat=ierr ) species(i), x, y, z
            if ( ierr .ne. 0 ) exit
            ikind(i) = 1
         end if
      end do

!     /*   file close   */
      close( iounit )

!     /*   only master process   */
      end if

!     /*   share with all other processes   */
      call my_mpi_bcast_int_0_world( ierr )

!-----------------------------------------------------------------------
!     /*   stop on error                                              */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank_world .eq. 0 ) then

!     /*   if error found   */
      if ( ierr .ne. 0 ) then

!        /*   print comments   */
         write( 6, '(a)' ) &
     &      'Error - structure.dat read incorrectly.'
         write( 6, '(a)' )

!     /*   if error found   */
      end if

!     /*   only master process   */
      end if

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine read_init_MPI', 24 )

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

!     /*   share with all other processes   */
      call my_mpi_bcast_char_1_world( species, len(species), natom )

!     /*   share with all other processes   */
      call my_mpi_bcast_int_1_world( ikind, natom )

!-----------------------------------------------------------------------
!     /*   nkind: total number of kinds                               */
!-----------------------------------------------------------------------

      nkind = 0

      do i = 1, natom
         nkind = max( nkind, ikind(i) )
      end do

!-----------------------------------------------------------------------
!     /*   mspec: total number of species                             */
!-----------------------------------------------------------------------

      mspec = 1

      do i = 2, natom

         iflag = 1

         do j = 1, i-1

            if ( species(j)(1:8) .eq. species(i)(1:8) ) then

               iflag = 0

               exit

            end if

         end do

         mspec = mspec + iflag

      end do

      return
      end

