!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up simulation parameters
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setparams
!***********************************************************************
!=======================================================================
!
!     set up parameters
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   method, natom, nkind, nbead, nref, nys, bath_type, ncolor, &
     &   nnhc, ipotential, iounit, ensemble, input_style, mspec, &
     &   qmmm_potential, qmmm_embedding, oniom_hi_potential, &
     &   oniom_lo_potential, species, ikind

      implicit none

      integer :: itest

!-----------------------------------------------------------------------
!     test if input file exist
!-----------------------------------------------------------------------

      call testfile ( 'input.dat', 9, itest )

      if ( itest .eq. 1 ) then
         write ( 6, '(a)' ) 'Error - file missing: input.dat.'
         write ( 6, '(a)' ) 
         call error_handling( 1, 'subroutine setparams', 20 )
      end if

      call testfile ( 'input_default.dat', 17, itest )

      if ( itest .eq. 1 ) then
         write ( 6, '(a)' ) 'Error - file missing: input_default.dat.'
         write ( 6, '(a)' ) 
         call error_handling( 1, 'subroutine setparams', 20 )
      end if

!-----------------------------------------------------------------------
!     method
!-----------------------------------------------------------------------

!     /*   method   */
      call read_char ( method, 80, '<method>', 8, iounit )

!-----------------------------------------------------------------------
!     ensemble
!-----------------------------------------------------------------------

!     /*   method   */
      call read_char ( ensemble, 10, '<ensemble>', 10, iounit )

!-----------------------------------------------------------------------
!     system
!-----------------------------------------------------------------------

!     /*   option of initial geometry file   */
      call read_char &
     &   ( input_style, 10, '<input_style>', 13, iounit )

      if      ( input_style(1:4) .eq. 'OLD ' ) then

!        /*   number of atoms   */
         call read_int1 ( natom, '<natom>', 7, iounit )

!        /*   number of species   */
         call read_int1 ( nkind, '<nspec>', 7, iounit )

!        /*   number of species   */
         mspec = nkind

!        /*   atomic species   */
         if ( .not. allocated( species ) ) &
     &      allocate( species(natom) )

!        /*   numbering of species   */
         if ( .not. allocated( ikind ) ) &
     &      allocate( ikind(natom) )

      else if ( input_style(1:4) .eq. 'NEW ' ) then

         call testfile ( 'structure.dat', 13, itest )

         if ( itest .eq. 1 ) then
            write( 6, '(a)' ) 'Error - file missing: structure.dat.'
            write( 6, '(a)' )
            call error_handling( 1, 'subroutine setparams', 20 )
         end if

!        /*   read atoms from structure.dat   */
         call read_init

      else

         write ( 6, '(a)' ) &
     &      'Error - input_style should be either OLD or NEW.'
         write ( 6, '(a)' )

         call error_handling( 1, 'subroutine setparams', 20 )

      end if

!     /*   number of beads:  only even number is valid here.   */
      call read_int1 ( nbead, '<nbead>', 7, iounit )

!-----------------------------------------------------------------------
!     time step
!-----------------------------------------------------------------------

!     /*   time step decompsition for reference harmonic forces   */
      call read_int1 ( nref, '<nref>', 6, iounit )

!     /*   Suzuki-Yoshida decomposition for thermostat   */
      call read_int1 ( nys, '<nys>', 5, iounit )

!-----------------------------------------------------------------------
!     thermostat
!-----------------------------------------------------------------------

!     /*   type of thermostat   */
      call read_char ( bath_type, 10, '<bath_type>', 11, iounit )

!     /*   number of "colors" of the Nose-Hoover chains   */
      call read_int1 ( ncolor, '<ncolor>', 8, iounit )

!     /*   length of Nose-Hoover chain   */
      call read_int1 ( nnhc, '<nnhc>', 6, iounit )

!-----------------------------------------------------------------------
!     potential
!-----------------------------------------------------------------------

!     /*   type of potential model   */
      call read_char ( ipotential, 80, '<ipotential>', 12, iounit )

!-----------------------------------------------------------------------
!     QM/MM potential
!-----------------------------------------------------------------------

      call read_char &
     &   ( qmmm_potential, 80, '<qmmm_potential>', 16, iounit )

      call read_char &
     &   ( qmmm_embedding, 80, '<qmmm_embedding>', 16, iounit )

      if  ( ( qmmm_embedding(1:3) .ne. 'EE ' ) .and. &
     &      ( qmmm_embedding(1:3) .ne. 'ME ' ) ) then

            write ( 6, '(a)' ) &
     &         'Error - QM/MM embedding should be EE or ME.'
            write ( 6, '(a)' )

         call error_handling( 1, 'subroutine setparams', 20 )

      end if

!-----------------------------------------------------------------------
!     ONIOM potential
!-----------------------------------------------------------------------

      call read_char &
     &   ( oniom_hi_potential, 80, '<oniom_hi_potential>', 20, iounit )

      call read_char &
     &   ( oniom_lo_potential, 80, '<oniom_lo_potential>', 20, iounit )

!-----------------------------------------------------------------------
!     extensive message passing algorithm
!-----------------------------------------------------------------------

!     call read_char ( xmpi, 3, '<xmpi>', 6, iounit )

      return
      end





!***********************************************************************
      subroutine read_init
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nkind, iounit, species, ikind, mspec

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, i, j, iflag

      real(8) :: x, y, z

      character(len=120) :: charline

!-----------------------------------------------------------------------
!     /*   read number of atoms                                       */
!-----------------------------------------------------------------------

!     /*   file open   */
      open ( iounit, file = 'structure.dat' )

!     /*   number of atoms   */
      read ( iounit, *, iostat=ierr ) natom

!     /*   file close   */
      close( iounit )

!-----------------------------------------------------------------------
!     /*   stop on error                                              */
!-----------------------------------------------------------------------

!     /*   if error found   */
      if ( ierr .ne. 0 ) then

!        /*   print comments   */
         write( 6, '(a)' ) &
     &      'Error - structure.dat read incorrectly (at 1).'
         write( 6, '(a)' )

!     /*   if error found   */
      end if

!     /*   stop on error   */
      call error_handling( ierr, 'subroutine read_init', 20 )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   atomic species   */
      if ( .not. allocated( species ) ) &
     &   allocate( species(natom) )

!     /*   species number   */
      if ( .not. allocated( ikind ) ) &
     &   allocate( ikind(natom) )

!-----------------------------------------------------------------------
!     /*   read atomic species                                        */
!-----------------------------------------------------------------------

!     /*   file open   */
      open ( iounit, file = 'structure.dat' )

!     /*   skip two lines   */
      read ( iounit, *, iostat=ierr )
      read ( iounit, *, iostat=ierr )

!     /*   read species   */

      do i = 1, natom
         read ( iounit, '(a)', iostat=ierr ) charline
         read ( charline, *, iostat=ierr ) species(i), x, y, z, ikind(i)
         if ( ierr .ne. 0 ) then
            read ( charline, *, iostat=ierr ) species(i), x, y, z
            if ( ierr .ne. 0 ) exit
            ikind(i) = 1
         end if
      end do

!     /*   file close   */
      close( iounit )

!-----------------------------------------------------------------------
!     /*   stop on error                                              */
!-----------------------------------------------------------------------

!     /*   if error found   */
      if ( ierr .ne. 0 ) then

!        /*   print comments   */
         write( 6, '(a)' ) &
     &      'Error - structure.dat read incorrectly (at 2).'
         write( 6, '(a)' )

!     /*   if error found   */
      end if

!     /*   stop on error   */
      call error_handling( ierr, 'subroutine read_init', 20 )

!-----------------------------------------------------------------------
!     /*   nkind: total number of kinds                               */
!-----------------------------------------------------------------------

      nkind = 0

      do i = 1, natom
         nkind = max( nkind, ikind(i) )
      end do

!-----------------------------------------------------------------------
!     /*   mspec: total number of species                             */
!-----------------------------------------------------------------------

      mspec = 1

      do i = 2, natom

         iflag = 1

         do j = 1, i-1

            if ( species(j)(1:8) .eq. species(i)(1:8) ) then

               iflag = 0

               exit

            end if

         end do

         mspec = mspec + iflag

      end do

      return
      end

