!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up simulation conditions
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setcondition_MPI
!***********************************************************************
!=======================================================================
!
!     set up conditions
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   dt, temperature, boltz, beta, pi, time_bath, omega_bath, &
     &   beadspread, au_time, pressure, au_energy, au_length, &
     &   beta_beadspread, time_baro, omega_baro, pimd_command, &
     &   nstep, iprint_rest, iounit, iread_exit, iboundary, irem_type, &
     &   iorder_hmc, ends_string, nph_type, nth_type, npt_type, &
     &   ntt_type, equation_om, mbox, afed_type, tamd_type, logmfd_type

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   read starting flags                                        */
!-----------------------------------------------------------------------

      call read_start_MPI

!-----------------------------------------------------------------------
!     /*   nstep = number of MD steps                                 */
!-----------------------------------------------------------------------

      call read_int1_MPI ( nstep, '<nstep>', 7, iounit )

!-----------------------------------------------------------------------
!     /*   dt = time step size                                        */
!-----------------------------------------------------------------------

      call read_real1_MPI ( dt, '<dt>', 4, iounit )

!     /*   [fs] -> [a.u.]   */
      dt = dt * (1.d-15/au_time)

!-----------------------------------------------------------------------
!     /*   temperature [K]                                            */
!-----------------------------------------------------------------------

      call read_real1_MPI ( temperature, '<temperature>', 13, iounit )

!     /*   inverse of temperature   */
      beta = 1.d0 / (boltz*temperature)

!-----------------------------------------------------------------------
!     /*   species                                                    */
!-----------------------------------------------------------------------

      call read_species_MPI

!-----------------------------------------------------------------------
!     /*   output step intervals                                      */
!-----------------------------------------------------------------------

!     /*   restart file   */
      call read_int1_MPI ( iprint_rest, '<iprint_rest>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   step intervals for reading exit file                       */
!-----------------------------------------------------------------------

!     /*   exit file   */
      call read_int1_MPI ( iread_exit, '<iread_exit>', 12, iounit )

!-----------------------------------------------------------------------
!     /*   set degrees of freedom                                     */
!-----------------------------------------------------------------------

      call set_dof

!-----------------------------------------------------------------------
!     /*   boundary condition                                         */
!-----------------------------------------------------------------------

      call boundary_MPI

!-----------------------------------------------------------------------
!     /*   pressure [MPa]                                             */
!-----------------------------------------------------------------------

      call read_real1_MPI ( pressure, '<pressure>', 10, iounit )

!     /*   [mpa] -> [a.u.]   */
      pressure = pressure / (au_energy/au_length**3/10**6)

!-----------------------------------------------------------------------
!     /*   tension                                                    */
!-----------------------------------------------------------------------

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
         call read_tension_MPI
      end if

!-----------------------------------------------------------------------
!     /*   characteristic frequency of the bath                       */
!-----------------------------------------------------------------------

      call read_real1_MPI ( time_bath, '<time_bath>', 11, iounit )

!     /*   [fs] -> [a.u.]   */
      time_bath  = time_bath * (1.d-15/au_time)

!     /*   angular frequency   */
      omega_bath = 2.d0*pi/time_bath

!-----------------------------------------------------------------------
!     /*   characteristic frequency of the barostat                   */
!-----------------------------------------------------------------------

      call read_real1_MPI ( time_baro, '<time_baro>', 11, iounit )

!     /*   [fs] -> [a.u.]   */
      time_baro  = time_baro * (1.d-15/au_time)

!     /*   angular frequency   */
      omega_baro = 2.d0*pi/time_baro

!-----------------------------------------------------------------------
!     /*   temperature for bead spread [K] (valid for initial run)    */
!-----------------------------------------------------------------------

      call read_real1_MPI ( beadspread, '<beadspread>', 12, iounit )

!     /*   inverse of beadspread temperature   */
      beta_beadspread = 1.d0/(boltz*beadspread)

!-----------------------------------------------------------------------
!     /*   type of box fluctuation                                    */
!-----------------------------------------------------------------------

      call read_char_MPI ( nph_type, 8, '<nph_type>', 10, iounit )
      call read_char_MPI ( nth_type, 8, '<nth_type>', 10, iounit )
      call read_char_MPI ( npt_type, 8, '<npt_type>', 10, iounit )
      call read_char_MPI ( ntt_type, 8, '<ntt_type>', 10, iounit )

!-----------------------------------------------------------------------
!     /*   order of Suzuki Trotter expansion                          */
!-----------------------------------------------------------------------

      call read_int1_MPI ( iorder_hmc, '<iorder_hmc>', 12, iounit )

!-----------------------------------------------------------------------
!     /*   irem_type:   type of replica exchange method               */
!-----------------------------------------------------------------------

      call read_char_MPI( irem_type, 8, '<irem_type>', 11, iounit )

!-----------------------------------------------------------------------
!     /*   ends_string:  ends of string method                        */
!-----------------------------------------------------------------------

      call read_char_MPI( ends_string, 8, '<ends_string>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   PIMD execution command (pimd.x)                            */
!-----------------------------------------------------------------------

      call read_char_MPI &
     &   ( pimd_command, 80, '<pimd_command>', 14, iounit )

!-----------------------------------------------------------------------
!     /*   overdamped or underdamped                                  */
!-----------------------------------------------------------------------

      call read_char_MPI( equation_om, 12, '<equation_om>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   type of adiabatic free energy dynamics                     */
!-----------------------------------------------------------------------

      call read_char_MPI ( afed_type, 12, '<afed_type>', 11, iounit )

!-----------------------------------------------------------------------
!     /*   type of temperature accelerated molecular dynamics         */
!-----------------------------------------------------------------------

      call read_char_MPI ( tamd_type, 8, '<tamd_type>', 11, iounit )

!-----------------------------------------------------------------------
!     /*   type of logarithmic mean force dynamics                    */
!-----------------------------------------------------------------------

      call read_char_MPI ( logmfd_type, 8, '<logmfd_type>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   initialize mbox                                            */
!-----------------------------------------------------------------------

      mbox(:,:,:) = 0

!-----------------------------------------------------------------------
!     /*   atomic mass change                                         */
!-----------------------------------------------------------------------

      call setatomchanges_MPI

      return
      end

