!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, B. Thomsen
!      Last updated:    Jul 30, 2019 by M. Shiga
!      Description:     consecutive static calculations
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine scancycle_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, dipx, dipy, dipz, fx, fy, fz, au_length, &
     &   iounit_scn, istep_start, istep, nstep, nbead, natom, iounit, &
     &   istep, iexit, scan_exe_shell, species, mbox, myrank, nprocs, &
     &   iprint_minfo

      use analysis_variables, only : &
     &   iprint_xsf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: k, l, ierr, m1, m2, m3

!     /*   real numbers   */
      real(8) :: scale

!     /*   characters   */
      character(len=10) :: char

!     /*   characters   */
      character(len=3)  :: char_num

!     /*   conversion from bohr to angstrom   */
      real(8), parameter :: bohr2ang = au_length * 1.d+10

!     /*   real numbers   */
      real(8) :: xa, ya, za, xb, yb, zb

!-----------------------------------------------------------------------
!     /*   execution commmand for postprocess in scan                 */
!-----------------------------------------------------------------------

      call read_char_MPI &
     &   ( scan_exe_shell, 80, '<scan_exe_shell>', 16, iounit )

!-----------------------------------------------------------------------
!     /*   check if we should print minfo and xsf files               */
!-----------------------------------------------------------------------

      call read_int1_MPI ( iprint_minfo, '<iprint_minfo>', 14, iounit )

      call read_int1_MPI ( iprint_xsf,  '<iprint_xsf>',  12, iounit )

!-----------------------------------------------------------------------
!     /*   unit of atomic coordinates                                 */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         open ( iounit_scn, file = 'structure.dat' )

         read( iounit_scn, *, iostat=ierr )
         read( iounit_scn, *, iostat=ierr ) char

         close ( iounit_scn )

         scale = 1.d0

         if ( char(1:5) .eq. 'BOHR ' ) then
            scale = 1.d0
         else if ( char(1:9) .eq. 'ANGSTROM ' ) then
            scale = 1.d0 / bohr2ang
         else
            ierr = 1
         end if

      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine scancycle_MPI', 24 )

!     /*   mpi communication   */
      call my_mpi_bcast_real_0 ( scale )

!-----------------------------------------------------------------------
!     /*   open file                                                  */
!-----------------------------------------------------------------------

      open ( iounit_scn, file = 'structure.dat' )

!-----------------------------------------------------------------------
!     /*   start main loop                                            */
!-----------------------------------------------------------------------

      do istep = 1, nstep

!-----------------------------------------------------------------------
!        /*   read configuration                                      */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then

            do l = 1, nbead

               read( iounit_scn, *, iostat=ierr )
               read( iounit_scn, *, iostat=ierr )

               do k = 1, natom
                  read( iounit_scn, *, iostat=ierr ) &
     &               char, x(k,l), y(k,l), z(k,l)
               end do

            end do

         end if

         call my_mpi_bcast_int_0( ierr )

         if ( ierr .ne. 0 ) return

         if ( myrank .eq. 0 ) then

            x(:,:) = x(:,:) * scale
            y(:,:) = y(:,:) * scale
            z(:,:) = z(:,:) * scale

         end if

         call my_mpi_bcast_real_2( x, natom, nbead )
         call my_mpi_bcast_real_2( y, natom, nbead )
         call my_mpi_bcast_real_2( z, natom, nbead )

!-----------------------------------------------------------------------
!        /*   for restart                                             */
!-----------------------------------------------------------------------

         if ( istep .le. istep_start ) cycle

!-----------------------------------------------------------------------
!        /*   Cartesian position -> normal mode position              */
!-----------------------------------------------------------------------

         call nm_trans_MPI( 1 )

!-----------------------------------------------------------------------
!        /*   force calculation                                       */
!-----------------------------------------------------------------------

         call getforce_MPI

!-----------------------------------------------------------------------
!        /*   output:  coordinates                                    */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'coord_scan.out', access = 'append' )

            do l = 1, nbead
            do k = 1, natom
               write( iounit, '(i8,i4,3e24.16)' ) &
     &            istep, l, x(k,l), y(k,l), z(k,l)
            end do
            end do

            close( iounit )

         end if

         if ( myrank .eq. 0 ) then

            do l = 1, nbead

               call int3_to_char( l, char_num )

               open ( iounit, file = 'scan.' // char_num // '.xyz' )

               write( iounit, '(i8)' ) natom
               write( iounit, '(a)' ) 'ANGSTROM'

               do k = 1, natom

                  xa = x(k,l) * bohr2ang
                  ya = y(k,l) * bohr2ang
                  za = z(k,l) * bohr2ang

                  xb = x(k,l)
                  yb = y(k,l)
                  zb = z(k,l)

                  m1 = mbox(1,k,l)
                  m2 = mbox(2,k,l)
                  m3 = mbox(3,k,l)

                  call pbc_unfold_MPI( xb, yb, zb, m1, m2, m3 )

                  xb = xb * bohr2ang
                  yb = yb * bohr2ang
                  zb = zb * bohr2ang

                  write( iounit, '(a4,6f16.8)' ) &
     &               species(k)(1:4), xa, ya, za, xb, yb, zb

               end do

               close( iounit )

            end do

         end if

!-----------------------------------------------------------------------
!        /*   output:  potential                                      */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'potential_scan.out', &
     &             access = 'append' )

            do l = 1, nbead
               write( iounit, '(i8,i4,e24.16)' ) istep, l, pot(l)
            end do

            close( iounit )

         end if

!-----------------------------------------------------------------------
!        /*   output:  forces                                         */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'forces_scan.out', access = 'append' )

            do l = 1, nbead
            do k = 1, natom
               write( iounit, '(i8,i4,3e24.16)' ) &
     &            istep, l, fx(k,l)*nbead, fy(k,l)*nbead, fz(k,l)*nbead
            end do
            end do

            close( iounit )

         end if

!-----------------------------------------------------------------------
!        /*   output:  dipole                                         */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'dipole_scan.out', access = 'append' )

            do l = 1, nbead
                write( iounit, '(i8,i4,3f12.5)' ) &
     &             istep, l, dipx(l), dipy(l), dipz(l)
            end do

            close( iounit )

         end if

!-----------------------------------------------------------------------
!        /*   output:  average potential                              */
!-----------------------------------------------------------------------

         call standard_scan_MPI

!-----------------------------------------------------------------------
!        /*   post process                                            */
!-----------------------------------------------------------------------

!        /*   if execution shell is not blank   */
         if ( scan_exe_shell(1:1) .ne. ' ' ) then

!           /*   loop of beads   */
            do l = 1, nbead

!              /*   skip if `ibead is not my job'   */
               if ( mod(l-1,nprocs) .ne. myrank ) cycle

!              /*   bead number   */
               call int3_to_char( l, char_num )

!              /*   change execution permission of shell   */
               call system &
     &            ( "chmod +x " // &
     &              trim(scan_exe_shell) // '.' // char_num // '.sh' )

!              /*   execution of shell   */
               call system &
     &            ( trim(scan_exe_shell) // '.' // char_num // '.sh' )

!           /*   loop of beads   */
            end do

!        /*   if execution shell is not blank   */
         end if

!-----------------------------------------------------------------------
!        /*   print the minfo files if requested                      */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then
            if ( iprint_minfo .ne. -1 ) then
               do l = 1, nbead
                  call print_minfo( l, istep )
               end do
            end if
         end if

!-----------------------------------------------------------------------
!        /*   print the xsf files if requested                        */
!-----------------------------------------------------------------------

         if ( iprint_xsf .ne. -1 ) then

            call analysis_xsf_MPI( 2 )

         end if

!-----------------------------------------------------------------------
!        /*   for restart                                             */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'step.ini' )

            write( iounit, '(i8)' ) istep

            close( iounit )

         end if

!-----------------------------------------------------------------------
!     /*   exit if `exit.dat' exists                                  */
!-----------------------------------------------------------------------

         call softexit_MPI

         if ( iexit .eq. 1 ) exit

!-----------------------------------------------------------------------
!     /*   end of main loop                                           */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   close file                                                 */
!-----------------------------------------------------------------------

      close( iounit_scn )

      return
      end





!***********************************************************************
      subroutine standard_scan_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   potential, char_date, iprint_std, iounit_std, istep, iounit, &
     &   myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         if ( myrank .ne. 0 ) return

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '===================================================='
            write(iounit_std,'(a)') &
     &      '    step  potential [au]  wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------'

            write(         6,'(a)') &
     &      '===================================================='
            write(         6,'(a)') &
     &      '    step  potential [au]  wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '===================================================='
            write(         6,'(a)') &
     &      '    step  potential [au]  wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------'

         end if

      end if

      if ( iprint_std .le. 0 ) return

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then
!        /*   wall clock time   */
         call getdate
!        /*   output   */
         write(iounit_std,'(i8,f16.8,2x,a26)') &
     &      istep, potential, char_date
         write(         6,'(i8,f16.8,2x,a26)') &
     &      istep, potential, char_date
      end if

      return
      end
