!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     polymers run path integral MD
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine run_polymers_atom_MPI
!***********************************************************************
!=======================================================================
!
!     the molecular dynamics cycle:  thermostat type III.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref

      use polymers_variables, only : &
     &   jpoly, myrank_top, nprocs_top, npoly, istep_poly

!     /*   local variables   */
      implicit none

!     /*   initialize f_poly, e_poly  */
      call save_force_polymers_atom_MPI( 1 )

!     /*   reset step counter istep_poly   */
      call reset_step_polymers_atom_MPI

!     /*   loop of polymers   */
      do jpoly = 1, npoly

!        /*   select my polymer   */
         if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle

!        /*   initialize step   */
         istep_start = istep_poly(jpoly)
         istep = istep_start
         istep_end = istep

!        /*   reset or read averages f_avg, e_avg   */
         call analysis_polymers_atom_MPI ( 1 )

!        /*   read geometry and thermostats   */
         call restart_polymers_atom_MPI( 1 )

!        /*   get centroids of my polymer   */
         call save_pos_polymers_atom_MPI( 1 )

!        /*   skip last step   */
         if ( istep .ge. nstep ) go to 100

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   get harmonic force  */
         call getforce_ref

!        /*   calculate the hamiltonian and temperature   */
         call standard_polymers_atom_MPI

!        /*   run pimd   */
         do istep = istep_start+1, nstep

!           /*   current step   */
            istep_end = istep

!           /*   update thermostats attached to centroids  */
            call update_mnhc_cent

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   start multiple time step cycle   */
            do iref = 1, nref

!              /*  update thermostats attached to non-centroid modes  */
               call update_mnhc_mode_MPI

!              /*   update the velocities by harmonic forces   */
               call update_vel_ref

!              /*   update all the normal mode coordinates   */
               call update_pos

!              /*   get harmonic forces   */
               call getforce_ref

!              /*   update the velocities by harmonic forces   */
               call update_vel_ref

!              /*  update thermostats attached to non-centroid modes  */
               call update_mnhc_mode_MPI

            end do

!           /*   normal mode position -> Cartesian position   */
            call nm_trans_MPI(0)

!           /*   get interatomic forces   */
            call getforce_MPI

!           /*   Cartesian force -> normal mode force   */
            call nm_trans_force_MPI(1)

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   update thermostats attached to centroids  */
            call update_mnhc_cent

!           /*   normal mode velocity -> Cartesian velocity   */
            call nm_trans_velocity_MPI(0)

!           /*   calculate the hamiltonian and temperature   */
            call standard_polymers_atom_MPI

!           /*   do some analysis   */
            call analysis_polymers_atom_MPI ( 2 )

!           /*   print restart file   */
            call backup_polymers_atom_MPI

!           /*   exit if `exit.dat' exists   */
            call softexit_MPI
            if ( iexit .eq. 1 ) exit

!        /*   run pimd   */
         end do

!        /*   current step   */
         istep = istep_end
         istep_poly(jpoly) = istep

         /*   print restart file   */
         call backup_polymers_atom_MPI

!        /*   exit if `exit.dat' exists   */
         if ( iexit .eq. 1 ) exit

!        /*   f_avg, e_avg -> f_poly, e_poly  */
  100    call save_force_polymers_atom_MPI( 2 )

!     /*   loop of polymers   */
      end do

!     /*   communicate f_poly, e_poly  */
      call save_force_polymers_atom_MPI( 3 )

!     /*   wait   */
      call my_mpi_barrier_world

      return
      end





!***********************************************************************
      subroutine run_polymers_cons_MPI
!***********************************************************************
!=======================================================================
!
!     the molecular dynamics cycle:  thermostat type III.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref

      use polymers_variables, only : &
     &   jpoly, myrank_top, nprocs_top, npoly, istep_poly

!     /*   local variables   */
      implicit none

!     /*   initialize f_poly, e_poly  */
      call save_force_polymers_cons_MPI( 1 )

!     /*   reset step counter istep_poly   */
      call reset_step_polymers_cons_MPI

!     /*   loop of polymers   */
      do jpoly = 1, npoly

!        /*   select my polymer   */
         if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle

!        /*   initialize step   */
         istep_start = istep_poly(jpoly)
         istep = istep_start
         istep_end = istep

!        /*   read or reset averages f_avg, e_avg  */
         call analysis_polymers_atom_MPI ( 1 )

!        /*   read geometry   */
         call restart_polymers_atom_MPI( 1 )

!        /*   read constraints   */
         call restart_polymers_cons_MPI( 1 )

!        /*   get centroids of my polymer   */
         call save_pos_polymers_atom_MPI( 1 )

!        /*   get centroids of my polymer   */
         call save_pos_polymers_cons_MPI( 1 )

!        /*   skip last step   */
         if ( istep .ge. nstep ) go to 100

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   get harmonic force  */
         call getforce_ref

!        /*   get constrained force  */
         call getforce_ref_cons_cent_MPI

!        /*   calculate the hamiltonian and temperature   */
         call standard_polymers_cons_MPI

!        /*   run pimd   */
         do istep = istep_start+1, nstep

!           /*   current step   */
            istep_end = istep

!           /*   update thermostats attached to centroids  */
            call update_mnhc_cent

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   start multiple time step cycle   */
            do iref = 1, nref

!              /*  update thermostats attached to non-centroid modes  */
               call update_mnhc_mode_MPI

!              /*   update the velocities by harmonic forces   */
               call update_vel_ref

!              /*   update all the normal mode coordinates   */
               call update_pos

!              /*   get harmonic forces   */
               call getforce_ref

!              /*   get constrained force  */
               call getforce_ref_cons_cent_MPI

!              /*   update the velocities by harmonic forces   */
               call update_vel_ref

!              /*  update thermostats attached to non-centroid modes  */
               call update_mnhc_mode_MPI

            end do

!           /*   normal mode position -> Cartesian position   */
            call nm_trans_MPI(0)

!           /*   get interatomic forces   */
            call getforce_MPI

!           /*   Cartesian force -> normal mode force   */
            call nm_trans_force_MPI(1)

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   update thermostats attached to centroids  */
            call update_mnhc_cent

!           /*   normal mode velocity -> Cartesian velocity   */
            call nm_trans_velocity_MPI(0)

!           /*   calculate the hamiltonian and temperature   */
            call standard_polymers_cons_MPI

!           /*   do some analysis   */
            call analysis_polymers_cons_MPI ( 2 )

!           /*   print restart file   */
            call backup_polymers_cons_MPI

!           /*   exit if `exit.dat' exists   */
            call softexit_MPI
            if ( iexit .eq. 1 ) exit

!        /*   run pimd   */
         end do

!        /*   current step   */
         istep = istep_end
         istep_poly(jpoly) = istep

!        /*   print restart file   */
         call backup_polymers_cons_MPI

!        /*   exit if `exit.dat' exists   */
         if ( iexit .eq. 1 ) exit

!        /*   f_avg, e_avg -> f_poly, e_poly  */
  100    call save_force_polymers_cons_MPI( 2 )

!     /*   loop of polymers   */
      end do

!     /*   communicate f_poly, e_poly  */
      call save_force_polymers_cons_MPI( 3 )

!     /*   wait   */
      call my_mpi_barrier_world

      return
      end





!***********************************************************************
      subroutine reset_step_polymers_atom_MPI
!***********************************************************************

!     /*   shared variables   */
      use common_variables, only : &
     &   nstep, iounit, myrank_world

!     /*   shared variables   */
      use polymers_variables, only : &
     &   istep_poly, npoly, jpoly, myrank_top, nprocs_top

!     /*   local variables   */
      implicit none
      character(len=3) :: char_num
      integer :: ierr

!     /*   read step   */
      if ( myrank_world .eq. 0 ) then
         do jpoly = 1, npoly
            call int3_to_char( jpoly, char_num )
            open ( iounit, file = 'poly.' //char_num// '/step.ini' )
            read ( iounit, *, iostat=ierr ) istep_poly(jpoly)
            close( iounit )
            if ( ierr .ne. 0 ) istep_poly(jpoly) = 0
         end do
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_1_world( istep_poly, npoly )

!     /*   if all steps reached nstep   */
      do jpoly = 1, npoly
         if ( istep_poly(jpoly) .lt. nstep ) return
      end do

!     /*   write geometry and thermostats   */
      do jpoly = 1, npoly
         if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle
         call restart_polymers_atom_MPI( 1 )
         call save_pos_polymers_atom_MPI( 1 )
         istep_poly(jpoly) = 0
         call restart_polymers_atom_MPI( 2 )
         call save_pos_polymers_atom_MPI( 2 )
      end do

!     /*   wait   */
      call my_mpi_barrier_world

      return
      end





!***********************************************************************
      subroutine reset_step_polymers_cons_MPI
!***********************************************************************

!     /*   shared variables   */
      use common_variables, only : &
     &   nstep, iounit, myrank_world

!     /*   shared variables   */
      use polymers_variables, only : &
     &   istep_poly, npoly, jpoly, myrank_top, nprocs_top

!     /*   local variables   */
      implicit none
      character(len=3) :: char_num
      integer :: ierr

!     /*   read step   */
      if ( myrank_world .eq. 0 ) then
         do jpoly = 1, npoly
            call int3_to_char( jpoly, char_num )
            open ( iounit, file = 'poly.' //char_num// '/step.ini' )
            read ( iounit, *, iostat=ierr ) istep_poly(jpoly)
            close( iounit )
            if ( ierr .ne. 0 ) istep_poly(jpoly) = 0
         end do
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_1_world( istep_poly, npoly )

!     /*   if all steps reached nstep   */
      do jpoly = 1, npoly
         if ( istep_poly(jpoly) .lt. nstep ) return
      end do

!     /*   write geometry and thermostats   */
      do jpoly = 1, npoly
         if ( mod( jpoly-1, nprocs_top ) .ne. myrank_top ) cycle
         call restart_polymers_atom_MPI( 1 )
         call restart_polymers_cons_MPI( 1 )
         call save_pos_polymers_atom_MPI( 1 )
         call save_pos_polymers_cons_MPI( 1 )
         istep_poly(jpoly) = 0
         call restart_polymers_atom_MPI( 2 )
         call restart_polymers_cons_MPI( 2 )
         call save_pos_polymers_atom_MPI( 2 )
         call save_pos_polymers_cons_MPI( 2 )
      end do

!     /*   wait   */
      call my_mpi_barrier_world

      return
      end
