!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     script to run MOLPRO
!
!///////////////////////////////////////////////////////////////////////
!-----------------------------------------------------------------------
!
!     input files:
!
!        input.dat          - input file of pimd code.
!
!        input_default.dat  - default input file of pimd code.
!                           - reads number of atoms, number of states.
!                           - reads molpro execution command.
!
!        molpro.dat         - a prototype of molpro input file
!                             includes cartesian coordinates in anstrom.
!                           - cartesian coordinates should start from
!                             third line below the keyword 'geometry{'.
!
!        geometry.ini       - restart file of pimd code.
!                           - reads cartesian coordinates in bohr.
!
!     intermediate files:
!
!        input.molpro       - molpro input file copied from molpro.dat.
!                           - read by molpro code.
!
!        output.molpro      - molpro standard output.
!                           - written by molpro code.
!
!        molpro.xml         - molpro output file.
!                           - written by molpro code.
!
!        table.molpro       - data tables of molpro output
!                           - user must edit molpro.dat appropriately
!                             so that data tables are written
!                             by molpro code in the format given below.
!
!                           - format for n-atoms and m-states:
!
!
!
!                             <energy>
!                             E
!                             e(1)
!                             e(2)
!                             ....
!                             e(m)
!
!
!
!                             <grad>  1
!                             GRADX     GRADY     GRADZ
!                             gradx(1), grady(1), gradz(1) of 1st state
!                             gradx(2), grady(2), gradz(2) of 1st state
!                             ........, ........, ........ .. .........
!                             gradx(n), grady(n), gradz(n) of 1st state
!
!                             <grad>  2
!                             GRADX     GRADY     GRADZ
!                             gradx(1), grady(1), gradz(1) of 2nd state
!                             gradx(2), grady(2), gradz(2) of 2nd state
!                             ........, ........, ........ .. .........
!                             gradx(n), grady(n), gradz(n) of 2nd state
!
!                             .....   .
!                             ........, ........, ........ .. .........
!                             ........, ........, ........ .. .........
!
!                             <grad>  m
!                             GRADX     GRADY     GRADZ
!                             gradx(1), grady(1), gradz(1) of mth state
!                             gradx(2), grady(2), gradz(2) of mth state
!                             ........, ........, ........ .. .........
!                             gradx(n), grady(n), gradz(n) of mth state
!
!
!
!                             <dipole>
!                             DMX       DMY       DMZ
!                             dmx,      dmy,      dmz of 1st state
!                             dmx,      dmy,      dmz of 2nd state
!                             dmx,      dmy,      dmz of 3rd state
!
!
!
!                             <nacme> 1 2
!                             GRADX     GRADY     GRADZ
!                             gradx(1), grady(1), gradz(1) of states 1-2
!                             gradx(2), grady(2), gradz(2) of states 1-2
!                             ........, ........, ........ .. .........
!                             gradx(n), grady(n), gradz(n) of states 1-2
!
!                             <nacme> 1 3
!                             GRADX     GRADY     GRADZ
!                             gradx(1), grady(1), gradz(1) of states 1-3
!                             gradx(2), grady(2), gradz(2) of states 1-3
!                             ........, ........, ........ .. ...... ...
!                             gradx(n), grady(n), gradz(n) of states 1-3
!
!                             ....... . .
!                             ........, ........, ........ .. ...... ...
!                             ........, ........, ........ .. ...... ...
!
!                             <nacme> m-1, m
!                             GRADX     GRADY     GRADZ
!                             gradx(1), grady(1), gradz(1) of (m-1)-m
!                             gradx(2), grady(2), gradz(2) of (m-1)-m
!                             ........, ........, ........ .. .......
!                             gradx(n), grady(n), gradz(n) of (m-1)-m
!
!
!
!                           - non-adiabatic coupling matrix elements
!                             <nacme> are ignored for single-state case.
!
!                           -  format for single-state:
!
!
!
!                             <energy>
!                             e(1)
!
!
!
!                             <grad>  1
!                             gradx(1), grady(1), gradz(1) of 1st state
!                             gradx(2), grady(2), gradz(2) of 1st state
!                             ........, ........, ........ .. .........
!                             gradx(n), grady(n), gradz(n) of 1st state
!
!
!
!     output files:
!
!        results.molpro     - summary of all the results.
!                           - to be read by pimd code.
!
!-----------------------------------------------------------------------
!
!***********************************************************************
      program run_molpro
!***********************************************************************

!     /*   number of atoms   */
      integer :: natom

!     /*   number of states   */
      integer :: nstate

!     /*   atomic symbol   */
      character(len=4)  :: symbol

!     /*   geometry   */
      real(8), dimension(:,:),   allocatable:: x, y, z

!     /*   potential   */
      real(8), dimension(:,:),   allocatable:: vstate

!     /*   gradient   */
      real(8), dimension(:,:,:), allocatable:: gxstate, gystate, gzstate

!     /*   nonadiabatic coupling matrix elements   */
      real(8), dimension(:,:,:), allocatable:: dxstate, dystate, dzstate

!     /*   dipole moment   */
      real(8), dimension(:), allocatable:: dipxstate
      real(8), dimension(:), allocatable:: dipystate
      real(8), dimension(:), allocatable:: dipzstate

!     /*   molpro execution command   */
      character(len=80) :: molpro_command

!     /*   constants   */
      real(8), parameter :: au_length = 0.529177249d-10
      real(8), parameter :: bohr2ang  = au_length*1.d+10

!     /*   file numbers   */
      integer :: iounit         = 10
      integer :: iounit_molpro  = 11

!     /*   integers   */
      integer :: i, j, k, l, m, ierr

!     /*   characters   */
      character(len=80) :: char_line

!     /*   real numbers   */
      real(8) :: xa, ya, za

!-----------------------------------------------------------------------
!     /*   read number of atoms                                       */
!-----------------------------------------------------------------------

      natom = 0

      open ( iounit, file = 'structure.dat' )
         read( iounit, *, iostat=ierr ) natom
      close( iounit )

      if ( ierr .ne. 0 )  then
         open ( iounit, file = 'centroid.dat' )
         do
            read ( iounit, *, iostat=ierr ) char_line
            if ( ierr .ne. 0 ) exit
            natom = natom + 1
         end do
         close( iounit )
      end if

      ierr = 0
      if ( natom .le. 0 ) ierr = 1

      call error_handling ( ierr, 'run_molpro', 10 )

!-----------------------------------------------------------------------
!     /*   read number of states                                      */
!-----------------------------------------------------------------------

      open ( iounit, file = 'input.dat' )
         do
            read ( iounit, *, iostat=ierr )  char_line
            if ( char_line(1:8) .eq. '<nstate>' ) then
               read( iounit, *, iostat=ierr ) nstate
               exit
            end if
            if ( ierr .ne. 0 ) exit
         end do
      close( iounit )

      if ( ierr .ne. 0 )  then
         open ( iounit, file = 'input_default.dat' )
            do
               read ( iounit, *, iostat=ierr )  char_line
               if ( char_line(1:8) .eq. '<nstate>' ) then
                  read( iounit, *, iostat=ierr ) nstate
                  exit
               end if
            end do
         close( iounit )
         call error_handling ( ierr, 'run_molpro', 10 )
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      allocate ( x(natom,1) )
      allocate ( y(natom,1) )
      allocate ( z(natom,1) )

      allocate ( vstate(nstate,nstate) )

      allocate ( gxstate(nstate,nstate,natom) )
      allocate ( gystate(nstate,nstate,natom) )
      allocate ( gzstate(nstate,nstate,natom) )

      allocate ( dxstate(nstate,nstate,natom) )
      allocate ( dystate(nstate,nstate,natom) )
      allocate ( dzstate(nstate,nstate,natom) )

      allocate ( dipxstate(nstate) )
      allocate ( dipystate(nstate) )
      allocate ( dipzstate(nstate) )

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      vstate(:,:)     =  0.d0

      dxstate(:,:,:)  =  0.d0
      dystate(:,:,:)  =  0.d0
      dzstate(:,:,:)  =  0.d0

      gxstate(:,:,:)  =  0.d0
      gystate(:,:,:)  =  0.d0
      gzstate(:,:,:)  =  0.d0

      dipxstate(:)    =  0.d0
      dipystate(:)    =  0.d0
      dipzstate(:)    =  0.d0

!-----------------------------------------------------------------------
!     /*   read geometry                                              */
!-----------------------------------------------------------------------

      open ( iounit, file = 'geometry.ini' )

         do i = 1, natom
            read( iounit, *, iostat=ierr ) &
     &         j, x(i,1), y(i,1), z(i,1)
         end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   set molpro execution command                               */
!-----------------------------------------------------------------------

      open ( iounit, file = 'input.dat' )
         do
            read ( iounit, *, iostat=ierr )  char_line
            if ( char_line(1:16) .eq. '<molpro_command>' ) then
                  read ( iounit, *, iostat=ierr ) molpro_command
               exit
            end if
            if ( ierr .ne. 0 ) exit
         end do
      close( iounit )

      if ( ierr .ne. 0 )  then
         open ( iounit, file = 'input_default.dat' )
            do
               read ( iounit, *, iostat=ierr )  char_line
               if ( char_line(1:16) .eq. '<molpro_command>' ) then
                     read ( iounit, *, iostat=ierr ) molpro_command
                  exit
               end if
            end do
         close( iounit )
         call error_handling ( ierr, 'run_molpro', 10 )
      end if

!-----------------------------------------------------------------------
!     /*   make molpro input:  input.molpro                           */
!-----------------------------------------------------------------------

!     /*   open the molpro prototype file   */
      open( iounit, file = 'molpro.dat' )

!     /*   open the molpro input file   */
      open( iounit_molpro, file = 'input.molpro' )

      do

!        /*   read a line   */
         read ( iounit, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   adjusted   */
         char_line = adjustl(char_line)

!        /*   if matched   */
         if ( char_line(1:10) .eq. 'geometry={' ) then

            write( iounit_molpro, '(a)' ) char_line

            do i = 1, 2
               read ( iounit, '(a)' )        char_line
               write( iounit_molpro, '(a)' ) char_line
            end do

            do i = 1, natom

               read ( iounit, * ) symbol

!              /*   in angstroms   */
               xa = x(i,1)*bohr2ang
               ya = y(i,1)*bohr2ang
               za = z(i,1)*bohr2ang

!              /*   write atomic number and coordinates   */
               write ( iounit_molpro, '(a4,1x,3e24.16)' ) &
     &            symbol, xa, ya, za

            end do

         else

!           /*   write a copy of the line   */
            write( iounit_molpro, '(a)' ) char_line

         end if

      end do

!     /*   add a blank line at the end   */
      write ( iounit_molpro, '(a)' )

!     /*   close files   */
      close(iounit)
      close(iounit_molpro)

!-----------------------------------------------------------------------
!     /*   run molpro
!-----------------------------------------------------------------------

!     /*   run molpro   */
      call system( molpro_command // ' < input.molpro > output.molpro' )

!-----------------------------------------------------------------------
!     /*   read molpro output:  potential                             */
!-----------------------------------------------------------------------

!     /*   open the molpro output file   */
      open( iounit_molpro, file = 'table.molpro' )

      do

!        /*   read a line   */
         read ( iounit_molpro, *, iostat=ierr ) char_line

!        /*   read error   */
         if ( ierr .ne. 0 ) exit

!        /*   look for energy   */
         if ( char_line(1:8) .eq. '<energy>' ) then

!           /*   read skip one line   */
            read( iounit_molpro, * )

!           /*   read potential of all states   */
            do j = 1, nstate
               read( iounit_molpro, * ) vstate(j,j)
            end do

         end if

      end do

!     /*   close file   */
      close( iounit_molpro )

!-----------------------------------------------------------------------
!     /*   read molpro output:  gradient                              */
!-----------------------------------------------------------------------

!     /*   open the molpro output file   */
      open( iounit_molpro, file = 'table.molpro' )

      do

!        /*   read a line   */
         read ( iounit_molpro, *, iostat=ierr ) char_line

!        /*   read error   */
         if ( ierr .ne. 0 ) exit

!        /*   look for gradient   */
         if ( char_line(1:6) .eq. '<grad>' ) then

!           /*   go back one line   */
            backspace( iounit_molpro )

!           /*   read state   */
            read( iounit_molpro, *, iostat=ierr ) char_line, k

!           /*   read skip one line   */
            read( iounit_molpro, *, iostat=ierr )

!           /*   out of range  */
            if ( k .le. 0      ) cycle
            if ( k .gt. nstate ) cycle

!           /*   read gradient from molpro output   */
            do i = 1, natom
               read( iounit_molpro, *, iostat=ierr ) &
     &            gxstate(k,k,i), gystate(k,k,i), gzstate(k,k,i)
            end do

         end if

      end do

!     /*   close file   */
      close( iounit_molpro )

!-----------------------------------------------------------------------
!     /*   read molpro output:  dipole                                */
!-----------------------------------------------------------------------

!     /*   open the molpro output file   */
      open( iounit_molpro, file = 'table.molpro' )

      do

!        /*   read a line   */
         read ( iounit_molpro, *, iostat=ierr ) char_line

!        /*   read error   */
         if ( ierr .ne. 0 ) exit

!        /*   look for gradient   */
         if ( char_line(1:8) .eq. '<dipole>' ) then

!           /*   read skip one line   */
            read( iounit_molpro, *, iostat=ierr )

!           /*   read gradient from molpro output   */
            do k = 1, nstate
               read( iounit_molpro, *, iostat=ierr ) &
     &            dipxstate(k), dipystate(k), dipzstate(k)
            end do

         end if

      end do

!     /*   close file   */
      close( iounit_molpro )

!-----------------------------------------------------------------------
!     /*   read molpro output:  nonadiabatic coupling matrix          */
!-----------------------------------------------------------------------

!     /*   open the molpro output file   */
      open( iounit_molpro, file = 'table.molpro' )

      do

!        /*   read a line   */
         read ( iounit_molpro, *, iostat=ierr ) char_line

!        /*   read error   */
         if ( ierr .ne. 0 ) exit

!        /*   look for nacm   */
         if ( char_line(1:7) .eq. '<nacme>' ) then

!           /*   go back one line   */
            backspace( iounit_molpro )

!           /*   read pair of states   */
            read( iounit_molpro, *, iostat=ierr ) char_line, l, m

!           /*   read skip one line   */
            read( iounit_molpro, *, iostat=ierr )

!           /*   out of range  */
            if ( l .le. 0      ) cycle
            if ( l .gt. nstate ) cycle
            if ( m .le. 0      ) cycle
            if ( m .gt. nstate ) cycle

!           /*   read nonadiabatic coupling matrix   */
            do i = 1, natom
               read( iounit_molpro, *, iostat=ierr ) &
     &            dxstate(l,m,i), dystate(l,m,i), dzstate(l,m,i)
            end do

!           /*   antisymmetrize   */
            do i = 1, natom
               dxstate(m,l,i) = - dxstate(l,m,i)
               dystate(m,l,i) = - dystate(l,m,i)
               dzstate(m,l,i) = - dzstate(l,m,i)
            end do

         end if

      end do

!     /*   close file   */
      close(iounit_molpro)

!-----------------------------------------------------------------------
!     /*   write                                                      */
!-----------------------------------------------------------------------

!     /*   open the molpro output file   */
      open( iounit, file = 'results.molpro' )

!     /*   potential   */

      do k = 1, nstate
      do j = 1, nstate
         write( iounit, '(e24.16)' ) vstate(j,k)
      end do
      end do

!     /*   gradient   */

      do k = 1, nstate
      do j = 1, nstate
      do i = 1, natom
         write( iounit, '(3e24.16)' ) &
     &      gxstate(j,k,i), gystate(j,k,i), gzstate(j,k,i)
      end do
      end do
      end do

!     /*   dipole moment   */

      do k = 1, nstate
         write( iounit, '(3e24.16)' ) &
     &      dipxstate(k), dipystate(k), dipzstate(k)
      end do

!     /*   nonadiabatic coupling matrix   */

      do k = 1, nstate
      do j = 1, nstate
      do i = 1, natom
         write( iounit, '(3e24.16)' ) &
     &      dxstate(j,k,i), dystate(j,k,i), dzstate(j,k,i)
      end do
      end do
      end do

!     /*   close file  */
      close( iounit )

!-----------------------------------------------------------------------
!     /*   remove files                                               */
!-----------------------------------------------------------------------

      call system(' rm -f molpro.xml*' )

!      call system(' rm -f output.molpro' )
!      call system(' rm -f input.molpro' )

      stop
      end





!***********************************************************************
      subroutine error_handling ( ierr, char_tag, length_tag )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: ierr, length_tag

      character(len=length_tag) :: char_tag

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then
         write(6,'(a)') 'STOP:  error at: ' // char_tag // '.'
         stop
      end if

      return
      end

