!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Oct 29, 2021 by M. Shiga
!      Description:     Ehrenfest RPMD
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine rpmfecycle_MPI
!***********************************************************************

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_rpmfe_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_rpmfe_MPI

!     /*   do some analysis   */
      call analysis_mfe_MPI ( 1 )

!     /*   main md cycle  */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI(0)

!        /*   update cstate   */
         call update_cstate_MPI

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   update all the normal mode coordinates   */
         call update_posvel_anal

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   ab initio calculation   */
         call getforce_rpmfe_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI(0)

!        /*   update cstate   */
         call update_cstate_MPI

!        /*   hamiltonian   */
         call standard_rpmfe_MPI

!        /*   do some analysis   */
         call analysis_mfe_MPI ( 2 )

!        /*   output restart   */
         call backup_rpmfe_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!     /*   main md cycle  */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine setup_rpmfe_MPI
!***********************************************************************
!=======================================================================
!
!     set up rpmfe
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, ipos_start, iounit, nstep, natom, nbead, myrank

      use multistate_variables, only : &
     &   vstate, dxstate, dystate, dzstate, gxstate, gystate, gzstate, &
     &   dipxstate, dipystate, dipzstate, cstate, pop_state, pop_sum, &
     &   branch_ratio, dxstate_old, dystate_old, dzstate_old, &
     &   istate_tfs, nstate, istate_init

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, itest

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams_MPI

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix_MPI

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_rpmd_MPI

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file, and               */
!     /*   non-centroid modes are in Gaussian distribution            */
      if     ( ipos_start .eq. 0 ) then
         call init_centroid_MPI
         call init_mode_MPI
!     /*   read normal mode position                                  */
      else if ( ipos_start .eq. 1 ) then
         call restart_position_MPI( 1 )
!     /*   read Cartesian position                                    */
      else if ( ipos_start .eq. 2 ) then
         call restart_position_MPI( 2 )
      else
         call error_handling_MPI( 1, 'subroutine setup_rpmfe_MPI', 26)
      end if

!     /*   Maxwell distribution of velocities                         */
      if     ( ivel_start .eq. 0 ) then
         call init_velocity_MPI
      else if ( ivel_start .eq. 1 ) then
!        /*   read normal mode momentum (scaled)                      */
         call restart_velocity_MPI( 1 )
      else if ( ivel_start .eq. 2 ) then
!        /*   read Cartesian momentum                                 */
         call restart_velocity_MPI( 2 )
      else
         call error_handling_MPI( 1, 'subroutine setup_rpmfe_MPI', 26)
      end if

!-----------------------------------------------------------------------
!     /*   subtract translational and rotational part of velocities   */
!-----------------------------------------------------------------------

      call correct_velocity_MPI

!-----------------------------------------------------------------------
!     /*   number of states                                           */
!-----------------------------------------------------------------------

      call read_int1_MPI ( nstate, '<nstate>', 8, iounit )

!-----------------------------------------------------------------------
!     /*   memory allocation for multi-state calculations             */
!-----------------------------------------------------------------------

!     /*   potential   */
      if ( .not. allocated( vstate ) ) &
     &   allocate( vstate(nstate,nstate,nbead) )

!     /*   gradients   */
      if ( .not. allocated( gxstate ) ) &
     &   allocate( gxstate(nstate,nstate,natom,nbead) )
      if ( .not. allocated( gystate ) ) &
     &   allocate( gystate(nstate,nstate,natom,nbead) )
      if ( .not. allocated( gzstate ) ) &
     &   allocate( gzstate(nstate,nstate,natom,nbead) )

!     /*   nonadiabatic coupling   */
      if ( .not. allocated( dxstate ) ) &
     &   allocate( dxstate(nstate,nstate,natom,nbead) )
      if ( .not. allocated( dystate ) ) &
     &   allocate( dystate(nstate,nstate,natom,nbead) )
      if ( .not. allocated( dzstate ) ) &
     &   allocate( dzstate(nstate,nstate,natom,nbead) )

!     /*   dipole moment   */
      if ( .not. allocated( dipxstate ) ) &
     &   allocate( dipxstate(nstate,nbead) )
      if ( .not. allocated( dipystate ) ) &
     &   allocate( dipystate(nstate,nbead) )
      if ( .not. allocated( dipzstate ) ) &
     &   allocate( dipzstate(nstate,nbead) )

!     /*   state coefficients   */
      if ( .not. allocated( cstate ) ) &
     &   allocate( cstate(nstate,nbead) )

!     /*   state population   */
      if ( .not. allocated( pop_state ) ) &
     &   allocate( pop_state(nstate,nbead) )

!     /*   sum of state population   */
      if ( .not. allocated( pop_sum ) ) &
     &   allocate( pop_sum(nbead) )

!     /*   occupied state   */
      if ( .not. allocated( istate_tfs ) ) &
     &   allocate( istate_tfs(nbead) )

!     /*   branching ratio   */
      if ( .not. allocated( branch_ratio ) ) &
     &   allocate( branch_ratio(nstate) )

!     /*   nonadiabatic coupling   */
      if ( .not. allocated( dxstate_old ) ) &
     &   allocate( dxstate_old(nstate,nstate,natom,nbead) )
      if ( .not. allocated( dystate_old ) ) &
     &   allocate( dystate_old(nstate,nstate,natom,nbead) )
      if ( .not. allocated( dzstate_old ) ) &
     &   allocate( dzstate_old(nstate,nstate,natom,nbead) )

!-----------------------------------------------------------------------
!     /*   test existence of step.ini                                 */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         call testfile ( 'step.ini', 8, itest )
      end if

      call my_mpi_bcast_int_0 ( itest )

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( itest .eq. 1 ) then

!        /*   read initial state   */
         call read_int1_MPI ( istate_init, '<istate_init>', 13, iounit )

!        /*   occupied state initialized for all the beads   */

         do j = 1, nbead
            istate_tfs(j) = istate_init
         end do

!        /*   state coefficients initialized for all the beads   */

         do j = 1, nbead
         do i = 1, nstate

            if ( i .eq. istate_tfs(j) ) then

               cstate(i,j) = dcmplx( 1.d0, 0.d0 )

            else

               cstate(i,j) = dcmplx( 0.d0, 0.d0 )

            end if

         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   restart                                                    */
!-----------------------------------------------------------------------

      if ( itest .ne. 1 ) then

!        /*   read occupied state and state coefficients   */
         call restart_cstate_MPI ( 1 )

      end if

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         call backup_rpmfe_MPI

         stop

      end if

      return
      end





!***********************************************************************
      subroutine backup_rpmfe_MPI
!***********************************************************************
!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nstep, istep_end, iprint_rest, iounit, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( mod(istep_end,iprint_rest) .eq. 0 ) then
         continue
      else if ( istep_end .eq. nstep ) then
         continue
      else
         return
      end if

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

      call restart_position_MPI( 3 )
      call restart_velocity_MPI( 3 )

!-----------------------------------------------------------------------
!     /*   write state coefficients to cstate.ini                     */
!-----------------------------------------------------------------------

      call restart_cstate_MPI ( 2 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
         close( iounit )

      end if

      return
      end





!***********************************************************************
      subroutine getforce_rpmfe_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, pot, potential, vir, nbead, ipotential, natom

      use multistate_variables, only : &
     &   vstate, gxstate, gystate, gzstate, dxstate, dystate, dzstate, &
     &   dipxstate, dipystate, dipzstate, cstate, nstate

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l
      real(8) :: dp

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   potential   */
      vstate(:,:,:) = 0.d0

!     /*   forces   */
      gxstate(:,:,:,:) = 0.d0
      gystate(:,:,:,:) = 0.d0
      gzstate(:,:,:,:) = 0.d0

!     /*   forces   */
      dxstate(:,:,:,:) = 0.d0
      dystate(:,:,:,:) = 0.d0
      dzstate(:,:,:,:) = 0.d0

!     /*   dipole moment   */
      dipxstate(:,:)   = 0.d0
      dipystate(:,:)   = 0.d0
      dipzstate(:,:)   = 0.d0

!-----------------------------------------------------------------------
!     /*   calculate nonadiabatic potential                           */
!     /*      input  = x, y, z                                        */
!     /*      output = vstate, gxstate, gystate, gzstate,             */
!     /*               dxstate, dystate, dzstate                      */
!-----------------------------------------------------------------------

!     ===  molpro  ===

      if      ( ipotential(1:7) .eq. 'MOLPRO ' ) then

         call force_molpro_MPI

!     ===  tully's model  ===

      else if ( ipotential(1:6) .eq. 'TULLY ' ) then

         call force_tully_MPI

!     ===  error  ===

      else

         call error_handling_MPI &
     &      ( 1, 'subroutine getforce_rpmfe_MPI', 29 )

      end if

!-----------------------------------------------------------------------
!     /*   potential                                                  */
!-----------------------------------------------------------------------

      pot(:) = 0.d0

      do l = 1, nbead
      do j = 1, nstate
      do i = 1, nstate
         pot(l) = pot(l) &
     &          + dreal( dconjg(cstate(i,l))*vstate(i,j,l)*cstate(j,l) )
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   forces                                                     */
!-----------------------------------------------------------------------

      do l = 1, nbead
      do k = 1, natom

         fx(k,l) = 0.d0
         fy(k,l) = 0.d0
         fz(k,l) = 0.d0

         do j = 1, nstate
         do i = 1, nstate

            fx(k,l) = fx(k,l) &
     &              - dreal( dconjg(cstate(i,l))*gxstate(i,j,k,l) &
     &                       *cstate(j,l) )
            fy(k,l) = fy(k,l) &
     &              - dreal( dconjg(cstate(i,l))*gystate(i,j,k,l) &
     &                       *cstate(j,l) )
            fz(k,l) = fz(k,l) &
     &              - dreal( dconjg(cstate(i,l))*gzstate(i,j,k,l) &
     &                       *cstate(j,l) )

            fx(k,l) = fx(k,l) &
     &              + dreal( dconjg(cstate(i,l))*dxstate(i,j,k,l) &
     &                       *vstate(i,i,l)*cstate(j,l) )
            fx(k,l) = fx(k,l) &
     &              - dreal( dconjg(cstate(i,l))*dxstate(i,j,k,l) &
     &                       *vstate(j,j,l)*cstate(j,l) )
            fy(k,l) = fy(k,l) &
     &              + dreal( dconjg(cstate(i,l))*dystate(i,j,k,l) &
     &                       *vstate(i,i,l)*cstate(j,l) )
            fy(k,l) = fy(k,l) &
     &              - dreal( dconjg(cstate(i,l))*dystate(i,j,k,l) &
     &                       *vstate(j,j,l)*cstate(j,l) )
            fz(k,l) = fz(k,l) &
     &              + dreal( dconjg(cstate(i,l))*dzstate(i,j,k,l) &
     &                       *vstate(i,i,l)*cstate(j,l) )
            fz(k,l) = fz(k,l) &
     &              - dreal( dconjg(cstate(i,l))*dzstate(i,j,k,l) &
     &                       *vstate(j,j,l)*cstate(j,l) )

         end do
         end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   nonadiabatic coupling vector                               */
!-----------------------------------------------------------------------

      call checkphase_MPI

!-----------------------------------------------------------------------
!     /*   external potential                                         */
!-----------------------------------------------------------------------

      call force_external_MPI

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      dp = dble(nbead)

      potential = 0.d0

      do j = 1, nbead

         potential = potential + pot(j)

         do i = 1, natom

            fx(i,j) = fx(i,j)/dp
            fy(i,j) = fy(i,j)/dp
            fz(i,j) = fz(i,j)/dp

         end do

      end do

      potential = potential/dp

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = vir(i,j)/dp
      end do
      end do

!-----------------------------------------------------------------------
!     /*   apply virial correction                                    */
!-----------------------------------------------------------------------

      call correct_virial

!-----------------------------------------------------------------------
!     /*   correct forces on centroid                                 */
!-----------------------------------------------------------------------

      call correct_force_MPI

      return
      end





!***********************************************************************
      subroutine standard_rpmfe_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     no thermostat.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, &
     &   ebath_cent, ebath_mode, char_date, ndof, iprint_std, &
     &   iounit, iounit_std, nbead, natom, istep, myrank

      use multistate_variables, only : &
     &   pop_state, occ_state, cstate, nstate

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, imode, iatom, i, l

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( myrank .ne. 0 ) return

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'occ wall clock time       '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'occ wall clock time       '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'occ wall clock time       '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

      end if

      if ( iprint_std .le. 0 ) return

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   population of states                                       */
!-----------------------------------------------------------------------

      do l = 1, nbead
      do i = 1, nstate
         pop_state(i,l) = dreal(dconjg(cstate(i,l))*cstate(i,l))
      end do
      end do

!-----------------------------------------------------------------------
!     /*   state occupancy                                            */
!-----------------------------------------------------------------------

      occ_state = 0.d0

      do l = 1, nbead
      do i = 1, nstate
         occ_state = occ_state + dble(i) * pop_state(i,l) / dble(nbead)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then
!        /*   wall clock time   */
         call getdate
!        /*   output   */
         write(iounit_std,'(i8,2f16.8,f10.2,f5.2,1x,a21)') &
     &      istep, hamiltonian, potential, temp, occ_state, &
     &      char_date(6:26)
         write(         6,'(i8,2f16.8,f10.2,f5.2,1x,a21)') &
     &      istep, hamiltonian, potential, temp, occ_state, &
     &      char_date(6:26)
      end if

      return
      end
