!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Feb 12, 2022 by M. Shiga
!      Description:     ring polymer molecular dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine rpmdcycle_nve_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   without constraints                                        */
!-----------------------------------------------------------------------

      if ( ncons .eq. 0 ) then

         call rpmdcycle_anal_MPI

      else

         call rpmdcycle_cons_MPI

      end if

      return
      end





!***********************************************************************
      subroutine rpmdcycle_respa_MPI
!***********************************************************************
!=======================================================================
!
!     The RPMD molecular dynamics cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_nve_0_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   start multiple time step cycle   */
         do iref = 1, nref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*   update all the normal mode coordinates   */
            call update_pos

!           /*   get harmonic forces   */
            call getforce_ref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

         end do

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_nve_0_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pimd_nve_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine rpmdcycle_anal_MPI
!***********************************************************************
!=======================================================================
!
!     The RPMD molecular dynamics cycle (analytical version).
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_nve_0_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   update all the normal mode coordinates   */
         call update_posvel_anal

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_nve_0_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pimd_nve_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine rpmdcycle_xyz_MPI
!***********************************************************************
!=======================================================================
!
!     The RPMD molecular dynamics cycle (in Cartesian).
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   normal mode velocity -> Cartesian velocity   */
      call nm_trans_velocity_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   get harmonic force  */
      call getforce_ref_xyz

!     /*   calculate the hamiltonian and temperature   */
      call standard_nve_0_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_vel_xyz

!        /*   start multiple time step cycle   */
         do iref = 1, nref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref_xyz

!           /*   update all the normal mode coordinates   */
            call update_pos_xyz_MPI

!           /*   get harmonic forces   */
            call getforce_ref_xyz

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref_xyz

         end do

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   update the velocities by interatomic forces   */
         call update_vel_xyz

!        /*   Cartesian velocity -> normal mode velocity   */
         call nm_trans_velocity_MPI(1)

!        /*   Cartesian position -> normal mode position   */
         call nm_trans_MPI(1)

!        /*   Cartesian force    -> normal mode force      */
         call nm_trans_force_MPI(1)

!        /*   calculate the hamiltonian and temperature   */
         call standard_nve_0_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pimd_nve_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine rpmdcycle_nvt_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!
!     The RPMD molecular dynamics cycle (NVT).
!
!-----------------------------------------------------------------------

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_rpmd_nvt_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update thermostats attached to centroids  */
         call update_nhc_cent

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   update all the normal mode coordinates   */
         call update_posvel_anal

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   update thermostats attached to centroids  */
         call update_nhc_cent

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_rpmd_nvt_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_cmd_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine rpmdcycle_cons_MPI
!***********************************************************************
!=======================================================================
!
!     The RPMD molecular dynamics cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   get constrained force  */
      call getforce_ref_cons_cent_MPI

!     /*   calculate the hamiltonian and temperature   */
      call standard_nve_cons_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   start multiple time step cycle   */
         do iref = 1, nref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*   update all the normal mode coordinates   */
            call update_pos

!           /*   get harmonic forces   */
            call getforce_ref

!           /*   get constrained force  */
            call getforce_ref_cons_cent_MPI

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

         end do

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_nve_cons_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pimd_nve_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end
