!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     molecular dynamics of rigid rotors
!
!///////////////////////////////////////////////////////////////////////
!=======================================================================
!
!
!        MOLECULAR DYNAMICS FOR RIGID-BODY MOLECULES
!
!
!        At the moment, we have the following restrictions:
!
!           - input <molecules> in the order of
!             nonlinear, linear, and monoatomic molecules.
!
!           - linear molecules should be arranged in molecular z-axis.
!
!           - input <components> in the order of
!             nonlinear, linear, and monoatomic molecules.
!
!           - nbead must be always 1.
!
!           - on restart, Gear method does not reproduce energy exactly.
!
!=======================================================================
!***********************************************************************
      subroutine rotor_nvt_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     //   initialization
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   apply boundary condition for molecules   */
      call pbc_rotor_MPI

!     /*   transfer from molecular to laboratory axis   */
      call trans_position_rotor

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   transfer to velocities   */
      call trans_velocity_rotor

!     /*   calculate kinetic energy   */
      call kinetic_energy_rotor

!     /*   calculate the hamiltonian and temperature   */
      call standard_rotor_nvt_MPI

!     /*   do some analysis   */
      call analysis_rotor_MPI( 1 )

!-----------------------------------------------------------------------
!     //   md loop start
!-----------------------------------------------------------------------

      do istep = istep_start + 1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update thermostat: predictor part   */
         call update_bath_pre_nvt

!        /*   update molecular center of masses: predictor part   */
         call update_center_of_mass_pre

!        /*   update molecular rotations: predictor part   */
         call update_rotation_pre

!        /*   update molecular quaternions: predictor part   */
         call update_quaternion_pre

!        /*   apply boundary condition for molecules   */
         call pbc_rotor_MPI

!        /*   transfer from molecular to laboratory axis   */
         call trans_position_rotor

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   calculate center of mass forces   */
         call force_center_of_mass

!        /*   calculate molecular torque   */
         call torque_rotor

!        /*   update molecular center of masses: corrector part   */
         call update_center_of_mass_cor_nvt

!        /*   update molecular rotations: corrector part   */
         call update_rotation_cor_nvt

!        /*   transfer from angular velocities to quaternions   */
         call trans_o_qdot

!        /*   update molecular quaternions: corrector part   */
         call update_quaternion_cor

!        /*   transfer from molecular to laboratory axis   */
         call trans_position_rotor

!        /*   apply boundary condition for molecules   */
         call pbc_rotor_MPI

!        /*   transfer to velocities   */
         call trans_velocity_rotor

!        /*   calculate kinetic energy   */
         call kinetic_energy_rotor

!        /*   update thermostat: corrector part   */
         call update_bath_cor_nvt

!        /*   calculate the hamiltonian and temperature   */
         call standard_rotor_nvt_MPI

!        /*   output restart   */
         call backup_rotor_nvt_MPI

!        /*   do some analysis   */
         call analysis_rotor_MPI( 2 )

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!-----------------------------------------------------------------------
!     //   md loop end
!-----------------------------------------------------------------------

      end do

!     /*   current step   */
      istep = istep_end

!     /*   get interatomic forces   */
!      call getforce_MPI

!     /*   calculate the hamiltonian and temperature   */
!      call standard_rotor_nvt_MPI

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine rotor_nve_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     //   initialization
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   apply boundary condition for molecules   */
      call pbc_rotor_MPI

!     /*   transfer from molecular to laboratory axis   */
      call trans_position_rotor

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   transfer to velocities   */
      call trans_velocity_rotor

!     /*   calculate kinetic energy   */
      call kinetic_energy_rotor

!     /*   calculate the hamiltonian and temperature   */
      call standard_rotor_nve_MPI

!     /*   do some analysis   */
      call analysis_rotor_MPI( 1 )

!-----------------------------------------------------------------------
!     //   md loop start
!-----------------------------------------------------------------------

      do istep = istep_start + 1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update molecular center of masses: predictor part   */
         call update_center_of_mass_pre

!        /*   update molecular rotations: predictor part   */
         call update_rotation_pre

!        /*   update molecular quaternions: predictor part   */
         call update_quaternion_pre

!        /*   apply boundary condition for molecules   */
         call pbc_rotor_MPI

!        /*   transfer from molecular to laboratory axis   */
         call trans_position_rotor

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   calculate center of mass forces   */
         call force_center_of_mass

!        /*   calculate molecular torque   */
         call torque_rotor

!        /*   update molecular center of masses: corrector part   */
         call update_center_of_mass_cor_nve

!        /*   update molecular rotations: corrector part   */
         call update_rotation_cor_nve

!        /*   transfer from angular velocities to quaternions   */
         call trans_o_qdot

!        /*   update molecular quaternions: corrector part   */
         call update_quaternion_cor

!        /*   transfer from molecular to laboratory axis   */
         call trans_position_rotor

!        /*   apply boundary condition for molecules   */
         call pbc_rotor_MPI

!        /*   transfer to velocities   */
         call trans_velocity_rotor

!        /*   calculate kinetic energy   */
         call kinetic_energy_rotor

!        /*   calculate the hamiltonian and temperature   */
         call standard_rotor_nve_MPI

!        /*   output restart   */
         call backup_rotor_nve_MPI

!        /*   do some analysis   */
         call analysis_rotor_MPI( 2 )

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!-----------------------------------------------------------------------
!     //   md loop end
!-----------------------------------------------------------------------

      end do

!     /*   current step   */
      istep = istep_end

!     /*   get interatomic forces   */
!      call getforce_MPI

!     /*   calculate the hamiltonian and temperature   */
!      call standard_rotor_nve_MPI

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine rotor_velocity_scaling_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     //   initialization
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   apply boundary condition for molecules   */
      call pbc_rotor_MPI

!     /*   transfer from molecular to laboratory axis   */
      call trans_position_rotor

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   apply velocity scaling   */
      call velocity_scaling_rotor

!     /*   transfer to velocities   */
      call trans_velocity_rotor

!     /*   calculate kinetic energy   */
      call kinetic_energy_rotor

!     /*   calculate the hamiltonian and temperature   */
      call standard_rotor_nve_MPI

!     /*   do some analysis   */
      call analysis_rotor_MPI( 1 )

!-----------------------------------------------------------------------
!     //   md loop start
!-----------------------------------------------------------------------

      do istep = istep_start + 1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update molecular center of masses: predictor part   */
         call update_center_of_mass_pre

!        /*   update molecular rotations: predictor part   */
         call update_rotation_pre

!        /*   update molecular quaternions: predictor part   */
         call update_quaternion_pre

!        /*   apply boundary condition for molecules   */
         call pbc_rotor_MPI

!        /*   transfer from molecular to laboratory axis   */
         call trans_position_rotor

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   calculate center of mass forces   */
         call force_center_of_mass

!        /*   calculate molecular torque   */
         call torque_rotor

!        /*   update molecular center of masses: corrector part   */
         call update_center_of_mass_cor_nve

!        /*   update molecular rotations: corrector part   */
         call update_rotation_cor_nve

!        /*   transfer from angular velocities to quaternions   */
         call trans_o_qdot

!        /*   update molecular quaternions: corrector part   */
         call update_quaternion_cor

!        /*   apply velocity scaling   */
         call velocity_scaling_rotor

!        /*   transfer from molecular to laboratory axis   */
         call trans_position_rotor

!        /*   apply boundary condition for molecules   */
         call pbc_rotor_MPI

!        /*   transfer to velocities   */
         call trans_velocity_rotor

!        /*   calculate kinetic energy   */
         call kinetic_energy_rotor

!        /*   calculate the hamiltonian and temperature   */
         call standard_rotor_nve_MPI

!        /*   output restart   */
         call backup_rotor_nve_MPI

!        /*   do some analysis   */
         call analysis_rotor_MPI( 2 )

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!-----------------------------------------------------------------------
!     //   md loop end
!-----------------------------------------------------------------------

      end do

!     /*   current step   */
      istep = istep_end

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine trans_o_qdot
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use rotor_variables, only : q4, o4, qdot, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: k

!-----------------------------------------------------------------------
!     //   time derivative of quaternion: 6-dimensional molecule
!-----------------------------------------------------------------------

      do k = 1, nmol_6

         qdot(1,k) = 0.5d0 * ( - q4(0,3,k) * o4(0,1,k) &
     &                         - q4(0,4,k) * o4(0,2,k) &
     &                         + q4(0,2,k) * o4(0,3,k) )

         qdot(2,k) = 0.5d0 * (   q4(0,4,k) * o4(0,1,k) &
     &                         - q4(0,3,k) * o4(0,2,k) &
     &                         - q4(0,1,k) * o4(0,3,k) )

         qdot(3,k) = 0.5d0 * (   q4(0,1,k) * o4(0,1,k) &
     &                         + q4(0,2,k) * o4(0,2,k) &
     &                         + q4(0,4,k) * o4(0,3,k) )

         qdot(4,k) = 0.5d0 * ( - q4(0,2,k) * o4(0,1,k) &
     &                         + q4(0,1,k) * o4(0,2,k) &
     &                         - q4(0,3,k) * o4(0,3,k) )

      end do

!-----------------------------------------------------------------------
!     //   time derivative of quaternion: 5-dimensional molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5

         qdot(1,k) = 0.5d0 * ( - q4(0,3,k) * o4(0,1,k) &
     &                         - q4(0,4,k) * o4(0,2,k) )
         qdot(2,k) = 0.5d0 * (   q4(0,4,k) * o4(0,1,k) &
     &                         - q4(0,3,k) * o4(0,2,k) )
         qdot(3,k) = 0.5d0 * (   q4(0,1,k) * o4(0,1,k) &
     &                         + q4(0,2,k) * o4(0,2,k) )
         qdot(4,k) = 0.5d0 * ( - q4(0,2,k) * o4(0,1,k) &
     &                         + q4(0,1,k) * o4(0,2,k) )

      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine trans_o_q1
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use rotor_variables, only : q4, o4, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: k

!-----------------------------------------------------------------------
!     //   time derivative of quaternion: 6-dimensional molecule
!-----------------------------------------------------------------------

      do k = 1, nmol_6

         q4(1,1,k) = 0.5d0 * ( - q4(0,3,k) * o4(0,1,k) &
     &                         - q4(0,4,k) * o4(0,2,k) &
     &                         + q4(0,2,k) * o4(0,3,k) )
         q4(1,2,k) = 0.5d0 * (   q4(0,4,k) * o4(0,1,k) &
     &                         - q4(0,3,k) * o4(0,2,k) &
     &                         - q4(0,1,k) * o4(0,3,k) )
         q4(1,3,k) = 0.5d0 * (   q4(0,1,k) * o4(0,1,k) &
     &                         + q4(0,2,k) * o4(0,2,k) &
     &                         + q4(0,4,k) * o4(0,3,k) )
         q4(1,4,k) = 0.5d0 * ( - q4(0,2,k) * o4(0,1,k) &
     &                         + q4(0,1,k) * o4(0,2,k) &
     &                         - q4(0,3,k) * o4(0,3,k) )

      end do

!-----------------------------------------------------------------------
!     //   time derivative of quaternion: 5-dimensional molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5

         q4(1,1,k) = 0.5d0 * ( - q4(0,3,k) * o4(0,1,k) &
     &                         - q4(0,4,k) * o4(0,2,k) )
         q4(1,2,k) = 0.5d0 * (   q4(0,4,k) * o4(0,1,k) &
     &                         - q4(0,3,k) * o4(0,2,k) )
         q4(1,3,k) = 0.5d0 * (   q4(0,1,k) * o4(0,1,k) &
     &                         + q4(0,2,k) * o4(0,2,k) )
         q4(1,4,k) = 0.5d0 * ( - q4(0,2,k) * o4(0,1,k) &
     &                         + q4(0,1,k) * o4(0,2,k) )

      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine trans_position_rotor
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : x, y, z, ux, uy, uz

      use rotor_variables, only : &
     &   r5, q4, x_mol, y_mol, z_mol, list_atom_mol, &
     &   natom_per_mol, nmol_6, nmol_5, nmol_3

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: k, l, m

!     //   real numbers
      real(8) :: a(3,3)

!-----------------------------------------------------------------------
!     //   transfer from molecular to laboratory frame: 6-d molecule
!-----------------------------------------------------------------------

      do k = 1, nmol_6

         a(1,1) =   2.d0 * ( q4(0,2,k) * q4(0,2,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(1,2) =   2.d0 * ( q4(0,3,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,2,k) )
         a(1,3) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     + q4(0,1,k) * q4(0,4,k) )
         a(2,1) = - 2.d0 * ( q4(0,1,k) * q4(0,2,k) &
     &                     + q4(0,3,k) * q4(0,4,k) )
         a(2,2) =   2.d0 * ( q4(0,1,k) * q4(0,1,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(2,3) =   2.d0 * ( q4(0,2,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,3,k) )
         a(3,1) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     - q4(0,1,k) * q4(0,4,k) )
         a(3,2) = - 2.d0 * ( q4(0,1,k) * q4(0,3,k) &
     &                     + q4(0,2,k) * q4(0,4,k) )
         a(3,3) =   2.d0 * ( q4(0,3,k) * q4(0,3,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            x(m,1) = r5(0,1,k) + a(1,1) * x_mol(l,k) &
     &                         + a(2,1) * y_mol(l,k) &
     &                         + a(3,1) * z_mol(l,k)
            y(m,1) = r5(0,2,k) + a(1,2) * x_mol(l,k) &
     &                         + a(2,2) * y_mol(l,k) &
     &                         + a(3,2) * z_mol(l,k)
            z(m,1) = r5(0,3,k) + a(1,3) * x_mol(l,k) &
     &                         + a(2,3) * y_mol(l,k) &
     &                         + a(3,3) * z_mol(l,k)

         end do

      end do

!-----------------------------------------------------------------------
!     //   transfer from molecular to laboratory frame: 5-d molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5

         a(3,1) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     - q4(0,1,k) * q4(0,4,k) )
         a(3,2) = - 2.d0 * ( q4(0,1,k) * q4(0,3,k) &
     &                     + q4(0,2,k) * q4(0,4,k) )
         a(3,3) =   2.d0 * ( q4(0,3,k) * q4(0,3,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            x(m,1) = r5(0,1,k) + a(3,1) * z_mol(l,k)
            y(m,1) = r5(0,2,k) + a(3,2) * z_mol(l,k)
            z(m,1) = r5(0,3,k) + a(3,3) * z_mol(l,k)

         end do

      end do

!-----------------------------------------------------------------------
!     //   transfer from molecular to laboratory frame: 3-d molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + nmol_5 + 1, nmol_6 + nmol_5 + nmol_3

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            x(m,1) = r5(0,1,k)
            y(m,1) = r5(0,2,k)
            z(m,1) = r5(0,3,k)

         end do

      end do

!-----------------------------------------------------------------------
!     //   atomic coordinates in laboratory axis
!-----------------------------------------------------------------------

      ux(:,1) = x(:,1)
      uy(:,1) = y(:,1)
      uz(:,1) = z(:,1)

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine torque_rotor
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : fx, fy, fz

      use rotor_variables, only : &
     &   x_mol, y_mol, z_mol, q4, list_atom_mol, &
     &   fnx, fny, fnz, natom_per_mol, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: j, k, l, m

!     //   real numbers
      real(8) :: a(3,3), fm(3)

!-----------------------------------------------------------------------
!     //   torque: 6-dimensional molecule
!-----------------------------------------------------------------------

      do k = 1, nmol_6

         a(1,1) =   2.d0 * ( q4(0,2,k) * q4(0,2,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(1,2) =   2.d0 * ( q4(0,3,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,2,k) )
         a(1,3) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     + q4(0,1,k) * q4(0,4,k) )
         a(2,1) = - 2.d0 * ( q4(0,1,k) * q4(0,2,k) &
     &                     + q4(0,3,k) * q4(0,4,k) )
         a(2,2) =   2.d0 * ( q4(0,1,k) * q4(0,1,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(2,3) =   2.d0 * ( q4(0,2,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,3,k) )
         a(3,1) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     - q4(0,1,k) * q4(0,4,k) )
         a(3,2) = - 2.d0 * ( q4(0,1,k) * q4(0,3,k) &
     &                     + q4(0,2,k) * q4(0,4,k) )
         a(3,3) =   2.d0 * ( q4(0,3,k) * q4(0,3,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0

         fnx(k) = 0.d0
         fny(k) = 0.d0
         fnz(k) = 0.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            do j = 1, 3

               fm(j) = fx(m,1) * a(j,1) &
     &               + fy(m,1) * a(j,2) &
     &               + fz(m,1) * a(j,3)

            end do

            fnx(k) = fnx(k) + y_mol(l,k) * fm(3) - z_mol(l,k) * fm(2)
            fny(k) = fny(k) + z_mol(l,k) * fm(1) - x_mol(l,k) * fm(3)
            fnz(k) = fnz(k) + x_mol(l,k) * fm(2) - y_mol(l,k) * fm(1)

         end do

      end do

!-----------------------------------------------------------------------
!     //   torque: 5-dimensional molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5

         a(1,1) =   2.d0 * ( q4(0,2,k) * q4(0,2,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(1,2) =   2.d0 * ( q4(0,3,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,2,k) )
         a(1,3) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     + q4(0,1,k) * q4(0,4,k) )
         a(2,1) = - 2.d0 * ( q4(0,1,k) * q4(0,2,k) &
     &                     + q4(0,3,k) * q4(0,4,k) )
         a(2,2) =   2.d0 * ( q4(0,1,k) * q4(0,1,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(2,3) =   2.d0 * ( q4(0,2,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,3,k) )

         fnx(k) = 0.d0
         fny(k) = 0.d0
         fnz(k) = 0.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            do j = 1, 2

               fm(j) = fx(m,1) * a(j,1) &
     &               + fy(m,1) * a(j,2) &
     &               + fz(m,1) * a(j,3)

            end do

            fnx(k) = fnx(k) - z_mol(l,k) * fm(2)
            fny(k) = fny(k) + z_mol(l,k) * fm(1)

         end do

      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_bath_pre_nvt
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : dt

      use rotor_variables, only : s5, zeta

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     //   predictor of thermostat
!-----------------------------------------------------------------------

      s5(0) =       s5(0) +        s5(1) +         s5(2) +         s5(3) &
     &     +        s5(4) +        s5(5)
      s5(1) =       s5(1) + 2.d0 * s5(2) +  3.d0 * s5(3) +  4.d0 * s5(4) &
     &     + 5.d0 * s5(5)
      s5(2) =       s5(2) + 3.d0 * s5(3) +  6.d0 * s5(4) + 10.d0 * s5(5)
      s5(3) =       s5(3) + 4.d0 * s5(4) + 10.d0 * s5(5)
      s5(4) =       s5(4) + 5.d0 * s5(5)

!-----------------------------------------------------------------------
!     //   zeta value of thermostat
!-----------------------------------------------------------------------

      zeta = s5(1) / dt / s5(0)

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_center_of_mass_pre
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use rotor_variables, only : r5, nmol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: j, k

!-----------------------------------------------------------------------
!     //   predictor of translation
!-----------------------------------------------------------------------

      do k = 1, nmol
      do j = 1, 3

         r5(0,j,k) =    r5(0,j,k) +        r5(1,j,k) +         r5(2,j,k) &
     &       +          r5(3,j,k) +        r5(4,j,k) +         r5(5,j,k)

         r5(1,j,k) =    r5(1,j,k) + 2.d0 * r5(2,j,k) +  3.d0 * r5(3,j,k) &
     &        +  4.d0 * r5(4,j,k) + 5.d0 * r5(5,j,k)

         r5(2,j,k) =    r5(2,j,k) + 3.d0 * r5(3,j,k) +  6.d0 * r5(4,j,k) &
     &        + 10.d0 * r5(5,j,k)

         r5(3,j,k) =    r5(3,j,k) + 4.d0 * r5(4,j,k) + 10.d0 * r5(5,j,k)

         r5(4,j,k) =    r5(4,j,k) + 5.d0 * r5(5,j,k)

      end do
      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_rotation_pre
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use rotor_variables, only : o4, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: j, k

!-----------------------------------------------------------------------
!     //   predictor of angular velocity: 6-dimensional molecule
!-----------------------------------------------------------------------

      do k = 1, nmol_6
      do j = 1, 3

         o4(0,j,k) =   o4(0,j,k) +        o4(1,j,k) +        o4(2,j,k) &
     &        +        o4(3,j,k) +        o4(4,j,k)

         o4(1,j,k) =   o4(1,j,k) + 2.d0 * o4(2,j,k) + 3.d0 * o4(3,j,k) &
     &        + 4.d0 * o4(4,j,k)

         o4(2,j,k) =   o4(2,j,k) + 3.d0 * o4(3,j,k) + 6.d0 * o4(4,j,k)

         o4(3,j,k) =   o4(3,j,k) + 4.d0 * o4(4,j,k)

      end do
      end do

!-----------------------------------------------------------------------
!     //   predictor of angular velocity: 5-dimensional molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5
      do j = 1, 2

         o4(0,j,k) =   o4(0,j,k) +        o4(1,j,k) +        o4(2,j,k) &
     &        +        o4(3,j,k) +        o4(4,j,k)

         o4(1,j,k) =   o4(1,j,k) + 2.d0 * o4(2,j,k) + 3.d0 * o4(3,j,k) &
     &        + 4.d0 * o4(4,j,k)

         o4(2,j,k) =   o4(2,j,k) + 3.d0 * o4(3,j,k) + 6.d0 * o4(4,j,k)

         o4(3,j,k) =   o4(3,j,k) + 4.d0 * o4(4,j,k)

      end do
      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_quaternion_pre
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use rotor_variables, only : q4, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: j, k

!-----------------------------------------------------------------------
!     //   predictor of quaternion
!-----------------------------------------------------------------------

      do k = 1, nmol_6 + nmol_5
      do j = 1, 4

         q4(0,j,k) =   q4(0,j,k) +        q4(1,j,k) +        q4(2,j,k) &
     &        +        q4(3,j,k) +        q4(4,j,k)

         q4(1,j,k) =   q4(1,j,k) + 2.d0 * q4(2,j,k) + 3.d0 * q4(3,j,k) &
     &        + 4.d0 * q4(4,j,k)

         q4(2,j,k) =   q4(2,j,k) + 3.d0 * q4(3,j,k) + 6.d0 * q4(4,j,k)

         q4(3,j,k) =   q4(3,j,k) + 4.d0 * q4(4,j,k)

      end do
      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_center_of_mass_cor_nvt
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : dt

      use rotor_variables, only : &
     &   r5, fgx, fgy, fgz, zeta, dt_gear, physmass_mol, a5_gear, nmol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, k

!     //   real numbers
      real(8) :: corx, cory, corz

!-----------------------------------------------------------------------
!     //   corrector of translation: forces
!-----------------------------------------------------------------------

      do k = 1, nmol

         corx = fgx(k) * dt_gear(2) / physmass_mol(k) - r5(2,1,k) &
     &        - zeta * r5(1,1,k) * 0.5d0 * dt

         cory = fgy(k) * dt_gear(2) / physmass_mol(k) - r5(2,2,k) &
     &        - zeta * r5(1,2,k) * 0.5d0 * dt

         corz = fgz(k) * dt_gear(2) / physmass_mol(k) - r5(2,3,k) &
     &        - zeta * r5(1,3,k) * 0.5d0 * dt

         do i = 0, 5

            r5(i,1,k) = r5(i,1,k) + a5_gear(i) * corx
            r5(i,2,k) = r5(i,2,k) + a5_gear(i) * cory
            r5(i,3,k) = r5(i,3,k) + a5_gear(i) * corz

         end do

      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_center_of_mass_cor_nve
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use rotor_variables, only : &
     &   r5, fgx, fgy, fgz, dt_gear, physmass_mol, a5_gear, nmol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, k

!     //   real numbers
      real(8) :: corx, cory, corz

!-----------------------------------------------------------------------
!     //   corrector of translation: forces
!-----------------------------------------------------------------------

      do k = 1, nmol

         corx = fgx(k) * dt_gear(2) / physmass_mol(k) - r5(2,1,k)
         cory = fgy(k) * dt_gear(2) / physmass_mol(k) - r5(2,2,k)
         corz = fgz(k) * dt_gear(2) / physmass_mol(k) - r5(2,3,k)

         do i = 0, 5

            r5(i,1,k) = r5(i,1,k) + a5_gear(i) * corx
            r5(i,2,k) = r5(i,2,k) + a5_gear(i) * cory
            r5(i,3,k) = r5(i,3,k) + a5_gear(i) * corz

         end do

      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_rotation_cor_nvt
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : dt

      use rotor_variables, only : &
     &   o4, fnx, fny, fnz, pmi, zeta, a4_gear, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, k

!     //   real numbers
      real(8) :: corx, cory, corz

!-----------------------------------------------------------------------
!     //   corrector of angular velocity: 6-dimensional molecule
!-----------------------------------------------------------------------

      do k = 1, nmol_6

         corx = &
     &      + fnx(k) * dt * dt / pmi(1,k) - o4(1,1,k) &
     &      + ( pmi(2,k) - pmi(3,k) ) / pmi(1,k) * o4(0,2,k) * o4(0,3,k) &
     &      - zeta * o4(0,1,k) * dt

         cory = &
     &      + fny(k) * dt * dt / pmi(2,k) - o4(1,2,k) &
     &      + ( pmi(3,k) - pmi(1,k) ) / pmi(2,k) * o4(0,3,k) * o4(0,1,k) &
     &      - zeta * o4(0,2,k) * dt

         corz = &
     &      + fnz(k) * dt * dt / pmi(3,k) - o4(1,3,k) &
     &      + ( pmi(1,k) - pmi(2,k) ) / pmi(3,k) * o4(0,1,k) * o4(0,2,k) &
     &      - zeta * o4(0,3,k) * dt

         do i = 0, 4

            o4(i,1,k) = o4(i,1,k) + a4_gear(i) * corx
            o4(i,2,k) = o4(i,2,k) + a4_gear(i) * cory
            o4(i,3,k) = o4(i,3,k) + a4_gear(i) * corz

         end do

      end do

!-----------------------------------------------------------------------
!     //   corrector of angular velocity: 5-dimensional molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5

         corx = fnx(k) * dt * dt / pmi(1,k) - o4(1,1,k) &
     &        - zeta * o4(0,1,k) * dt

         cory = fny(k) * dt * dt / pmi(2,k) - o4(1,2,k) &
     &        - zeta * o4(0,2,k) * dt

         do i = 0, 4

            o4(i,1,k) = o4(i,1,k) + a4_gear(i) * corx
            o4(i,2,k) = o4(i,2,k) + a4_gear(i) * cory

         end do

      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_rotation_cor_nve
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : dt

      use rotor_variables, only : &
     &   o4, fnx, fny, fnz, pmi, a4_gear, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, k

!     //   real numbers
      real(8) :: corx, cory, corz

!-----------------------------------------------------------------------
!     //   corrector of angular velocity: 6-dimensional molecule
!-----------------------------------------------------------------------

      do k = 1, nmol_6

         corx = &
     &      + fnx(k) * dt * dt / pmi(1,k) - o4(1,1,k) &
     &      + ( pmi(2,k) - pmi(3,k) ) / pmi(1,k) * o4(0,2,k) * o4(0,3,k)

         cory = &
     &      + fny(k) * dt * dt / pmi(2,k) - o4(1,2,k) &
     &      + ( pmi(3,k) - pmi(1,k) ) / pmi(2,k) * o4(0,3,k) * o4(0,1,k)

         corz = &
     &      + fnz(k) * dt * dt / pmi(3,k) - o4(1,3,k) &
     &      + ( pmi(1,k) - pmi(2,k) ) / pmi(3,k) * o4(0,1,k) * o4(0,2,k)

         do i = 0, 4

            o4(i,1,k) = o4(i,1,k) + a4_gear(i) * corx
            o4(i,2,k) = o4(i,2,k) + a4_gear(i) * cory
            o4(i,3,k) = o4(i,3,k) + a4_gear(i) * corz

         end do

      end do

!-----------------------------------------------------------------------
!     //   corrector of angular velocity: 5-dimensional molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5

         corx = fnx(k) * dt * dt / pmi(1,k) - o4(1,1,k)
         cory = fny(k) * dt * dt / pmi(2,k) - o4(1,2,k)

         do i = 0, 4

            o4(i,1,k) = o4(i,1,k) + a4_gear(i) * corx
            o4(i,2,k) = o4(i,2,k) + a4_gear(i) * cory

         end do

      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_quaternion_cor
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use rotor_variables, only : q4, qdot, a4_gear, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, j, k

!     //   real numbers
      real(8) :: cor, qt

!-----------------------------------------------------------------------
!     //   corrector of quaternion
!-----------------------------------------------------------------------

      do j = 1, 4
      do k = 1, nmol_6 + nmol_5

         cor = qdot(j,k) - q4(1,j,k)

         do i = 0, 4

            q4(i,j,k) = q4(i,j,k) + a4_gear(i) * cor

         end do

      end do
      end do

!-----------------------------------------------------------------------
!     //   normalize quaternion
!-----------------------------------------------------------------------

      do k = 1, nmol_6 + nmol_5

         qt = q4(0,1,k) * q4(0,1,k) + q4(0,2,k) * q4(0,2,k) &
     &      + q4(0,3,k) * q4(0,3,k) + q4(0,4,k) * q4(0,4,k)

         qt = sqrt(qt)

         do j = 1, 4

            q4(0,j,k) = q4(0,j,k) / qt

         end do

      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine pbc_rotor_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   box, boxinv, iboundary

      use rotor_variables, only : &
     &   r5, nmol

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: k

      real(8) :: ac, bc, cc, da, db, dc, dx, dy, dz, xc, yc, zc

!-----------------------------------------------------------------------
!     /*   periodic boundary                                          */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 1 ) then

         do k = 1, nmol

            xc = r5(0,1,k)
            yc = r5(0,2,k)
            zc = r5(0,3,k)

!           /*   shift according to molecular center of mass   */

            ac = boxinv(1,1)*xc + boxinv(1,2)*yc + boxinv(1,3)*zc
            bc = boxinv(2,1)*xc + boxinv(2,2)*yc + boxinv(2,3)*zc
            cc = boxinv(3,1)*xc + boxinv(3,2)*yc + boxinv(3,3)*zc

            da = - dble(nint(ac-0.5d0))
            db = - dble(nint(bc-0.5d0))
            dc = - dble(nint(cc-0.5d0))

            dx = box(1,1)*da + box(1,2)*db + box(1,3)*dc
            dy = box(2,1)*da + box(2,2)*db + box(2,3)*dc
            dz = box(3,1)*da + box(3,2)*db + box(3,3)*dc

!           /*   shift to the range 0 < x < box   */

            r5(0,1,k) = xc + dx
            r5(0,2,k) = yc + dy
            r5(0,3,k) = zc + dz

         end do

      end if

      return
      end





!***********************************************************************
      subroutine trans_gear_nvt( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : dt

      use rotor_variables, only : a4_gear, a5_gear, dt_gear, &
     &   s5, r5, o4, q4, zeta, nmol, nmol_5, nmol_6

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, j, k, ioption

!     //   integers
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     //   initial setting of gear parameter
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        //   gear parameter

         a4_gear(0) =  251.d0 / 720.d0
         a4_gear(1) =  1.d0
         a4_gear(2) =  11.d0 / 12.d0
         a4_gear(3) =  1.d0 / 3.d0
         a4_gear(4) =  1.d0 / 24.d0

         a5_gear(0) =  3.d0 / 16.d0
         a5_gear(1) =  251.d0 / 360.d0
         a5_gear(2) =  1.d0
         a5_gear(3) =  11.d0 / 18.d0
         a5_gear(4) =  1.d0 / 6.d0
         a5_gear(5) =  1.d0 / 60.d0

!        //   powers of dt

         dt_gear(1) = dt
         dt_gear(2) = dt**2.d0 / 2.d0
         dt_gear(3) = dt**3.d0 / 6.d0
         dt_gear(4) = dt**4.d0 / 24.d0
         dt_gear(5) = dt**5.d0 / 120.d0

         iset = 1

      end if

!-----------------------------------------------------------------------
!     //   forward transform
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!-----------------------------------------------------------------------
!        //   time derivative series of translational coordinate
!-----------------------------------------------------------------------

         do i = 1, 5
         do j = 1, 3
         do k = 1, nmol
            r5(i,j,k) = r5(i,j,k) * dt_gear(i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: 6-d molecules
!-----------------------------------------------------------------------

         do k = 1, nmol_6
         do j = 1, 3

            o4(0,j,k) = o4(0,j,k) * dt

            do i = 1, 4
               o4(i,j,k) = o4(i,j,k) * dt * dt_gear(i)
            end do

         end do
         end do

!-----------------------------------------------------------------------
!     //   time derivative series of rotation: 5-d molecules
!-----------------------------------------------------------------------

         do k = nmol_6 + 1, nmol_6 + nmol_5
         do j = 1, 2

            o4(0,j,k) = o4(0,j,k) * dt

            do i = 1, 4
               o4(i,j,k) = o4(i,j,k) * dt * dt_gear(i)
            end do

         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: quaternion
!-----------------------------------------------------------------------

         do i = 1, 4
         do j = 1, 4
         do k = 1, nmol_6 + nmol_5

            q4(i,j,k) = q4(i,j,k) * dt_gear(i)

         end do
         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of thermostat
!-----------------------------------------------------------------------

         s5(1) = s5(1) * dt_gear(1)
         s5(2) = s5(2) * dt_gear(2)
         s5(3) = s5(3) * dt_gear(3)
         s5(4) = s5(4) * dt_gear(4)
         s5(5) = s5(5) * dt_gear(5)

!-----------------------------------------------------------------------
!        //   zeta value
!-----------------------------------------------------------------------

         zeta = s5(1) / dt / s5(0)

      end if

!-----------------------------------------------------------------------
!     //   back transform
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!        //   time derivative series of translational coordinate
!-----------------------------------------------------------------------

         do i = 1, 5
         do j = 1, 3
         do k = 1, nmol
            r5(i,j,k) = r5(i,j,k) / dt_gear(i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: 6-d molecules
!-----------------------------------------------------------------------

         do k = 1, nmol_6
         do j = 1, 3

            o4(0,j,k) = o4(0,j,k) / dt

            do i = 1, 4
               o4(i,j,k) = o4(i,j,k) / dt / dt_gear(i)
            end do

         end do
         end do

!-----------------------------------------------------------------------
!     //   time derivative series of rotation: 5-d molecules
!-----------------------------------------------------------------------

         do k = nmol_6 + 1, nmol_6 + nmol_5
         do j = 1, 2

            o4(0,j,k) = o4(0,j,k) / dt

            do i = 1, 4
               o4(i,j,k) = o4(i,j,k) / dt / dt_gear(i)
            end do

         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: quaternion
!-----------------------------------------------------------------------

         do i = 1, 4
         do j = 1, 4
         do k = 1, nmol_6 + nmol_5

            q4(i,j,k) = q4(i,j,k) / dt_gear(i)

         end do
         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of thermostat
!-----------------------------------------------------------------------

         s5(1) = s5(1) / dt_gear(1)
         s5(2) = s5(2) / dt_gear(2)
         s5(3) = s5(3) / dt_gear(3)
         s5(4) = s5(4) / dt_gear(4)
         s5(5) = s5(5) / dt_gear(5)

!-----------------------------------------------------------------------
!        //   zeta value
!-----------------------------------------------------------------------

         zeta = s5(1) / s5(0)

      end if

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine trans_gear_nve( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : dt

      use rotor_variables, only : a4_gear, a5_gear, dt_gear, &
     &   r5, o4, q4, nmol, nmol_5, nmol_6

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, j, k, ioption

!     //   integers
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     //   initial setting of gear parameter
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        //   gear parameter

         a4_gear(0) =  251.d0 / 720.d0
         a4_gear(1) =  1.d0
         a4_gear(2) =  11.d0 / 12.d0
         a4_gear(3) =  1.d0 / 3.d0
         a4_gear(4) =  1.d0 / 24.d0

         a5_gear(0) =  3.d0 / 16.d0
         a5_gear(1) =  251.d0 / 360.d0
         a5_gear(2) =  1.d0
         a5_gear(3) =  11.d0 / 18.d0
         a5_gear(4) =  1.d0 / 6.d0
         a5_gear(5) =  1.d0 / 60.d0

!        //   powers of dt

         dt_gear(1) = dt
         dt_gear(2) = dt**2.d0 / 2.d0
         dt_gear(3) = dt**3.d0 / 6.d0
         dt_gear(4) = dt**4.d0 / 24.d0
         dt_gear(5) = dt**5.d0 / 120.d0

         iset = 1

      end if

!-----------------------------------------------------------------------
!     //   forward transform
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!-----------------------------------------------------------------------
!        //   gear's parameter
!-----------------------------------------------------------------------

         a4_gear(0) =  251.d0 / 720.d0
         a4_gear(1) =  1.d0
         a4_gear(2) =  11.d0 / 12.d0
         a4_gear(3) =  1.d0 / 3.d0
         a4_gear(4) =  1.d0 / 24.d0

         a5_gear(0) =  3.d0 / 16.d0
         a5_gear(1) =  251.d0 / 360.d0
         a5_gear(2) =  1.d0
         a5_gear(3) =  11.d0 / 18.d0
         a5_gear(4) =  1.d0 / 6.d0
         a5_gear(5) =  1.d0 / 60.d0

!-----------------------------------------------------------------------
!        //   powers of dt
!-----------------------------------------------------------------------

         dt_gear(1) = dt
         dt_gear(2) = dt**2.d0 / 2.d0
         dt_gear(3) = dt**3.d0 / 6.d0
         dt_gear(4) = dt**4.d0 / 24.d0
         dt_gear(5) = dt**5.d0 / 120.d0

!-----------------------------------------------------------------------
!        //   time derivative series of translational coordinate
!-----------------------------------------------------------------------

         do i = 1, 5
         do j = 1, 3
         do k = 1, nmol
            r5(i,j,k) = r5(i,j,k) * dt_gear(i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: 6-d molecules
!-----------------------------------------------------------------------

         do k = 1, nmol_6
         do j = 1, 3

            o4(0,j,k) = o4(0,j,k) * dt

            do i = 1, 4
               o4(i,j,k) = o4(i,j,k) * dt * dt_gear(i)
            end do

         end do
         end do

!-----------------------------------------------------------------------
!     //   time derivative series of rotation: 5-d molecules
!-----------------------------------------------------------------------

         do k = nmol_6 + 1, nmol_6 + nmol_5
         do j = 1, 2

            o4(0,j,k) = o4(0,j,k) * dt

            do i = 1, 4
               o4(i,j,k) = o4(i,j,k) * dt * dt_gear(i)
            end do

         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: quaternion
!-----------------------------------------------------------------------

         do i = 1, 4
         do j = 1, 4
         do k = 1, nmol_6 + nmol_5

            q4(i,j,k) = q4(i,j,k) * dt_gear(i)

         end do
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     //   back transform
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!-----------------------------------------------------------------------
!        //   time derivative series of translational coordinate
!-----------------------------------------------------------------------

         do i = 1, 5
         do j = 1, 3
         do k = 1, nmol
            r5(i,j,k) = r5(i,j,k) / dt_gear(i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: 6-d molecules
!-----------------------------------------------------------------------

         do k = 1, nmol_6
         do j = 1, 3

            o4(0,j,k) = o4(0,j,k) / dt

            do i = 1, 4
               o4(i,j,k) = o4(i,j,k) / dt / dt_gear(i)
            end do

         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: 5-d molecules
!-----------------------------------------------------------------------

         do k = nmol_6 + 1, nmol_6 + nmol_5
         do j = 1, 2

            o4(0,j,k) = o4(0,j,k) / dt

            do i = 1, 4
               o4(i,j,k) = o4(i,j,k) / dt / dt_gear(i)
            end do

         end do
         end do

!-----------------------------------------------------------------------
!        //   time derivative series of rotation: quaternion
!-----------------------------------------------------------------------

         do i = 1, 4
         do j = 1, 4
         do k = 1, nmol_6 + nmol_5

            q4(i,j,k) = q4(i,j,k) / dt_gear(i)

         end do
         end do
         end do


      end if

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine pc_prep_nve
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : dt

      use rotor_variables, only : a4_gear, a5_gear, dt_gear, &
     &   r5, o4, q4, nmol, nmol_5, nmol_6

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, j, k

!-----------------------------------------------------------------------
!     //   gear's parameter
!-----------------------------------------------------------------------

      a4_gear(0) =  251.d0 / 720.d0
      a4_gear(1) =  1.d0
      a4_gear(2) =  11.d0 / 12.d0
      a4_gear(3) =  1.d0 / 3.d0
      a4_gear(4) =  1.d0 / 24.d0

      a5_gear(0) =  3.d0 / 16.d0
      a5_gear(1) =  251.d0 / 360.d0
      a5_gear(2) =  1.d0
      a5_gear(3) =  11.d0 / 18.d0
      a5_gear(4) =  1.d0 / 6.d0
      a5_gear(5) =  1.d0 / 60.d0

!-----------------------------------------------------------------------
!     //   powers of dt
!-----------------------------------------------------------------------

      dt_gear(1) = dt
      dt_gear(2) = dt**2.d0 / 2.d0
      dt_gear(3) = dt**3.d0 / 6.d0
      dt_gear(4) = dt**4.d0 / 24.d0
      dt_gear(5) = dt**5.d0 / 120.d0

!-----------------------------------------------------------------------
!     //   time derivative series of translational coordinate
!-----------------------------------------------------------------------

      do i = 1, 5
      do j = 1, 3
      do k = 1, nmol
         r5(i,j,k) = dt_gear(i) * r5(i,j,k)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     //   time derivative series of rotation: 6-d molecules
!-----------------------------------------------------------------------

      do k = 1, nmol_6
      do j = 1, 3

         o4(0,j,k) = o4(0,j,k) * dt

         do i = 1, 4
            o4(i,j,k) = dt_gear(i) * o4(i,j,k) * dt
         end do

      end do
      end do

!-----------------------------------------------------------------------
!     //   time derivative series of rotation: 5-d molecules
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5
      do j = 1, 2

         o4(0,j,k) = o4(0,j,k) * dt

         do i = 1, 4
            o4(i,j,k) = dt_gear(i) * o4(i,j,k) * dt
         end do

      end do
      end do

!-----------------------------------------------------------------------
!     //   time derivative series of rotation: quaternion
!-----------------------------------------------------------------------

      do i = 1, 4
      do j = 1, 4
      do k = 1, nmol_6 + nmol_5
         q4(i,j,k) = dt_gear(i) * q4(i,j,k)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine moment_of_inertia
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : physmass

      use rotor_variables, only : &
     &   pmi, x_comp, y_comp, z_comp, &
     &   ncomp_6, ncomp_5, nmol_comp, natom_per_comp, list_atom_mol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, j, k, m, l

!-----------------------------------------------------------------------
!     //   moment of inertia: 6-dimensional molecules
!-----------------------------------------------------------------------

      j = 0

      do m = 1, ncomp_6
      do k = 1, nmol_comp(m)

         j = j + 1

         pmi(:,j) = 0.d0

         do l = 1, natom_per_comp(m)

            i = list_atom_mol(l,j)

            pmi(1,j) = pmi(1,j) &
     &                + physmass(i) * ( y_comp(l,m) * y_comp(l,m) &
     &                                + z_comp(l,m) * z_comp(l,m) )

            pmi(2,j) = pmi(2,j) &
     &                + physmass(i) * ( z_comp(l,m) * z_comp(l,m) &
     &                                + x_comp(l,m) * x_comp(l,m) )

            pmi(3,j) = pmi(3,j) &
     &                + physmass(i) * ( x_comp(l,m) * x_comp(l,m) &
     &                                + y_comp(l,m) * y_comp(l,m) )

         end do

      end do
      end do

!-----------------------------------------------------------------------
!     //   moment of inertia: 5-dimensional molecules
!-----------------------------------------------------------------------

      do m = ncomp_6 + 1, ncomp_6 + ncomp_5
      do k = 1, nmol_comp(m)

         j = j + 1

         pmi(:,j) = 0.d0

         do l = 1, natom_per_comp(m)

            i = list_atom_mol(l,j)

            pmi(1,j) = pmi(1,j) &
     &                + physmass(i) * z_comp(l,m) * z_comp(l,m)

            pmi(2,j) = pmi(2,j) &
     &                + physmass(i) * z_comp(l,m) * z_comp(l,m)

         end do

      end do
      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine update_bath_cor_nvt
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : temperature, temp, dt, boltz

      use rotor_variables, only : s5, zeta, dt_gear, &
     &   qmass_rotor, a5_gear, nmol_6, nmol_5, nmol_3

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      real(8) :: zt, gmt, corz

!-----------------------------------------------------------------------
!     //   corrector for thermostat
!-----------------------------------------------------------------------

      zt = ( 6.d0*dble(nmol_6) + 5.d0*dble(nmol_5) + 3.d0*dble(nmol_3) ) &
     &     * boltz * ( temp - temperature ) / qmass_rotor

      gmt = s5(1) * 0.5d0 * dt * zeta + s5(0) * zt * dt_gear(2)

      corz = gmt - s5(2)

      s5(0) = s5(0) + a5_gear(0) * corz
      s5(1) = s5(1) + a5_gear(1) * corz
      s5(2) = s5(2) + a5_gear(2) * corz
      s5(3) = s5(3) + a5_gear(3) * corz
      s5(4) = s5(4) + a5_gear(4) * corz
      s5(5) = s5(5) + a5_gear(5) * corz

      zeta = s5(1) / dt / s5(0)

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine standard_rotor_nvt_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temperature, temp, hamiltonian, boltz, ekin, hamiltonian_sys, &
     &   potential, istep, iprint_std, ndof, iounit, iounit_std, &
     &   char_date, myrank

      use rotor_variables, only : &
     &   s5, ebath_rotor, qmass_rotor, zeta

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: itest

!     //   integers
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( ( myrank .eq. 0 ) .and. ( iprint_std .gt. 0 ) ) then

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

         end if

      end if

!-----------------------------------------------------------------------
!     //   energies
!-----------------------------------------------------------------------

!     //   energy of the system
      hamiltonian_sys = ekin + potential

!     //   energy of thermostat
      ebath_rotor = 0.5d0 * qmass_rotor * zeta * zeta &
     &      + dble(ndof) * boltz * temperature * log( s5(0) )

!     //   total energy
      hamiltonian = hamiltonian_sys + ebath_rotor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( ( myrank .eq. 0 ) .and. ( iprint_std .gt. 0 ) ) then
      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   output   */
         write( iounit_std, '(i8,2f16.8,f10.2,2x,a26)' ) &
     &      istep, hamiltonian, potential, temp, char_date

         write(         6,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date

      end if
      end if

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine standard_rotor_nve_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, hamiltonian, boltz, ekin, hamiltonian_sys, &
     &   potential, istep, iprint_std, iounit, iounit_std, &
     &   char_date, myrank

      use rotor_variables, only : &
     &   ebath_rotor

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: itest

!     //   integers
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( ( myrank .eq. 0 ) .and. ( iprint_std .gt. 0 ) ) then

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write(iounit_std,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(iounit_std,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(iounit_std,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write(         6,'(a)') &
     &      '====================================================' // &
     &      '=========================='
            write(         6,'(a)') &
     &      '    step     energy [au]  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write(         6,'(a)') &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

         end if

         end if

      end if

!-----------------------------------------------------------------------
!     //   energies
!-----------------------------------------------------------------------

!     //   energy of the system
      hamiltonian_sys = ekin + potential

!     //   energy of thermostat
      ebath_rotor = 0.d0

!     //   total energy
      hamiltonian = hamiltonian_sys + ebath_rotor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( ( myrank .eq. 0 ) .and. ( iprint_std .gt. 0 ) ) then
      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   output   */
         write( iounit_std, '(i8,2f16.8,f10.2,2x,a26)' ) &
     &      istep, hamiltonian, potential, temp, char_date

         write(         6,'(i8,2f16.8,f10.2,2x,a26)') &
     &      istep, hamiltonian, potential, temp, char_date

      end if
      end if

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine kinetic_energy_rotor
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, physmass, vx, vy, vz, dt, boltz, &
     &   ndof, natom

      use rotor_variables, only : &
     &   temp_tra, temp_rot, physmass_mol, pmi, r5, &
     &   o4, ekin_tra, ekin_rot, ekin_2, &
     &   ncomp_6, ncomp_5, &
     &   ncomp_3, nmol_comp

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: k, m, j

!-----------------------------------------------------------------------
!     //   initialization
!-----------------------------------------------------------------------

!     //   total kinetic energy
      ekin = 0.d0

!-----------------------------------------------------------------------
!     //   kinetic energy and instantaneous temperature: 6-d molecules
!-----------------------------------------------------------------------

!     //   molecular number
      j = 0

!     //   loop of 6-d components
      do m = 1, ncomp_6

!        //   translational kinetic energy
         ekin_tra(m) = 0.d0

!        //   rotational kinetic energy
         ekin_rot(m) = 0.d0

!        //   loop of 6-d molecules
         do k = 1, nmol_comp(m)

!           //   molecular number
            j = j + 1

!           //   translational kinetic energy

            ekin_tra(m) = ekin_tra(m) &
     &                  + physmass_mol(j) &
     &                      * ( r5(1,1,j) * r5(1,1,j) &
     &                        + r5(1,2,j) * r5(1,2,j) &
     &                        + r5(1,3,j) * r5(1,3,j) )

!           //   rotational kinetic energy

            ekin_rot(m) = ekin_rot(m) &
     &                  + pmi(1,j) * o4(0,1,j) * o4(0,1,j) &
     &                  + pmi(2,j) * o4(0,2,j) * o4(0,2,j) &
     &                  + pmi(3,j) * o4(0,3,j) * o4(0,3,j)

!        //   loop of 6-d molecules
         end do

!        //   translational and rotational kinetic energies

         ekin_tra(m) = 0.5d0 * ekin_tra(m) / ( dt * dt )
         ekin_rot(m) = 0.5d0 * ekin_rot(m) / ( dt * dt )

!        //   total kinetic energy
         ekin = ekin + ekin_tra(m) + ekin_rot(m)

!        //   translational and rotational temperatures

         temp_tra(m) = ekin_tra(m) &
     &               / ( 1.5d0 * boltz * dble(nmol_comp(m)) )

         temp_rot(m) = ekin_rot(m) &
     &               / ( 1.5d0 * boltz * dble(nmol_comp(m)) )

!     //   loop of 6-d components
      end do

!-----------------------------------------------------------------------
!     //   kinetic energy and instantaneous temperature: 5-d molecules
!-----------------------------------------------------------------------

!     //   loop of 5-d components
      do m = ncomp_6 + 1, ncomp_6 + ncomp_5

!        //   translational kinetic energy
         ekin_tra(m) = 0.d0

!        //   rotational kinetic energy
         ekin_rot(m) = 0.d0

!        //   loop of 5-d molecules
         do k = 1, nmol_comp(m)

!           //   molecular number
            j = j + 1

!           //   translational kinetic energy

            ekin_tra(m) = ekin_tra(m) &
     &                  + physmass_mol(j) &
     &                      * ( r5(1,1,j) * r5(1,1,j) &
     &                        + r5(1,2,j) * r5(1,2,j) &
     &                        + r5(1,3,j) * r5(1,3,j) )

!           //   rotational kinetic energy

            ekin_rot(m) = ekin_rot(m) &
     &                  + pmi(1,j) * o4(0,1,j) * o4(0,1,j) &
     &                  + pmi(2,j) * o4(0,2,j) * o4(0,2,j)

!        //   loop of 5-d molecules
         end do

!        //   translational and rotational kinetic energies

         ekin_tra(m) = 0.5d0 * ekin_tra(m) / ( dt * dt )
         ekin_rot(m) = 0.5d0 * ekin_rot(m) / ( dt * dt )

!        //   total kinetic energy
         ekin = ekin + ekin_tra(m) + ekin_rot(m)

!        //   translational and rotational temperatures

         temp_tra(m) = ekin_tra(m) &
     &               / ( 1.5d0 * boltz * dble(nmol_comp(m)) )

         temp_rot(m) = ekin_rot(m) &
     &               / ( 1.0d0 * boltz * dble(nmol_comp(m)) )

!     //   loop of 5-d components
      end do

!     //   loop of 3-d components
      do m = ncomp_6 + ncomp_5 + 1, ncomp_6 + ncomp_5 + ncomp_3

!        //   translational kinetic energy
         ekin_tra(m) = 0.d0

!        //   rotational kinetic energy
         ekin_rot(m) = 0.d0

!        //   loop of 3-d molecules
         do k = 1, nmol_comp(m)

!           //   molecular number
            j = j + 1

!           //   translational kinetic energy

            ekin_tra(m) = ekin_tra(m) + physmass_mol(j) &
     &                      * ( r5(1,1,j) * r5(1,1,j) &
     &                        + r5(1,2,j) * r5(1,2,j) &
     &                        + r5(1,3,j) * r5(1,3,j) )

!        //   loop of 3-d molecules
         end do

!        //   translational and rotational kinetic energies
         ekin_tra(m) = 0.5d0 * ekin_tra(m) / ( dt * dt )

!        //   total kinetic energy
         ekin = ekin + ekin_tra(m)

!        //   translational and rotational temperatures

         temp_tra(m) = ekin_tra(m) &
     &               / ( 1.5d0 * boltz * dble(nmol_comp(m)) )

         temp_rot(m) = 0.d0

!     //   loop of 3-d components
      end do

!     //   instantaneous temperature
      temp = 2.d0 * ekin / boltz / ndof

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      ekin_2 = 0.d0

      do j = 1, natom

         ekin_2 = ekin_2 + physmass(j) * ( vx(j,1) * vx(j,1) &
     &                                   + vy(j,1) * vy(j,1) &
     &                                   + vz(j,1) * vz(j,1) )

      end do

      ekin_2 = 0.5d0 * ekin_2

      return
      end





!***********************************************************************
      subroutine analysis_rotor_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, nkind, natom_spec, iounit_avg, myrank

      use analysis_variables, only : &
     &   nkindpair, ikindpair, ikindpair_inv, iprint_dip, &
     &   npair_kindpair, iprint_rdf, iprint_trj, iprint_xyz, iprint_xsf, &
     &   iprint_dcd

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption, l, j1, j2, k1, k2

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   number of atomic pairs   */

         nkindpair = nkind*(nkind+1)/2

!-----------------------------------------------------------------------
!        /*   ikindpair = label for a pair of species                 */
!-----------------------------------------------------------------------

         if ( .not. allocated( ikindpair ) ) &
     &      allocate ( ikindpair(nkind,nkind) )
         if ( .not. allocated( ikindpair_inv ) ) &
     &      allocate ( ikindpair_inv(nkindpair,2) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            ikindpair(k1,k2)   = l
            ikindpair(k2,k1)   = l
            ikindpair_inv(l,1) = k1
            ikindpair_inv(l,2) = k2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   npair_kindpair = number of atom pairs                   */
!-----------------------------------------------------------------------

         if ( .not. allocated( npair_kindpair ) ) &
     &      allocate ( npair_kindpair(nkindpair) )

         l = 0
         do k1 =  1, nkind
         do k2 = k1, nkind
            l = l + 1
            j1 = natom_spec(k1)
            j2 = natom_spec(k2)
            if ( k1 .ne. k2 ) npair_kindpair(l) = j1*j2
            if ( k1 .eq. k2 ) npair_kindpair(l) = j1*(j1-1)/2
         end do
         end do

!-----------------------------------------------------------------------
!        /*   step intervals of analysis                              */
!-----------------------------------------------------------------------

         call read_int1_MPI( iprint_rdf,  '<iprint_rdf>',  12, iounit )
         call read_int1_MPI( iprint_trj,  '<iprint_trj>',  12, iounit )
         call read_int1_MPI( iprint_xyz,  '<iprint_xyz>',  12, iounit )
         call read_int1_MPI( iprint_xsf,  '<iprint_xsf>',  12, iounit )
         call read_int1_MPI( iprint_dcd,  '<iprint_dcd>',  12, iounit )
         call read_int1_MPI( iprint_dip,  '<iprint_dip>',  12, iounit )

!-----------------------------------------------------------------------
!        /*   check if file called `averages.ini' exists              */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then
            call testfile ( 'averages.ini', 12, itest )
         end if

         call my_mpi_bcast_int_0( itest )

!-----------------------------------------------------------------------
!        /*   if the file does not exist, initial start.              */
!-----------------------------------------------------------------------

         if ( itest .eq. 1 ) then

            call analysis_rdf_MPI( 0 )
            call analysis_trj_MPI( 0 )
            call analysis_xyz_MPI( 0 )
            call analysis_xsf_MPI( 0 )
            call analysis_dcd_MPI( 0 )
            call analysis_dip_MPI( 0 )

!-----------------------------------------------------------------------
!        /*   if the file exists, restart.                            */
!-----------------------------------------------------------------------

         else

            if ( myrank .eq. 0 ) then
               open ( iounit_avg, file = 'averages.ini')
            end if

            call analysis_rdf_MPI( 1 )
            call analysis_trj_MPI( 1 )
            call analysis_xyz_MPI( 1 )
            call analysis_xsf_MPI( 1 )
            call analysis_dcd_MPI( 1 )
            call analysis_dip_MPI( 1 )

            if ( myrank .eq. 0 ) then
               close( iounit_avg )
            end if

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         call analysis_rdf_MPI( 2 )
         call analysis_trj_MPI( 2 )
         call analysis_xyz_MPI( 2 )
         call analysis_xsf_MPI( 2 )
         call analysis_dcd_MPI( 2 )
         call analysis_dip_MPI( 2 )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         if ( myrank .eq. 0 ) then
            open ( iounit_avg, file = 'averages.ini' )
         end if

         call analysis_rdf_MPI( 3 )
         call analysis_trj_MPI( 3 )
         call analysis_xyz_MPI( 3 )
         call analysis_xsf_MPI( 3 )
         call analysis_dcd_MPI( 3 )
         call analysis_dip_MPI( 3 )

         if ( myrank .eq. 0 ) then
            close( iounit_avg )
         end if

      end if

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine reset_bath
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use rotor_variables, only : s5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     //   reset thermostat
!-----------------------------------------------------------------------

      s5(0) = 1.d0
      s5(1) = 0.d0
      s5(2) = 0.d0
      s5(3) = 0.d0
      s5(4) = 0.d0
      s5(5) = 0.d0

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine setup_rotor_nvt_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, boltz, omega_bath, beta, iounit, ndof, myrank

      use rotor_variables, only : &
     &   r5, s5, o4, q4, qmass_rotor, a5_gear, a4_gear, dt_gear, &
     &   x_mol, y_mol, z_mol, &
     &   pmi, qdot, fnx, fny, fnz, physmass_mol, &
     &   ekin_tra, ekin_rot, temp_tra, temp_rot, fgx, fgy, fgz, &
     &   ncomp, nmol, nmol_6, nmol_5, &
     &   nmol_3, natom_per_mol, list_atom_mol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: k, l, m, itest

      real(8) :: xg_mol, yg_mol, zg_mol

!-----------------------------------------------------------------------
!     //   initialize molecular components
!-----------------------------------------------------------------------

      call init_molcomp_MPI

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   gear parameter: fifth order
      if ( .not. allocated( a5_gear ) ) &
     &   allocate( a5_gear(0:5) )

!     //   gear parameter: fourth order
      if ( .not. allocated( a4_gear ) ) &
     &   allocate( a4_gear(0:4) )

!     //   powers of dt: fifth order
      if ( .not. allocated( dt_gear ) ) &
     &   allocate( dt_gear(5) )

!     //   thermostat: fifth order
      if ( .not. allocated( s5 ) ) &
     &   allocate( s5(0:5) )

!     //   cartesian coordinates
      if ( .not. allocated( r5 ) ) &
     &   allocate( r5(0:5,3,nmol) )

!     //   angular velocity
      if ( .not. allocated( o4 ) ) &
     &   allocate( o4(0:4,3,nmol_6+nmol_5) )

!     //   quaternion
      if ( .not. allocated( q4 ) ) &
     &   allocate( q4(0:4,4,nmol_6+nmol_5) )

!     //   translational forces
      if ( .not. allocated( fgx ) ) &
     &   allocate( fgx(nmol) )
      if ( .not. allocated( fgy ) ) &
     &   allocate( fgy(nmol) )
      if ( .not. allocated( fgz ) ) &
     &   allocate( fgz(nmol) )

!     //   moment of inertia
      if ( .not. allocated( pmi ) ) &
     &   allocate( pmi(3,nmol_6+nmol_5) )

!     //   velocity of quaternion
      if ( .not. allocated( qdot ) ) &
     &   allocate( qdot(4,nmol) )

!     //   torque
      if ( .not. allocated( fnx ) ) &
     &   allocate( fnx(nmol) )
      if ( .not. allocated( fny ) ) &
     &   allocate( fny(nmol) )
      if ( .not. allocated( fnz ) ) &
     &   allocate( fnz(nmol) )

!     //   kinetic energy of molecular translation
      if ( .not. allocated( ekin_tra ) ) &
     &   allocate( ekin_tra(ncomp) )

!     //   instantaneous temperature of molecular translation
      if ( .not. allocated( temp_tra ) ) &
     &   allocate( temp_tra(ncomp) )

!     //   kinetic energy of molecular rotation
      if ( .not. allocated( ekin_rot ) ) &
     &   allocate( ekin_rot(ncomp) )

!     //   instantaneous temperature of molecular rotation
      if ( .not. allocated( temp_rot ) ) &
     &   allocate( temp_rot(ncomp) )

!-----------------------------------------------------------------------
!     //   number of molecules
!-----------------------------------------------------------------------

!     //   number of degrees of freedom
      ndof = 6*nmol_6 + 5*nmol_5 + 3*nmol_3

!-----------------------------------------------------------------------
!     //   molecular masses
!-----------------------------------------------------------------------

      do k = 1, nmol

         physmass_mol(k) = 0.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            physmass_mol(k) = physmass_mol(k) + physmass(m)

         end do

      end do

!-----------------------------------------------------------------------
!     //   center of mass set to zero in molecular frame
!-----------------------------------------------------------------------

      do k = 1, nmol

         xg_mol = 0.d0
         yg_mol = 0.d0
         zg_mol = 0.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            xg_mol = xg_mol + physmass(m) * x_mol(l,k)
            yg_mol = yg_mol + physmass(m) * y_mol(l,k)
            zg_mol = zg_mol + physmass(m) * z_mol(l,k)

         end do

         xg_mol = xg_mol / physmass_mol(k)
         yg_mol = yg_mol / physmass_mol(k)
         zg_mol = zg_mol / physmass_mol(k)

         do l = 1, natom_per_mol(k)

            x_mol(l,k) = x_mol(l,k) - xg_mol
            y_mol(l,k) = y_mol(l,k) - yg_mol
            z_mol(l,k) = z_mol(l,k) - zg_mol

         end do

      end do

!-----------------------------------------------------------------------
!     //   moment of inertia
!-----------------------------------------------------------------------

      call moment_of_inertia

!-----------------------------------------------------------------------
!     //   mass of thermostat
!-----------------------------------------------------------------------

      qmass_rotor = ndof / beta / omega_bath**2

!-----------------------------------------------------------------------
!     /*   initial/restart configuration, velocity and thermostat     */
!-----------------------------------------------------------------------

!     /*   check if file called `rotor.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'rotor.ini', 9, itest )
      end if

      call my_mpi_bcast_int_0( itest )

!     /*   if yes, restart  */
      if ( itest .eq. 0 ) then

!        /*   restart   */
         call restart_nvt_rotor_MPI( 1 )

!        /*   gear method: forward transform   */
         call trans_gear_nvt( 1 )

!     /*   if no, initialize using centroid.dat or structure.dat   */
      else

!        /*   initialize   */
         call init_nvt_rotor_MPI

!        /*   gear method: forward transform   */
         call trans_gear_nvt( 1 )

!        /*   apply velocity scaling   */
         call velocity_scaling_rotor

      end if

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine setup_rotor_nve_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, boltz, omega_bath, beta, iounit, ndof, myrank

      use rotor_variables, only : &
     &   r5, o4, q4, qmass_rotor, a5_gear, a4_gear, dt_gear, &
     &   x_mol, y_mol, z_mol, pmi, qdot, fnx, fny, fnz, physmass_mol, &
     &   ekin_tra, ekin_rot, temp_tra, temp_rot, fgx, fgy, fgz, ncomp, &
     &   nmol, nmol_6, nmol_5, nmol_3, natom_per_mol, list_atom_mol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: k, l, m, itest

      real(8) :: xg_mol, yg_mol, zg_mol

!-----------------------------------------------------------------------
!     //   initialize molecular components
!-----------------------------------------------------------------------

      call init_molcomp_MPI

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   gear parameter: fifth order
      if ( .not. allocated( a5_gear ) ) &
     &   allocate( a5_gear(0:5) )

!     //   gear parameter: fourth order
      if ( .not. allocated( a4_gear ) ) &
     &   allocate( a4_gear(0:4) )

!     //   powers of dt: fifth order
      if ( .not. allocated( dt_gear ) ) &
     &   allocate( dt_gear(5) )

!     //   cartesian coordinates
      if ( .not. allocated( r5 ) ) &
     &   allocate( r5(0:5,3,nmol) )

!     //   angular velocity
      if ( .not. allocated( o4 ) ) &
     &   allocate( o4(0:4,3,nmol_6+nmol_5) )

!     //   quaternion
      if ( .not. allocated( q4 ) ) &
     &   allocate( q4(0:4,4,nmol_6+nmol_5) )

!     //   translational forces
      if ( .not. allocated( fgx ) ) &
     &   allocate( fgx(nmol) )
      if ( .not. allocated( fgy ) ) &
     &   allocate( fgy(nmol) )
      if ( .not. allocated( fgz ) ) &
     &   allocate( fgz(nmol) )

!     //   moment of inertia
      if ( .not. allocated( pmi ) ) &
     &   allocate( pmi(3,nmol_6+nmol_5) )

!     //   velocity of quaternion
      if ( .not. allocated( qdot ) ) &
     &   allocate( qdot(4,nmol) )

!     //   torque
      if ( .not. allocated( fnx ) ) &
     &   allocate( fnx(nmol) )
      if ( .not. allocated( fny ) ) &
     &   allocate( fny(nmol) )
      if ( .not. allocated( fnz ) ) &
     &   allocate( fnz(nmol) )

!     //   kinetic energy of molecular translation
      if ( .not. allocated( ekin_tra ) ) &
     &   allocate( ekin_tra(ncomp) )

!     //   instantaneous temperature of molecular translation
      if ( .not. allocated( temp_tra ) ) &
     &   allocate( temp_tra(ncomp) )

!     //   kinetic energy of molecular rotation
      if ( .not. allocated( ekin_rot ) ) &
     &   allocate( ekin_rot(ncomp) )

!     //   instantaneous temperature of molecular rotation
      if ( .not. allocated( temp_rot ) ) &
     &   allocate( temp_rot(ncomp) )

!-----------------------------------------------------------------------
!     //   number of molecules
!-----------------------------------------------------------------------

!     //   number of degrees of freedom
      ndof = 6*nmol_6 + 5*nmol_5 + 3*nmol_3

!-----------------------------------------------------------------------
!     //   masses
!-----------------------------------------------------------------------

      do k = 1, nmol

         physmass_mol(k) = 0.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            physmass_mol(k) = physmass_mol(k) + physmass(m)

         end do

      end do

!-----------------------------------------------------------------------
!     //   center of mass set to zero in molecular frame
!-----------------------------------------------------------------------

      do k = 1, nmol

         xg_mol = 0.d0
         yg_mol = 0.d0
         zg_mol = 0.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            xg_mol = xg_mol + physmass(m) * x_mol(l,k)
            yg_mol = yg_mol + physmass(m) * y_mol(l,k)
            zg_mol = zg_mol + physmass(m) * z_mol(l,k)

         end do

         xg_mol = xg_mol / physmass_mol(k)
         yg_mol = yg_mol / physmass_mol(k)
         zg_mol = zg_mol / physmass_mol(k)

         do l = 1, natom_per_mol(k)

            x_mol(l,k) = x_mol(l,k) - xg_mol
            y_mol(l,k) = y_mol(l,k) - yg_mol
            z_mol(l,k) = z_mol(l,k) - zg_mol

         end do

      end do

!-----------------------------------------------------------------------
!     //   moment of inertia
!-----------------------------------------------------------------------

      call moment_of_inertia

!-----------------------------------------------------------------------
!     //   mass of thermostat
!-----------------------------------------------------------------------

      qmass_rotor = ndof/beta/omega_bath**2

!-----------------------------------------------------------------------
!     /*   initial/restart configuration and velocity                 */
!-----------------------------------------------------------------------

!     /*   check if file called `rotor.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'rotor.ini', 9, itest )
      end if

      call my_mpi_bcast_int_0( itest )

!     /*   if yes, restart  */
      if ( itest .eq. 0 ) then

         call restart_nve_rotor_MPI( 1 )

!        /*   gear method: forward transform   */
         call trans_gear_nve( 1 )

!     /*   if no, initialize using centroid.dat or structure.dat   */
      else

!        /*   restart   */
         call init_nve_rotor_MPI

!        /*   gear method: forward transform   */
         call trans_gear_nve( 1 )

!        /*   apply velocity scaling   */
         call velocity_scaling_rotor

      end if

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine velocity_scaling_rotor
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temperature, boltz

      use rotor_variables, only : &
     &   physmass_mol, r5, o4, temp_tra, temp_rot, &
     &   ncomp_6, ncomp_5, ncomp_3, nmol_comp, nmol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k, l, m

      real(8) :: scalet, scaler, vrx, vry, vrz

      real(8) :: temp_min = 1.d0

!-----------------------------------------------------------------------
!     //   subtract overall translational velocity
!-----------------------------------------------------------------------

      vrx = 0.d0
      vry = 0.d0
      vrz = 0.d0

      do k = 1, nmol
         vrx = vrx + physmass_mol(k) * r5(1,1,k)
         vry = vry + physmass_mol(k) * r5(1,2,k)
         vrz = vrz + physmass_mol(k) * r5(1,3,k)
      end do

      vrx = vrx / dble(nmol)
      vry = vry / dble(nmol)
      vrz = vrz / dble(nmol)

      do k = 1, nmol
         r5(1,1,k) = r5(1,1,k) - vrx / physmass_mol(k)
         r5(1,2,k) = r5(1,2,k) - vry / physmass_mol(k)
         r5(1,3,k) = r5(1,3,k) - vrz / physmass_mol(k)
      end do

!-----------------------------------------------------------------------
!     //   instantaneous temperature
!-----------------------------------------------------------------------

      call kinetic_energy_rotor

!-----------------------------------------------------------------------
!     //   velocity scaling factor for 6-dimensional molecules
!-----------------------------------------------------------------------

      l = 0

      do m = 1, ncomp_6

         if ( temp_tra(m) .lt. temp_min ) then
            scalet = sqrt( temperature / temp_min    )
         else
            scalet = sqrt( temperature / temp_tra(m) )
         end if

         if ( temp_rot(m) .lt. temp_min ) then
            scaler = sqrt( temperature / temp_min    )
         else
            scaler = sqrt( temperature / temp_rot(m) )
         end if

         do k = 1, nmol_comp(m)

            l = l + 1

            do j = 1, 3
               r5(1,j,l) = scalet * r5(1,j,l)
            end do

            do j = 1, 3
               o4(0,j,l) = scaler * o4(0,j,l)
            end do

         end do

      end do

!-----------------------------------------------------------------------
!     //   velocity scaling factor for 5-dimensional molecules
!-----------------------------------------------------------------------

      do m = ncomp_6 + 1, ncomp_6 + ncomp_5

         if ( temp_tra(m) .lt. temp_min ) then
            scalet = sqrt( temperature / temp_min    )
         else
            scalet = sqrt( temperature / temp_tra(m) )
         end if

         if ( temp_rot(m) .lt. temp_min ) then
            scaler = sqrt( temperature / temp_min    )
         else
            scaler = sqrt( temperature / temp_rot(m) )
         end if

         do k = 1, nmol_comp(m)

            l = l + 1

            do j = 1, 3
               r5(1,j,l) = scalet * r5(1,j,l)
            end do

            do j = 1, 2
               o4(0,j,l) = scaler * o4(0,j,l)
            end do

         end do

      end do

!-----------------------------------------------------------------------
!     //   velocity scaling factor for 3-dimensional molecules
!-----------------------------------------------------------------------

      do m = ncomp_6 + ncomp_5 + 1, ncomp_6 + ncomp_5 + ncomp_3

         if ( temp_tra(m) .lt. temp_min ) then
            scalet = sqrt( temperature / temp_min    )
         else
            scalet = sqrt( temperature / temp_tra(m) )
         end if

         do k = 1, nmol_comp(m)

            l = l + 1

            do j = 1, 3
               r5(1,j,l) = scalet * r5(1,j,l)
            end do

         end do

      end do

!-----------------------------------------------------------------------
!     //   transfer from omega to qdot
!-----------------------------------------------------------------------

      call trans_o_q1

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine force_center_of_mass
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : fx, fy, fz

      use rotor_variables, only : &
     &   fgx, fgy, fgz, nmol, natom_per_mol, list_atom_mol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: k, l, m

!-----------------------------------------------------------------------

      fgx(:)   = 0.d0
      fgy(:)   = 0.d0
      fgz(:)   = 0.d0

      do k = 1, nmol
      do l = 1, natom_per_mol(k)

         m = list_atom_mol(l,k)

         fgx(k) = fgx(k) + fx(m,1)
         fgy(k) = fgy(k) + fy(m,1)
         fgz(k) = fgz(k) + fz(m,1)

      end do
      end do

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine backup_rotor_nvt_MPI
!***********************************************************************

!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, fictmass, istep_end, nstep, iexit, iprint_rest, &
     &   iounit, myrank

      implicit none

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( istep_end .eq. nstep ) then
         continue
      else if ( iexit .eq. 1 ) then
         continue
      else
         if ( iprint_rest .le. 0 ) then
            return
         else
            if ( mod(istep_end,iprint_rest) .eq. 0 ) then
               continue
            else
               return
            end if
         end if
      end if

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

      call trans_gear_nvt( 2 )

      call restart_nvt_rotor_MPI( 2 )

      call trans_gear_nvt( 1 )

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

      fictmass(:,1) = physmass(:)

      call restart_position_MPI( 3 )
      call restart_velocity_MPI( 3 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
         close( iounit )
      end if

      call my_mpi_bcast_int_0( istep_end )

!-----------------------------------------------------------------------
!     /*   save averages                                              */
!-----------------------------------------------------------------------

      call analysis_rotor_MPI( 3 )

      return
      end





!***********************************************************************
      subroutine backup_rotor_nve_MPI
!***********************************************************************

!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, fictmass, istep_end, nstep, iexit, iprint_rest, &
     &   iounit, myrank

      implicit none

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( istep_end .eq. nstep ) then
         continue
      else if ( iexit .eq. 1 ) then
         continue
      else
         if ( iprint_rest .le. 0 ) then
            return
         else
            if ( mod(istep_end,iprint_rest) .eq. 0 ) then
               continue
            else
               return
            end if
         end if
      end if

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

      call trans_gear_nve( 2 )

      call restart_nve_rotor_MPI( 2 )

      call trans_gear_nve( 1 )

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

      fictmass(:,1) = physmass(:)

      call restart_position_MPI( 3 )
      call restart_velocity_MPI( 3 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
         close( iounit )
      end if

      call my_mpi_bcast_int_0( istep_end )

!-----------------------------------------------------------------------
!     /*   save averages                                              */
!-----------------------------------------------------------------------

      call analysis_rotor_MPI( 3 )

      return
      end





!***********************************************************************
      subroutine restart_nvt_rotor_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : iounit, myrank

      use rotor_variables, only : &
     &   r5, s5, o4, q4, nmol, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      integer :: ioption

!-----------------------------------------------------------------------
!     //   read configuration, velocity, thermostat
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'rotor.ini' )

            do k = 1, nmol
            do j = 1, 3
               read ( iounit, * ) ( r5(i,j,k), i = 0, 5 )
            end do
            end do

            do k = 1, nmol_6
            do j = 1, 3
               read ( iounit, * ) ( o4(i,j,k), i = 0, 4 )
            end do
            end do

            do k = nmol_6 + 1, nmol_6 + nmol_5
            do j = 1, 2
               read ( iounit, * ) ( o4(i,j,k), i = 0, 4 )
            end do
            end do

            do k = 1, nmol_6 + nmol_5
            do j = 1, 4
               read ( iounit, * ) ( q4(i,j,k), i = 0, 4 )
            end do
            end do

            read ( iounit, * ) ( s5(i), i = 0, 5 )

            close( iounit )

         end if

         call my_mpi_bcast_real_03( r5, 5, 3, nmol )
         call my_mpi_bcast_real_03( o4, 4, 3, nmol_6+nmol_5 )
         call my_mpi_bcast_real_03( q4, 4, 4, nmol_6+nmol_5 )
         call my_mpi_bcast_real_01( s5, 5 )
 
      end if

!-----------------------------------------------------------------------
!     //   write configuration, velocity, thermostat
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'rotor.ini' )

            do k = 1, nmol
            do j = 1, 3
               write( iounit, '(6e24.16)' ) ( r5(i,j,k), i = 0, 5 )
            end do
            end do

            do k = 1, nmol_6
            do j = 1, 3
               write( iounit, '(5e24.16)' ) ( o4(i,j,k), i = 0, 4 )
            end do
            end do

            do k = nmol_6 + 1, nmol_6 + nmol_5
            do j = 1, 2
               write( iounit, '(5e24.16)' ) ( o4(i,j,k), i = 0, 4 )
            end do
            end do

            do k = 1, nmol_6 + nmol_5
            do j = 1, 4
               write( iounit, '(5e24.16)' ) ( q4(i,j,k), i = 0, 4 )
            end do
            end do

            write( iounit, '(6e24.16)' ) ( s5(i), i = 0, 5 )

            close( iounit )

         end if

      end if

      return
      end





!***********************************************************************
      subroutine restart_nve_rotor_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : iounit, myrank

      use rotor_variables, only : &
     &   r5, o4, q4, nmol, nmol_6, nmol_5

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      integer :: ioption

!-----------------------------------------------------------------------
!     //   read configuration, velocity
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'rotor.ini' )

            do k = 1, nmol
            do j = 1, 3
               read ( iounit, * ) ( r5(i,j,k), i = 0, 5 )
            end do
            end do

            do k = 1, nmol_6
            do j = 1, 3
               read ( iounit, * ) ( o4(i,j,k), i = 0, 4 )
            end do
            end do

            do k = nmol_6 + 1, nmol_6 + nmol_5
            do j = 1, 2
               read ( iounit, * ) ( o4(i,j,k), i = 0, 4 )
            end do
            end do

            do k = 1, nmol_6 + nmol_5
            do j = 1, 4
               read ( iounit, * ) ( q4(i,j,k), i = 0, 4 )
            end do
            end do

            close( iounit )

         end if

         call my_mpi_bcast_real_03( r5, 5, 3, nmol )
         call my_mpi_bcast_real_03( o4, 4, 3, nmol_6+nmol_5 )
         call my_mpi_bcast_real_03( q4, 4, 4, nmol_6+nmol_5 )
 
      end if

!-----------------------------------------------------------------------
!     //   write configuration, velocity
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'rotor.ini' )

            do k = 1, nmol
            do j = 1, 3
               write( iounit, '(6e24.16)' ) ( r5(i,j,k), i = 0, 5 )
            end do
            end do

            do k = 1, nmol_6
            do j = 1, 3
               write( iounit, '(5e24.16)' ) ( o4(i,j,k), i = 0, 4 )
            end do
            end do

            do k = nmol_6 + 1, nmol_6 + nmol_5
            do j = 1, 2
               write( iounit, '(5e24.16)' ) ( o4(i,j,k), i = 0, 4 )
            end do
            end do

            do k = 1, nmol_6 + nmol_5
            do j = 1, 4
               write( iounit, '(5e24.16)' ) ( q4(i,j,k), i = 0, 4 )
            end do
            end do

            write( iounit, '(6e24.16)' ) &
     &         1.d0, 0.d0, 0.d0, 0.d0, 0.d0, 0.d0

            close( iounit )

         end if

      end if

      return
      end





!***********************************************************************
      subroutine trans_velocity_rotor
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : vx, vy, vz, vux, vuy, vuz, dt

      use rotor_variables, only : &
     &   r5, q4, o4, x_mol, y_mol, z_mol, list_atom_mol, &
     &   natom_per_mol, nmol_6, nmol_5, nmol_3

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: k, l, m

!     //   real numbers
      real(8) :: a(3,3), vm(3)

!-----------------------------------------------------------------------
!     //   transfer from molecular to laboratory frame: 6-d molecule
!-----------------------------------------------------------------------

      vx(:,1) = 0.d0
      vy(:,1) = 0.d0
      vz(:,1) = 0.d0

      do k = 1, nmol_6

         a(1,1) =   2.d0 * ( q4(0,2,k) * q4(0,2,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(1,2) =   2.d0 * ( q4(0,3,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,2,k) )
         a(1,3) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     + q4(0,1,k) * q4(0,4,k) )
         a(2,1) = - 2.d0 * ( q4(0,1,k) * q4(0,2,k) &
     &                     + q4(0,3,k) * q4(0,4,k) )
         a(2,2) =   2.d0 * ( q4(0,1,k) * q4(0,1,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(2,3) =   2.d0 * ( q4(0,2,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,3,k) )
         a(3,1) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     - q4(0,1,k) * q4(0,4,k) )
         a(3,2) = - 2.d0 * ( q4(0,1,k) * q4(0,3,k) &
     &                     + q4(0,2,k) * q4(0,4,k) )
         a(3,3) =   2.d0 * ( q4(0,3,k) * q4(0,3,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            vm(1) = o4(0,2,k) * z_mol(l,k) - o4(0,3,k) * y_mol(l,k)
            vm(2) = o4(0,3,k) * x_mol(l,k) - o4(0,1,k) * z_mol(l,k)
            vm(3) = o4(0,1,k) * y_mol(l,k) - o4(0,2,k) * x_mol(l,k)

            vx(m,1) = ( r5(1,1,k) + a(1,1) * vm(1) &
     &                            + a(2,1) * vm(2) &
     &                            + a(3,1) * vm(3) ) / dt
            vy(m,1) = ( r5(1,2,k) + a(1,2) * vm(1) &
     &                            + a(2,2) * vm(2) &
     &                            + a(3,2) * vm(3) ) / dt
            vz(m,1) = ( r5(1,3,k) + a(1,3) * vm(1) &
     &                            + a(2,3) * vm(2) &
     &                            + a(3,3) * vm(3) ) / dt

         end do

      end do

!-----------------------------------------------------------------------
!     //   transfer from molecular to laboratory frame: 5-d molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + 1, nmol_6 + nmol_5

         a(1,1) =   2.d0 * ( q4(0,2,k) * q4(0,2,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(1,2) =   2.d0 * ( q4(0,3,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,2,k) )
         a(2,1) = - 2.d0 * ( q4(0,1,k) * q4(0,2,k) &
     &                     + q4(0,3,k) * q4(0,4,k) )
         a(2,2) =   2.d0 * ( q4(0,1,k) * q4(0,1,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(3,1) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     - q4(0,1,k) * q4(0,4,k) )
         a(3,2) = - 2.d0 * ( q4(0,1,k) * q4(0,3,k) &
     &                     + q4(0,2,k) * q4(0,4,k) )

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            vm(1) =   o4(0,2,k) * z_mol(l,k)
            vm(2) = - o4(0,1,k) * z_mol(l,k)

            vx(m,1) = ( r5(1,1,k) + a(1,1) * vm(1) &
     &                            + a(1,2) * vm(2) ) / dt
            vy(m,1) = ( r5(1,2,k) + a(2,1) * vm(1) &
     &                            + a(2,2) * vm(2) ) / dt
            vz(m,1) = ( r5(1,3,k) + a(3,1) * vm(1) &
     &                            + a(3,2) * vm(2) ) / dt

         end do

      end do

!-----------------------------------------------------------------------
!     //   transfer from molecular to laboratory frame: 3-d molecule
!-----------------------------------------------------------------------

      do k = nmol_6 + nmol_5 + 1, nmol_6 + nmol_5 + nmol_3

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            vx(m,1) = r5(1,1,k) / dt
            vy(m,1) = r5(1,2,k) / dt
            vz(m,1) = r5(1,3,k) / dt

         end do

      end do

!-----------------------------------------------------------------------
!     //   atomic coordinates in laboratory axis
!-----------------------------------------------------------------------

      vux(:,1) = vx(:,1)
      vuy(:,1) = vy(:,1)
      vuz(:,1) = vz(:,1)

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine init_nvt_rotor_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    beta, pi, au_length, iounit, x, y, z, dt, natom, input_style, &
     &    ikind, species, myrank

      use rotor_variables, only : &
     &   r5, q4, o4, physmass_mol, pmi, zeta, s5, nmol_6, nmol_5, &
     &   nmol_3

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   euler angles
!      real(8) :: tht, phi, psi

!     //   integers
      integer :: k, ierr

!     //   real numbers
      real(8) :: gasdev

!     //   characters
      character(len=10) :: char

!     //   conversion factor
      real(8), parameter :: bohr2ang = au_length * 1.d+10

!     //   characters
      character(len=120) :: charline

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   numbering of species   */
      if ( .not. allocated(ikind) ) allocate ( ikind(natom) )

!-----------------------------------------------------------------------
!     /*   atomic coordinates are read from file                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

      if ( input_style(1:4) .eq. 'OLD ' ) then

         open ( iounit, file = 'centroid.dat', status = 'unknown' )

         do k = 1, natom
            read ( iounit, *, iostat=ierr ) x(k,1), y(k,1), z(k,1)
         end do

         close( iounit )

      else if ( input_style(1:4) .eq. 'NEW ' ) then

         open ( iounit, file = 'structure.dat', status = 'unknown' )

         read ( iounit, *, iostat=ierr )
         read ( iounit, *, iostat=ierr ) char

         if ( char(1:5) .eq. 'BOHR ' ) then

            do k = 1, natom
               read ( iounit, '(a)', iostat=ierr ) charline
               read ( charline, *, iostat=ierr ) &
     &            species(k), x(k,1), y(k,1), z(k,1), ikind(k)
               if ( ierr .ne. 0 ) then
                  read ( charline, *, iostat=ierr ) &
     &               species(k), x(k,1), y(k,1), z(k,1)
                  if ( ierr .ne. 0 ) exit
                  ikind(k) = 1
               end if
            end do

         else if ( char(1:9) .eq. 'ANGSTROM ' ) then

            do k = 1, natom
               read ( iounit, '(a)', iostat=ierr ) charline
               read ( charline, *, iostat=ierr ) &
     &            species(k), x(k,1), y(k,1), z(k,1), ikind(k)
               if ( ierr .ne. 0 ) then
                  read ( charline, *, iostat=ierr ) &
     &               species(k), x(k,1), y(k,1), z(k,1)
                  if ( ierr .ne. 0 ) exit
                  ikind(k) = 1
               end if
            end do

            x(:,1) = x(:,1) / bohr2ang
            y(:,1) = y(:,1) / bohr2ang
            z(:,1) = z(:,1) / bohr2ang

         else

            ierr = 1

         end if

         close( iounit )

      end if

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         if ( input_style(1:4) .eq. 'OLD ' ) then
            write( 6, '(a)' ) 'Error - centroid.dat read incorrectly.'
            write( 6, '(a)' )
         else if ( input_style(1:4) .eq. 'NEW ' ) then
            write( 6, '(a)' ) 'Error - structure.dat read incorrectly.'
            write( 6, '(a)' )
         end if
      end if

      end if

!     /*   communication   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine init_nvt_rotor_MPI', 29 )

!     /*   communication   */
      call my_mpi_bcast_int_1( ikind, natom )
      call my_mpi_bcast_real_2( x, natom, 1 )
      call my_mpi_bcast_real_2( y, natom, 1 )
      call my_mpi_bcast_real_2( z, natom, 1 )

!-----------------------------------------------------------------------
!     /*   calculate molecular center and quaternion                  */
!-----------------------------------------------------------------------

      call opt_frame_rotor

!-----------------------------------------------------------------------
!     //   maxwell distribution of velocity
!-----------------------------------------------------------------------

      do k = 1, nmol_6 + nmol_5 + nmol_3

         r5(1,1,k) = sqrt( 1.d0 / beta / physmass_mol(k) ) * gasdev()
         r5(1,2,k) = sqrt( 1.d0 / beta / physmass_mol(k) ) * gasdev()
         r5(1,3,k) = sqrt( 1.d0 / beta / physmass_mol(k) ) * gasdev()

      end do

      r5(2,:,:) = 0.d0
      r5(3,:,:) = 0.d0
      r5(4,:,:) = 0.d0
      r5(5,:,:) = 0.d0

!-----------------------------------------------------------------------
!     //   randomize quaternion
!-----------------------------------------------------------------------

      q4(1,:,:) = 0.d0
      q4(2,:,:) = 0.d0
      q4(3,:,:) = 0.d0
      q4(4,:,:) = 0.d0

!      do k = 1, nmol_6 + nmol_5
!
!        tht = ranf1() * pi
!        phi = ranf1() * 2.d0 * pi
!        psi = ranf1() * 2.d0 * pi
!
!        q4(0,1,k) = sin( 0.5d0 * tht ) * sin( 0.5d0 * ( psi - phi ) )
!        q4(0,2,k) = sin( 0.5d0 * tht ) * cos( 0.5d0 * ( psi - phi ) )
!        q4(0,3,k) = cos( 0.5d0 * tht ) * sin( 0.5d0 * ( psi + phi ) )
!        q4(0,4,k) = cos( 0.5d0 * tht ) * cos( 0.5d0 * ( psi + phi ) )
!
!      end do

!-----------------------------------------------------------------------
!     //   maxwell distribution of angular velocity
!-----------------------------------------------------------------------

      o4(:,:,:) = 0.d0

      do k = 1, nmol_6

         o4(0,1,k) = sqrt( 1.d0 / beta / pmi(1,k) ) * gasdev()
         o4(0,2,k) = sqrt( 1.d0 / beta / pmi(2,k) ) * gasdev()
         o4(0,3,k) = sqrt( 1.d0 / beta / pmi(3,k) ) * gasdev()

      end do

      do k = nmol_6 + 1, nmol_6 + nmol_5

         o4(0,1,k) = sqrt( 1.d0 / beta / pmi(1,k) ) * gasdev()
         o4(0,2,k) = sqrt( 1.d0 / beta / pmi(2,k) ) * gasdev()

      end do

!-----------------------------------------------------------------------
!     //   transfer from omega to qdot
!-----------------------------------------------------------------------

      call trans_o_q1

!-----------------------------------------------------------------------
!     //   thermostat
!-----------------------------------------------------------------------

      s5(0) = 1.d0
      s5(1) = 0.d0
      s5(2) = 0.d0
      s5(3) = 0.d0
      s5(4) = 0.d0
      s5(5) = 0.d0

      zeta = s5(1) / dt / s5(0)

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine init_nve_rotor_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    beta, pi, au_length, iounit, x, y, z, natom, input_style, &
     &    ikind, species, myrank

      use rotor_variables, only : &
     &   r5, q4, o4, physmass_mol, pmi, nmol_6, nmol_5, nmol_3

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   euler angles
!      real(8) :: tht, phi, psi

!     //   integers
      integer :: k, ierr

!     //   real numbers
      real(8) :: gasdev

!     //   real numbers
!      real(8) :: ranf1

!     //   characters
      character(len=10) :: char

!     //   conversion factor
      real(8), parameter :: bohr2ang = au_length * 1.d+10

!     //   characters
      character(len=120) :: charline

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   numbering of species   */
      if ( .not. allocated(ikind) ) allocate ( ikind(natom) )

!-----------------------------------------------------------------------
!     /*   atomic coordinates are read from file                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

      if ( input_style(1:4) .eq. 'OLD ' ) then

         open ( iounit, file = 'centroid.dat', status = 'unknown' )

         do k = 1, natom
            read ( iounit, *, iostat=ierr ) x(k,1), y(k,1), z(k,1)
         end do

         close( iounit )

      else if ( input_style(1:4) .eq. 'NEW ' ) then

         open ( iounit, file = 'structure.dat', status = 'unknown' )

         read ( iounit, *, iostat=ierr )
         read ( iounit, *, iostat=ierr ) char

         if ( char(1:5) .eq. 'BOHR ' ) then

            do k = 1, natom
               read ( iounit, '(a)', iostat=ierr ) charline
               read ( charline, *, iostat=ierr ) &
     &            species(k), x(k,1), y(k,1), z(k,1), ikind(k)
               if ( ierr .ne. 0 ) then
                  read ( charline, *, iostat=ierr ) &
     &               species(k), x(k,1), y(k,1), z(k,1)
                  if ( ierr .ne. 0 ) exit
                  ikind(k) = 1
               end if
            end do

         else if ( char(1:9) .eq. 'ANGSTROM ' ) then

            do k = 1, natom
               read ( iounit, '(a)', iostat=ierr ) charline
               read ( charline, *, iostat=ierr ) &
     &            species(k), x(k,1), y(k,1), z(k,1), ikind(k)
               if ( ierr .ne. 0 ) then
                  read ( charline, *, iostat=ierr ) &
     &               species(k), x(k,1), y(k,1), z(k,1)
                  if ( ierr .ne. 0 ) exit
                  ikind(k) = 1
               end if
            end do

            x(:,1) = x(:,1) / bohr2ang
            y(:,1) = y(:,1) / bohr2ang
            z(:,1) = z(:,1) / bohr2ang

         else

            ierr = 1

         end if

         close( iounit )

      end if

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         if ( input_style(1:4) .eq. 'OLD ' ) then
            write( 6, '(a)' ) 'Error - centroid.dat read incorrectly.'
            write( 6, '(a)' )
         else if ( input_style(1:4) .eq. 'NEW ' ) then
            write( 6, '(a)' ) 'Error - structure.dat read incorrectly.'
            write( 6, '(a)' )
         end if
      end if

      end if

!     /*   communication   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine init_nve_rotor_MPI', 29 )

!     /*   communication   */
      call my_mpi_bcast_int_1( ikind, natom )
      call my_mpi_bcast_real_2( x, natom, 1 )
      call my_mpi_bcast_real_2( y, natom, 1 )
      call my_mpi_bcast_real_2( z, natom, 1 )

!-----------------------------------------------------------------------
!     /*   calculate molecular center and quaternion                  */
!-----------------------------------------------------------------------

      call opt_frame_rotor

!-----------------------------------------------------------------------
!     //   maxwell distribution of velocity
!-----------------------------------------------------------------------

      do k = 1, nmol_6 + nmol_5 + nmol_3

         r5(1,1,k) = sqrt( 1.d0 / beta / physmass_mol(k) ) * gasdev()
         r5(1,2,k) = sqrt( 1.d0 / beta / physmass_mol(k) ) * gasdev()
         r5(1,3,k) = sqrt( 1.d0 / beta / physmass_mol(k) ) * gasdev()

      end do

      r5(2,:,:) = 0.d0
      r5(3,:,:) = 0.d0
      r5(4,:,:) = 0.d0
      r5(5,:,:) = 0.d0

!-----------------------------------------------------------------------
!     //   randomize quaternion
!-----------------------------------------------------------------------

      q4(1,:,:) = 0.d0
      q4(2,:,:) = 0.d0
      q4(3,:,:) = 0.d0
      q4(4,:,:) = 0.d0

!      do k = 1, nmol_6 + nmol_5
!
!        tht = ranf1() * pi
!        phi = ranf1() * 2.d0 * pi
!        psi = ranf1() * 2.d0 * pi
!
!        q4(0,1,k) = sin( 0.5d0 * tht ) * sin( 0.5d0 * ( psi - phi ) )
!        q4(0,2,k) = sin( 0.5d0 * tht ) * cos( 0.5d0 * ( psi - phi ) )
!        q4(0,3,k) = cos( 0.5d0 * tht ) * sin( 0.5d0 * ( psi + phi ) )
!        q4(0,4,k) = cos( 0.5d0 * tht ) * cos( 0.5d0 * ( psi + phi ) )
!
!      end do

!-----------------------------------------------------------------------
!     //   maxwell distribution of angular velocity
!-----------------------------------------------------------------------

      o4(:,:,:) = 0.d0

      do k = 1, nmol_6

         o4(0,1,k) = sqrt( 1.d0 / beta / pmi(1,k) ) * gasdev()
         o4(0,2,k) = sqrt( 1.d0 / beta / pmi(2,k) ) * gasdev()
         o4(0,3,k) = sqrt( 1.d0 / beta / pmi(3,k) ) * gasdev()

      end do

      do k = nmol_6 + 1, nmol_6 + nmol_5

         o4(0,1,k) = sqrt( 1.d0 / beta / pmi(1,k) ) * gasdev()
         o4(0,2,k) = sqrt( 1.d0 / beta / pmi(2,k) ) * gasdev()

      end do

!-----------------------------------------------------------------------
!     //   transfer from omega to qdot
!-----------------------------------------------------------------------

      call trans_o_q1

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine opt_frame_rotor
!***********************************************************************
!=======================================================================
!
!     this routine optimizes the frame in laboratory axis
!     and compute quaternion
!
!=======================================================================
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, physmass

      use rotor_variables, only : &
     &   r5, q4, x_mol, y_mol, z_mol, physmass_mol, nmol, &
     &   nmol_6, nmol_5, natom_per_mol, list_atom_mol

      use lbfgs_variables, only : &
     &   pos, pos0, dm, grad, ws, frms, fmax, drms, dmax, drms_tol, &
     &   dmax_tol, fmax_tol, frms_tol, eps, func, stpmax, postol, &
     &   iprint, iflag, ndim, nup, nwork

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   iteration
      integer :: iter

!     //   maximum number of iterations
      integer :: maxiter = 1000

!     //   atomic position
      real(8) :: xt, yt, zt

!     //   euler angle
      real(8) :: phi, tht, psi

!     //   derivatives
      real(8) :: dxt(3), dyt(3), dzt(3)

!     //   rotation matrix
      real(8) :: a(3,3)

!     //   derivatives of rotation matrix
      real(8) :: a1(3,3), a2(3,3), a3(3,3)

!     //   integers
      integer :: i, j, k, l

!-----------------------------------------------------------------------
!     //   allocation
!-----------------------------------------------------------------------

!     //   number of dimensions - three euler angles
      ndim  = 3

!     //   dimension of the workspace in lbfgs
      nwork = ndim*(2*nup+1)+2*nup

!     //   euler angles
      if ( .not. allocated( pos ) ) &
     &   allocate( pos(ndim) )

!     //   euler angles
      if ( .not. allocated( pos0 ) ) &
     &   allocate( pos0(ndim) )

!     //   diagonal matrix in lbfgs
      if ( .not. allocated( dm ) ) &
     &   allocate( dm(ndim) )

!     //   gradient
      if ( .not. allocated( grad ) ) &
     &   allocate( grad(ndim) )

!     //   workspace in lbfgs
      if ( .not. allocated( ws ) ) &
     &   allocate( ws(nwork) )

!-----------------------------------------------------------------------
!     //   molecular center of mass
!-----------------------------------------------------------------------

      do k = 1, nmol

         r5(0,1,k) = 0.d0
         r5(0,2,k) = 0.d0
         r5(0,3,k) = 0.d0

         do j = 1, natom_per_mol(k)

            i = list_atom_mol(j,k)

            r5(0,1,k) = r5(0,1,k) + physmass(i) * x(i,1)
            r5(0,2,k) = r5(0,2,k) + physmass(i) * y(i,1)
            r5(0,3,k) = r5(0,3,k) + physmass(i) * z(i,1)

         end do

         r5(0,1,k) = r5(0,1,k) / physmass_mol(k)
         r5(0,2,k) = r5(0,2,k) / physmass_mol(k)
         r5(0,3,k) = r5(0,3,k) / physmass_mol(k)

      end do

!-----------------------------------------------------------------------
!     //   loop of molecules
!-----------------------------------------------------------------------

      do k = 1, nmol_6 + nmol_5

!-----------------------------------------------------------------------
!        //   reset euler angles
!-----------------------------------------------------------------------

         phi = 0.001d0
         tht = 0.002d0
         psi = 0.003d0

         pos(1) = phi
         pos(2) = tht
         pos(3) = psi

!-----------------------------------------------------------------------
!        //   loop of iteration
!-----------------------------------------------------------------------

         do iter = 1, maxiter

!-----------------------------------------------------------------------
!           //   euler angles
!-----------------------------------------------------------------------

            phi = pos(1)
            tht = pos(2)
            psi = pos(3)

!-----------------------------------------------------------------------
!           //   preserve euler angles
!-----------------------------------------------------------------------

            pos0(:) = pos(:)

!           //   a: rotation matrix

            a(1,1) = + cos(psi)*cos(phi) - cos(tht)*sin(phi)*sin(psi)
            a(1,2) = + cos(psi)*sin(phi) + cos(tht)*cos(phi)*sin(psi)
            a(1,3) = + sin(psi)*sin(tht)
            a(2,1) = - sin(psi)*cos(phi) - cos(tht)*sin(phi)*cos(psi)
            a(2,2) = - sin(psi)*sin(phi) + cos(tht)*cos(phi)*cos(psi)
            a(2,3) = + cos(psi)*sin(tht)
            a(3,1) = + sin(tht)*sin(phi)
            a(3,2) = - sin(tht)*cos(phi)
            a(3,3) = + cos(tht)

!-----------------------------------------------------------------------
!           //   a1: derivative of a with respect to phi
!-----------------------------------------------------------------------

            a1(1,1) = - cos(psi)*sin(phi) - cos(tht)*cos(phi)*sin(psi)
            a1(1,2) = + cos(psi)*cos(phi) - cos(tht)*sin(phi)*sin(psi)
            a1(1,3) = + 0.d0
            a1(2,1) = + sin(psi)*sin(phi) - cos(tht)*cos(phi)*cos(psi)
            a1(2,2) = - sin(psi)*cos(phi) - cos(tht)*sin(phi)*cos(psi)
            a1(2,3) = + 0.d0
            a1(3,1) = + sin(tht)*cos(phi)
            a1(3,2) = + sin(tht)*sin(phi)
            a1(3,3) = + 0.d0

!-----------------------------------------------------------------------
!           //   a2: derivative of a with respect to tht
!-----------------------------------------------------------------------

            a2(1,1) = + sin(tht)*sin(phi)*sin(psi)
            a2(1,2) = - sin(tht)*cos(phi)*sin(psi)
            a2(1,3) = + sin(psi)*cos(tht)
            a2(2,1) = + sin(tht)*sin(phi)*cos(psi)
            a2(2,2) = - sin(tht)*cos(phi)*cos(psi)
            a2(2,3) = + cos(psi)*cos(tht)
            a2(3,1) = + cos(tht)*sin(phi)
            a2(3,2) = - cos(tht)*cos(phi)
            a2(3,3) = - sin(tht)

!-----------------------------------------------------------------------
!           //   a3: derivative of a with respect to psi
!-----------------------------------------------------------------------

            a3(1,1) = - sin(psi)*cos(phi) - cos(tht)*sin(phi)*cos(psi)
            a3(1,2) = - sin(psi)*sin(phi) + cos(tht)*cos(phi)*cos(psi)
            a3(1,3) = + cos(psi)*sin(tht)
            a3(2,1) = - cos(psi)*cos(phi) + cos(tht)*sin(phi)*sin(psi)
            a3(2,2) = - cos(psi)*sin(phi) - cos(tht)*cos(phi)*sin(psi)
            a3(2,3) = - sin(psi)*sin(tht)
            a3(3,1) = + 0.d0
            a3(3,2) = + 0.d0
            a3(3,3) = + 0.d0

!-----------------------------------------------------------------------
!           //   func: target function, which characterizes
!           //         the mass-weighted deviation of
!           //         xt, yt, zt:  atoms in rotated molecular frame
!           //         and
!           //         x, y, z:     atoms in laboratory frame
!-----------------------------------------------------------------------

            func    = 0.d0

!-----------------------------------------------------------------------
!           //   gradient of func with respect to phi, tht and psi
!-----------------------------------------------------------------------

            grad(1) = 0.d0
            grad(2) = 0.d0
            grad(3) = 0.d0

!-----------------------------------------------------------------------
!           //   loop of atoms
!-----------------------------------------------------------------------

            do l = 1, natom_per_mol(k)

!-----------------------------------------------------------------------
!              //  atom number
!-----------------------------------------------------------------------

               j = list_atom_mol(l,k)

!-----------------------------------------------------------------------
!              //  atomic position
!-----------------------------------------------------------------------

               xt = r5(0,1,k) + a(1,1) * x_mol(l,k) &
     &                        + a(2,1) * y_mol(l,k) &
     &                        + a(3,1) * z_mol(l,k)
               yt = r5(0,2,k) + a(1,2) * x_mol(l,k) &
     &                        + a(2,2) * y_mol(l,k) &
     &                        + a(3,2) * z_mol(l,k)
               zt = r5(0,3,k) + a(1,3) * x_mol(l,k) &
     &                        + a(2,3) * y_mol(l,k) &
     &                        + a(3,3) * z_mol(l,k)

!-----------------------------------------------------------------------
!              //   derivatives of xt, yt and zt with respect to phi
!-----------------------------------------------------------------------

               dxt(1) = a1(1,1) * x_mol(l,k) &
     &                + a1(2,1) * y_mol(l,k) &
     &                + a1(3,1) * z_mol(l,k)
               dyt(1) = a1(1,2) * x_mol(l,k) &
     &                + a1(2,2) * y_mol(l,k) &
     &                + a1(3,2) * z_mol(l,k)
               dzt(1) = a1(1,3) * x_mol(l,k) &
     &                + a1(2,3) * y_mol(l,k) &
     &                + a1(3,3) * z_mol(l,k)

!-----------------------------------------------------------------------
!              //   derivatives of xt, yt and zt with respect to tht
!-----------------------------------------------------------------------

               dxt(2) = a2(1,1) * x_mol(l,k) &
     &                + a2(2,1) * y_mol(l,k) &
     &                + a2(3,1) * z_mol(l,k)
               dyt(2) = a2(1,2) * x_mol(l,k) &
     &                + a2(2,2) * y_mol(l,k) &
     &                + a2(3,2) * z_mol(l,k)
               dzt(2) = a2(1,3) * x_mol(l,k) &
     &                + a2(2,3) * y_mol(l,k) &
     &                + a2(3,3) * z_mol(l,k)

!-----------------------------------------------------------------------
!              //   derivatives of xt, yt and zt with respect to psi
!-----------------------------------------------------------------------

               dxt(3) = a3(1,1) * x_mol(l,k) &
     &                + a3(2,1) * y_mol(l,k) &
     &                + a3(3,1) * z_mol(l,k)
               dyt(3) = a3(1,2) * x_mol(l,k) &
     &                + a3(2,2) * y_mol(l,k) &
     &                + a3(3,2) * z_mol(l,k)
               dzt(3) = a3(1,3) * x_mol(l,k) &
     &                + a3(2,3) * y_mol(l,k) &
     &                + a3(3,3) * z_mol(l,k)

!-----------------------------------------------------------------------
!              //   target function
!-----------------------------------------------------------------------

               func = func + physmass(j) &
     &            * ( ( x(j,1) - xt ) * ( x(j,1) - xt ) &
     &              + ( y(j,1) - yt ) * ( y(j,1) - yt ) &
     &              + ( z(j,1) - zt ) * ( z(j,1) - zt ) )

!-----------------------------------------------------------------------
!              //   gradient with respect to phi
!-----------------------------------------------------------------------

               grad(1) = grad(1) - 2.d0 * physmass(j) &
     &            * ( ( x(j,1) - xt ) * dxt(1) &
     &              + ( y(j,1) - yt ) * dyt(1) &
     &              + ( z(j,1) - zt ) * dzt(1) )

!-----------------------------------------------------------------------
!              //   gradient with respect to tht
!-----------------------------------------------------------------------

               grad(2) = grad(2) - 2.d0 * physmass(j) &
     &            * ( ( x(j,1) - xt ) * dxt(2) &
     &              + ( y(j,1) - yt ) * dyt(2) &
     &              + ( z(j,1) - zt ) * dzt(2) )

!-----------------------------------------------------------------------
!              //   gradient with respect to psi
!-----------------------------------------------------------------------

               grad(3) = grad(3) - 2.d0 * physmass(j) &
     &            * ( ( x(j,1) - xt ) * dxt(3) &
     &              + ( y(j,1) - yt ) * dyt(3) &
     &              + ( z(j,1) - zt ) * dzt(3) )

!-----------------------------------------------------------------------
!           //   loop of atoms
!-----------------------------------------------------------------------

            end do

!-----------------------------------------------------------------------
!           //   call limited memory bfgs routine
!-----------------------------------------------------------------------

            stpmax = 1.d0

            call lbfgs ( ndim, nup, pos, func, grad, .false., dm, &
     &                   iprint, eps, postol, ws, iflag )

!-----------------------------------------------------------------------
!           //   check convergence:  root-mean-square of residual force
!-----------------------------------------------------------------------

            frms = 0.d0

            do j = 1, ndim
               frms = frms + grad(j)*grad(j)
            end do

            frms = sqrt(frms)/ndim

!-----------------------------------------------------------------------
!           //   check convergence:  maximum value of residual force
!-----------------------------------------------------------------------

            fmax = 0.d0

            do j = 1, ndim
               fmax = max ( abs( grad(j) ), fmax )
            end do

!-----------------------------------------------------------------------
!           //   check convergence:  root-mean-square of shift
!-----------------------------------------------------------------------

            drms = 0.d0

            do j = 1, ndim
               drms = drms + (pos(j)-pos0(j))*(pos(j)-pos0(j))
            end do

!-----------------------------------------------------------------------
!           //   check convergence:  maximum value of shift
!-----------------------------------------------------------------------

            dmax = 0.d0

            do j = 1, ndim
               dmax = max ( abs( pos(j)-pos0(j) ), dmax )
            end do

!-----------------------------------------------------------------------
!           //   terminate if converged
!-----------------------------------------------------------------------

            if ( ( dmax .lt. dmax_tol ) .and. &
     &           ( drms .lt. drms_tol ) .and. &
     &           ( fmax .lt. fmax_tol ) .and. &
     &           ( frms .lt. frms_tol ) ) then
               iflag = 0
            end if

            if ( iflag .eq. 0 ) exit
            if ( iflag .lt. 0 ) continue

!-----------------------------------------------------------------------
!        //   loop of iteration
!-----------------------------------------------------------------------

         end do

!-----------------------------------------------------------------------
!        //   quaternion
!-----------------------------------------------------------------------

         q4(0,1,k) = sin(0.5d0*tht) * sin(0.5d0*(psi-phi))
         q4(0,2,k) = sin(0.5d0*tht) * cos(0.5d0*(psi-phi))
         q4(0,3,k) = cos(0.5d0*tht) * sin(0.5d0*(psi+phi))
         q4(0,4,k) = cos(0.5d0*tht) * cos(0.5d0*(psi+phi))

!-----------------------------------------------------------------------
!     //   loop of molecules
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     //   coordinate associated to the frame
!-----------------------------------------------------------------------

      call trans_position_rotor

      return
      end





!***********************************************************************
      subroutine torque_test_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    fx, fy, fz, x, y, z, iounit, myrank

      use rotor_variables, only : &
     &    x_mol, y_mol, z_mol, r5, q4, &
     &    nmol_6, nmol_5, natom_per_mol, list_atom_mol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     //   integers
      integer :: i, k, l, m

!     //   real numbers
      real(8) :: tx, ty, tz, t2, sx, sy, sz, a(3,3), fxm, fym, fzm
      real(8) :: txl, tyl, tzl, txm, tym, tzm, txg, tyg, tzg

!-----------------------------------------------------------------------
!     //   master only
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     //   open file
!-----------------------------------------------------------------------

      open( iounit, file = 'torque_test.out' )

!-----------------------------------------------------------------------

      tx = 0.d0
      ty = 0.d0
      tz = 0.d0

      do k = 1, nmol_6 + nmol_5

         do l = 1, natom_per_mol(k)

            i = list_atom_mol(l,k)

            sx = x(i,1)
            sy = y(i,1)
            sz = z(i,1)

            tx = tx + sy*fz(i,1) - sz*fy(i,1)
            ty = ty + sz*fx(i,1) - sx*fz(i,1)
            tz = tz + sx*fy(i,1) - sy*fx(i,1)

         end do

      end do

      t2 = tx*tx + ty*ty + tz*tz

      write( iounit, '(a,4e16.8)' ) &
     &   'Total torque in laboratory origin:', tx, ty, tz, t2

!-----------------------------------------------------------------------

      txg = 0.d0
      tyg = 0.d0
      tzg = 0.d0

      do k = 1, nmol_6 + nmol_5

         sx = r5(0,1,k)
         sy = r5(0,2,k)
         sz = r5(0,3,k)

         fxm = 0.d0
         fym = 0.d0
         fzm = 0.d0

         do l = 1, natom_per_mol(k)

            i = list_atom_mol(l,k)

            fxm = fxm + fx(i,1)
            fym = fym + fy(i,1)
            fzm = fzm + fz(i,1)

         end do

         txg = txg + sy*fzm - sz*fym
         tyg = tyg + sz*fxm - sx*fzm
         tzg = tzg + sx*fym - sy*fxm

      end do

!-----------------------------------------------------------------------

      tx = 0.d0
      ty = 0.d0
      tz = 0.d0

      do k = 1, nmol_6 + nmol_5

         do l = 1, natom_per_mol(k)

            i = list_atom_mol(l,k)

            sx = x(i,1) - r5(0,1,k)
            sy = y(i,1) - r5(0,2,k)
            sz = z(i,1) - r5(0,3,k)

            tx = tx + sy*fz(i,1) - sz*fy(i,1)
            ty = ty + sz*fx(i,1) - sx*fz(i,1)
            tz = tz + sx*fy(i,1) - sy*fx(i,1)

         end do

      end do

      tx = tx + txg
      ty = ty + tyg
      tz = tz + tzg

      t2 = tx*tx + ty*ty + tz*tz

      write(iounit, '(a,4e16.8)' ) &
     &   'Torque around molecular origin:  ', tx, ty, tz, t2

!-----------------------------------------------------------------------

      tx = 0.d0
      ty = 0.d0
      tz = 0.d0

      txl = 0.d0
      tyl = 0.d0
      tzl = 0.d0

      do k = 1, nmol_6 + nmol_5

         a(1,1) =   2.d0 * ( q4(0,2,k) * q4(0,2,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(1,2) =   2.d0 * ( q4(0,3,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,2,k) )
         a(1,3) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     + q4(0,1,k) * q4(0,4,k) )
         a(2,1) = - 2.d0 * ( q4(0,1,k) * q4(0,2,k) &
     &                     + q4(0,3,k) * q4(0,4,k) )
         a(2,2) =   2.d0 * ( q4(0,1,k) * q4(0,1,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(2,3) =   2.d0 * ( q4(0,2,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,3,k) )
         a(3,1) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     - q4(0,1,k) * q4(0,4,k) )
         a(3,2) = - 2.d0 * ( q4(0,1,k) * q4(0,3,k) &
     &                     + q4(0,2,k) * q4(0,4,k) )
         a(3,3) =   2.d0 * ( q4(0,3,k) * q4(0,3,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0

         txm = 0.d0
         tym = 0.d0
         tzm = 0.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            tx = tx + y_mol(l,k) * ( fx(m,1)*a(3,1) &
     &                             + fy(m,1)*a(3,2) &
     &                             + fz(m,1)*a(3,3) ) &
     &              - z_mol(l,k) * ( fx(m,1)*a(2,1) &
     &                             + fy(m,1)*a(2,2) &
     &                             + fz(m,1)*a(2,3) )

            ty = ty + z_mol(l,k) * ( fx(m,1)*a(1,1) &
     &                             + fy(m,1)*a(1,2) &
     &                             + fz(m,1)*a(1,3) ) &
     &              - x_mol(l,k) * ( fx(m,1)*a(3,1) &
     &                             + fy(m,1)*a(3,2) &
     &                             + fz(m,1)*a(3,3) )

            tz = tz + x_mol(l,k) * ( fx(m,1)*a(2,1) &
     &                             + fy(m,1)*a(2,2) &
     &                             + fz(m,1)*a(2,3) ) &
     &              - y_mol(l,k) * ( fx(m,1)*a(1,1) &
     &                             + fy(m,1)*a(1,2) &
     &                             + fz(m,1)*a(1,3) )

            txm = txm + y_mol(l,k) * ( fx(m,1)*a(3,1) &
     &                               + fy(m,1)*a(3,2) &
     &                               + fz(m,1)*a(3,3) ) &
     &                - z_mol(l,k) * ( fx(m,1)*a(2,1) &
     &                               + fy(m,1)*a(2,2) &
     &                               + fz(m,1)*a(2,3) )

            tym = tym + z_mol(l,k) * ( fx(m,1)*a(1,1) &
     &                               + fy(m,1)*a(1,2) &
     &                               + fz(m,1)*a(1,3) ) &
     &                - x_mol(l,k) * ( fx(m,1)*a(3,1) &
     &                               + fy(m,1)*a(3,2) &
     &                               + fz(m,1)*a(3,3) )

            tzm = tzm + x_mol(l,k) * ( fx(m,1)*a(2,1) &
     &                               + fy(m,1)*a(2,2) &
     &                               + fz(m,1)*a(2,3) ) &
     &                - y_mol(l,k) * ( fx(m,1)*a(1,1) &
     &                               + fy(m,1)*a(1,2) &
     &                               + fz(m,1)*a(1,3) )

         end do

         txl = txl + txm*a(1,1)+tym*a(2,1)+tzm*a(3,1)
         tyl = tyl + txm*a(1,2)+tym*a(2,2)+tzm*a(3,2)
         tzl = tzl + txm*a(1,3)+tym*a(2,3)+tzm*a(3,3)

      end do

      t2 = tx*tx + ty*ty + tz*tz

      write(iounit, '(a,4e16.8)' ) &
     &   'Torque around molecular origin:  ', tx, ty, tz, t2

      txl = txl + txg
      tyl = tyl + tyg
      tzl = tzl + tzg

      t2 = txl*txl + tyl*tyl + tzl*tzl

      write(iounit, '(a,4e16.8)' ) &
     &   'Total torque:                    ', txl, tyl, tzl, t2

!-----------------------------------------------------------------------

      tx = 0.d0
      ty = 0.d0
      tz = 0.d0

      txm = 0.d0
      tym = 0.d0
      tzm = 0.d0

      do k = 1, nmol_6 + nmol_5

         a(1,1) =   2.d0 * ( q4(0,2,k) * q4(0,2,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(1,2) =   2.d0 * ( q4(0,3,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,2,k) )
         a(1,3) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     + q4(0,1,k) * q4(0,4,k) )
         a(2,1) = - 2.d0 * ( q4(0,1,k) * q4(0,2,k) &
     &                     + q4(0,3,k) * q4(0,4,k) )
         a(2,2) =   2.d0 * ( q4(0,1,k) * q4(0,1,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0
         a(2,3) =   2.d0 * ( q4(0,2,k) * q4(0,4,k) &
     &                     - q4(0,1,k) * q4(0,3,k) )
         a(3,1) =   2.d0 * ( q4(0,2,k) * q4(0,3,k) &
     &                     - q4(0,1,k) * q4(0,4,k) )
         a(3,2) = - 2.d0 * ( q4(0,1,k) * q4(0,3,k) &
     &                     + q4(0,2,k) * q4(0,4,k) )
         a(3,3) =   2.d0 * ( q4(0,3,k) * q4(0,3,k) &
     &                     + q4(0,4,k) * q4(0,4,k) ) - 1.d0

         do l = 1, natom_per_mol(k)

            m = list_atom_mol(l,k)

            sx = x_mol(l,k)*a(1,1)+y_mol(l,k)*a(2,1)+z_mol(l,k)*a(3,1)
            sy = x_mol(l,k)*a(1,2)+y_mol(l,k)*a(2,2)+z_mol(l,k)*a(3,2)
            sz = x_mol(l,k)*a(1,3)+y_mol(l,k)*a(2,3)+z_mol(l,k)*a(3,3)

            tx = tx + sy*fz(m,1) - sz*fy(m,1)
            ty = ty + sz*fx(m,1) - sx*fz(m,1)
            tz = tz + sx*fy(m,1) - sy*fx(m,1)

            txl = sy*fz(m,1) - sz*fy(m,1)
            tyl = sz*fx(m,1) - sx*fz(m,1)
            tzl = sx*fy(m,1) - sy*fx(m,1)

            txm = txm + txl*a(1,1)+tyl*a(1,2)+tzl*a(1,3)
            tym = tym + txl*a(2,1)+tyl*a(2,2)+tzl*a(2,3)
            tzm = tzm + txl*a(3,1)+tyl*a(3,2)+tzl*a(3,3)

         end do

      end do

      t2 = txm*txm + tym*tym + tzm*tzm

      write(iounit, '(a,4e16.8)' ) &
     &   'Torque around molecular origin:  ', txm, tym, tzm, t2

      tx = tx + txg
      ty = ty + tyg
      tz = tz + tzg

      t2 = tx*tx + ty*ty + tz*tz

      write(iounit, '(a,4e16.8)' ) &
     &   'Total torque:                    ', tx, ty, tz, t2

!-----------------------------------------------------------------------
!     //   close file
!-----------------------------------------------------------------------

      close( iounit )

!-----------------------------------------------------------------------

      return
      end
