!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     restart metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine restart_bath_tass_MPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :   write
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, qmass, &
     &   natom, nbead, nnhc, ncolor, iounit, myrank

      use tass_variables, only : &
     &   rbath_tass, qmass_tass, vbath_tass, ntass

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k, m

      real(8):: pbath, pxbath, pybath, pzbath

!-----------------------------------------------------------------------
!     /*   read position and momentum                                 */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

         open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   thermostat attached to system   */

            do i = 1, nbead
            do j = 1, nnhc
            do k = 1, natom
               read(iounit,*) &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               read(iounit,*) &
     &         pxbath,       pybath,       pzbath
               vxbath(k,j,i) = pxbath/sqrt(qmass(i))
               vybath(k,j,i) = pybath/sqrt(qmass(i))
               vzbath(k,j,i) = pzbath/sqrt(qmass(i))
            end do
            end do
            end do

!           /*   thermostat attached to cv   */

            do k = 1, nbead
            do m = 1, ncolor
            do j = 1, nnhc
            do i = 1, ntass
               read(iounit,*) rbath_tass(i,j,m,k), pbath
               vbath_tass(i,j,m,k) = pbath/sqrt(qmass_tass(m))
            end do
            end do
            end do
            end do

         close( iounit )

!        /*   master process only   */
         end if

!        /*   mpi communication   */
         call my_mpi_bcast_real_3(  xbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3(  ybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3(  zbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3( vxbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3( vybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3( vzbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_4( vbath_tass,ntass,nnhc,ncolor,nbead )

!-----------------------------------------------------------------------
!     /*   write                                                      */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

         open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   thermostat attached to system   */

            do i = 1, nbead
            do j = 1, nnhc
            do k = 1, natom
               pxbath = sqrt(qmass(i))*vxbath(k,j,i)
               pybath = sqrt(qmass(i))*vybath(k,j,i)
               pzbath = sqrt(qmass(i))*vzbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         pxbath,       pybath,       pzbath
            end do
            end do
            end do

!           /*   thermostat attached to cv   */

            do k = 1, nbead
            do m = 1, ncolor
            do j = 1, nnhc
            do i = 1, ntass
               pbath = sqrt(qmass_tass(m))*vbath_tass(i,j,m,k)
               write(iounit,'(3e24.16)') rbath_tass(i,j,m,k), pbath
            end do
            end do
            end do
            end do

         close( iounit )

!        /*   master process only   */
         end if

!        /*   sychronize all processes   */
         call my_mpi_barrier

      else

!        /*   check error   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_bath_tass_MPI', 32 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_cv_tass_MPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :   write
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead, iounit, myrank

      use tass_variables, only : &
     &   rtass, vtass, stass, ntass

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, m, n, ierr

!-----------------------------------------------------------------------
!     /*   default                                                    */
!-----------------------------------------------------------------------

      ierr = 0

!-----------------------------------------------------------------------
!     /*   read position and momentum                                 */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

         open ( iounit, file = 'cv.ini', status = 'unknown' )

!           /*   initial position of cv   */

            do m = 1, nbead
            do n = 1, ntass
               read ( iounit, *, iostat=ierr ) rtass(n,m), vtass(n,m)
            end do
            end do

         close( iounit )

!        /*   master process only   */
         end if

!        /*   mpi communication   */
         call my_mpi_bcast_real_2( rtass, ntass, nbead )
         call my_mpi_bcast_real_2( vtass, ntass, nbead )

!        /*   check error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine restart_cv_tass_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   write                                                      */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

         open ( iounit, file = 'cv.ini', status = 'unknown' )

!           /*   initial position of cv   */

            do m = 1, nbead
            do n = 1, ntass
               write( iounit, '(3e24.16)' ) &
     &            rtass(n,m), vtass(n,m), stass(n,m)
            end do
            end do

         close( iounit )

!        /*   master process only   */
         end if

!        /*   sychronize all processes   */
         call my_mpi_barrier

      else

!        /*   check error   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_cv_tass_MPI', 30 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_hills_tass_MPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :   write
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead, iounit, myrank

      use tass_variables, only : &
     &   pot_lifted, gh_meta, gw_meta, gc_meta, rtass, gc_meta_save, &
     &   icv_start, ng_meta, ntass, mg_meta, time_tass_save

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, m, n, ierr, idummy

!-----------------------------------------------------------------------
!     /*   default                                                    */
!-----------------------------------------------------------------------

      ierr = 0

!-----------------------------------------------------------------------
!     /*   read hills information                                     */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

!        //   ng_meta:  number of hills  //

         open ( iounit, file = 'hills.ini', status = 'unknown' )

            ng_meta = 0
            m = 0
            do
               do n = 1, ntass
                  read ( iounit, *, iostat=ierr ) idummy
               end do
               if ( ierr .ne. 0 ) then
                  exit
               else
                  m = m + 1
                  ng_meta = m
               end if
            end do

         close( iounit )

!        //   read hills   //

         ierr = 0

         open ( iounit, file = 'hills.ini', status = 'unknown' )

            do m = 1, ng_meta
            do n = 1, ntass
               read ( iounit, *, iostat=ierr ) &
     &            idummy, gh_meta(m), gw_meta(n,m), gc_meta(n,m)
            end do
            end do

         close( iounit )

!        /*   master process only   */
         end if

!        /*   mpi communication   */
         call my_mpi_bcast_int_0 ( ng_meta )
         call my_mpi_bcast_real_1( gh_meta, mg_meta )
         call my_mpi_bcast_real_2( gw_meta, ntass, mg_meta )
         call my_mpi_bcast_real_2( gc_meta, ntass, mg_meta )

!        /*   check error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine restart_hills_tass_MPI', 33 )

!-----------------------------------------------------------------------
!        /*   history data                                            */
!-----------------------------------------------------------------------

         if      ( icv_start .eq. 0 ) then

!           //   initialize history data   //

!           /*   potential lifted by gaussian   */
            pot_lifted = 0.d0

!           /*   last gaussian center   */

            do m = 1, nbead
            do n = 1, ntass
               gc_meta_save(n,m) = rtass(n,m)
            end do
            end do

         else if ( icv_start .eq. 1 ) then

!           /*   master process only   */
            if ( myrank .eq. 0 ) then

!           //   read history data to restart previous run   //

            open ( iounit, file = 'cv.ini', status = 'unknown' )

!              /*   read skip   */

               do m = 1, nbead
               do n = 1, ntass
                  read ( iounit, *, iostat=ierr )
               end do
               end do

!              /*   potential lifted by gaussian   */
               read ( iounit, *, iostat=ierr ) pot_lifted

!              /*   last gaussian center   */

               do m = 1, nbead
               do n = 1, ntass
                  read ( iounit, *, iostat=ierr ) gc_meta_save(n,m)
               end do
               end do

               do m = 1, nbead
                  read ( iounit, *, iostat=ierr ) time_tass_save(m)
               end do

            close ( iounit )

!           /*   master process only   */
            end if

!           /*   mpi communication   */
            call my_mpi_bcast_real_0( pot_lifted )
            call my_mpi_bcast_real_2( gc_meta_save, ntass, nbead )
            call my_mpi_bcast_real_1( time_tass_save, nbead )

         else if ( icv_start .eq. 2 ) then

            continue

         end if

!        /*   check error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine restart_hills_tass_MPI', 33 )

!-----------------------------------------------------------------------
!     /*   write history data                                         */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

         open ( iounit, file = 'cv.ini', access = 'append' )

!           /*   potential lifted by gaussian   */
            write( iounit, '(e24.16)' ) pot_lifted

!           /*   last gaussian center   */

            do m = 1, nbead
            do n = 1, ntass
               write( iounit, '(e24.16)' ) gc_meta_save(n,m)
            end do
            end do

            do m = 1, nbead
               write( iounit, '(e24.16)' ) time_tass_save(m)
            end do

         close( iounit )

!        /*   master process only   */
         end if

!        /*   sychronize all processes   */
         call my_mpi_barrier

      else

!        /*   check error   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_hills_tass_MPI', 33 )

      end if

      return
      end
