!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 19, 2020 by M. Shiga
!      Description:     restart box size
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine restart_box_rehmc ( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file     irw = 1 :   read box
!
!                                 irw = 2 :   write box
!
!                                 irw = 3 :   read box, boxdot
!
!                                 irw = 4 :   write box, boxdot
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   box, boxinv, boxdot, vbox, volume, vlog, vvol, boxdot_bead, &
     &   box_bead, vbox_bead, vlog_bead, vvol_bead, volume_bead, &
     &   boxinv_bead, istep_end, iounit, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer:: irw, i, j, k, l, ierr

!     /*   real numbers  */
      real(8) :: det3

!-----------------------------------------------------------------------
!     /*   read box                                                   */
!-----------------------------------------------------------------------

!     /*   option: irw == 1   */
      if ( irw .eq. 1 ) then

!        /*   open file   */
         open ( iounit, file = 'box.ini', status = 'unknown' )

!        /*   loop of beads   */
         do l = 1, nbead

!           /*   read data   */
            read ( iounit, *, iostat=ierr ) &
     &         k, box(1,1), box(1,2), box(1,3)
            read ( iounit, *, iostat=ierr ) &
     &         k, box(2,1), box(2,2), box(2,3)
            read ( iounit, *, iostat=ierr ) &
     &         k, box(3,1), box(3,2), box(3,3)

!           /*   detect error signal   */
            if ( ierr .ne. 0 ) exit

!           /*   inverse matrix of cell matrix   */
            call inv3 ( box, boxinv )

!           /*   volume   */
            volume  = det3( box )

!           /*   copy data   */
            box_bead(:,:,l) = box(:,:)
            boxinv_bead(:,:,l) = boxinv(:,:)
            volume_bead(l) = volume

!        /*   loop of beads   */
         end do

!        /*   check error   */
         call error_handling &
     &      ( ierr, 'subroutine restart_box_rehmc', 28 )

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   write box                                                  */
!-----------------------------------------------------------------------

!     /*   option: irw == 2   */
      else if ( irw .eq. 2 ) then

!        /*   last step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'box.ini', status = 'unknown' )

!        /*   loop of beads   */
         do l = 1, nbead

!           /*   copy data   */
            box(:,:) = box_bead(:,:,l)

!           /*   write data   */
            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, box(1,1), box(1,2), box(1,3)
            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, box(2,1), box(2,2), box(2,3)
            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, box(3,1), box(3,2), box(3,3)

!           /*   detect error signal   */
            if ( ierr .ne. 0 ) exit

!        /*   loop of beads   */
         end do

!        /*   check error   */
         call error_handling &
     &      ( ierr, 'subroutine restart_box_rehmc', 28 )

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   read box                                                   */
!-----------------------------------------------------------------------

!     /*   option: irw == 3   */
      else if ( irw .eq. 3 ) then

!        /*   open file   */
         open ( iounit, file = 'box.ini', status = 'unknown' )

!        /*   loop of beads   */
         do l = 1, nbead

!           /*   read data   */
            read( iounit, *, iostat=ierr ) &
     &         k, box(1,1), box(1,2), box(1,3)
            read( iounit, *, iostat=ierr ) &
     &         k, box(2,1), box(2,2), box(2,3)
            read( iounit, *, iostat=ierr ) &
     &         k, box(3,1), box(3,2), box(3,3)

!           /*   read data   */
            read( iounit, *, iostat=ierr ) &
     &         k, boxdot(1,1), boxdot(1,2), boxdot(1,3)
            read( iounit, *, iostat=ierr ) &
     &         k, boxdot(2,1), boxdot(2,2), boxdot(2,3)
            read( iounit, *, iostat=ierr ) &
     &         k, boxdot(3,1), boxdot(3,2), boxdot(3,3)

!           /*   detect error signal   */
            if ( ierr .ne. 0 ) exit

!           /*   inverse matrix of cell matrix   */
            call inv3 ( box, boxinv )

!           /*   volume   */
            volume  = det3( box )

!           /*   volume velocity   */
            vvol   = boxdot(1,1) * ( 3.d0*volume**(2.d0/3.d0) )

!           /*   logarithm of box velocity   */
            vlog   = boxdot(1,1) / volume**(1.d0/3.d0)

!           /*   logarithm of box velocity   */
            vbox(:,:) = 0.d0

!           /*   logarithm of box velocity   */
            do k = 1, 3
            do j = 1, 3
            do i = 1, 3
               vbox(i,j) = vbox(i,j) + boxdot(i,k)*boxinv(k,j)
            end do
            end do
            end do

!           /*   logarithm of box velocity   */
            do j = 1, 3-1
            do i = j+1, 3
               vbox(i,j) = 0.5d0 * ( vbox(i,j) + vbox(j,i) )
               vbox(j,i) = vbox(i,j)
            end do
            end do

!           /*   copy data   */
            box_bead(:,:,l) = box(:,:)
            boxinv_bead(:,:,l) = boxinv(:,:)
            boxdot_bead(:,:,l) = boxdot(:,:)
            volume_bead(l) = volume
            vvol_bead(l) = vvol
            vlog_bead(l) = vlog
            vbox_bead(:,:,l) = vbox(:,:)

!        /*   loop of beads   */
         end do

!        /*   check error   */
         call error_handling &
     &      ( ierr, 'subroutine restart_box_rehmc', 28 )

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   write box                                                  */
!-----------------------------------------------------------------------

!     /*   option: irw == 4   */
      else if ( irw .eq. 4 ) then

!        /*   last step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'box.ini', status = 'unknown' )

!        /*   loop of beads   */
         do l = 1, nbead

!           /*   copy data   */
            box(:,:) = box_bead(:,:,l)
            boxdot(:,:) = boxdot_bead(:,:,l)

            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, box(1,1), box(1,2), box(1,3)
            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, box(2,1), box(2,2), box(2,3)
            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, box(3,1), box(3,2), box(3,3)

            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, boxdot(1,1), boxdot(1,2), boxdot(1,3)
            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, boxdot(2,1), boxdot(2,2), boxdot(2,3)
            write( iounit, '(i8,3e24.16)', iostat=ierr ) &
     &         k, boxdot(3,1), boxdot(3,2), boxdot(3,3)

!           /*   detect error signal   */
            if ( ierr .ne. 0 ) exit

!        /*   loop of beads   */
         end do

!        /*   check error   */
         call error_handling &
     &      ( ierr, 'subroutine restart_box_rehmc', 28 )

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!        /*   error handling                                          */
!-----------------------------------------------------------------------

!     /*   otherwise   */
      else

!        /*   check error   */
         call error_handling ( 1, 'subroutine restart_box_rehmc', 28 )

!     /*   option   */
      end if

      return
      end
