!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 19, 2020 by M. Shiga
!      Description:     restart box size
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine restart_box ( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file     irw = 1 :   read box
!
!                                 irw = 2 :   write box
!
!                                 irw = 3 :   read box, boxdot
!
!                                 irw = 4 :   write box, boxdot
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   box, boxinv, boxdot, vbox, volume, vlog, vvol, &
     &   istep_end, iounit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer:: irw, i, j, k, ierr

!     /*   real numbers  */
      real(8) :: det3

!-----------------------------------------------------------------------
!     /*   read box                                                   */
!-----------------------------------------------------------------------

!     /*   option: irw == 1   */
      if ( irw .eq. 1 ) then

!        /*   open file   */
         open ( iounit, file = 'box.ini', status = 'unknown' )

!        /*   read data   */
         read(iounit,*,iostat=ierr) &
     &      k, box(1,1), box(1,2), box(1,3)
         read(iounit,*,iostat=ierr) &
     &      k, box(2,1), box(2,2), box(2,3)
         read(iounit,*,iostat=ierr) &
     &      k, box(3,1), box(3,2), box(3,3)

!        /*   close file   */
         close(iounit)

!        /*   check error   */
         call error_handling ( ierr, 'subroutine restart_box', 22 )

!        /*   inverse matrix of cell matrix   */
         call inv3 ( box, boxinv )

!        /*   volume   */
         volume  = det3( box )

!-----------------------------------------------------------------------
!     /*   write box                                                  */
!-----------------------------------------------------------------------

!     /*   option: irw == 2   */
      else if ( irw .eq. 2 ) then

!        /*   last step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'box.ini', status = 'unknown' )

!        /*   write data   */
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, box(1,1), box(1,2), box(1,3)
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, box(2,1), box(2,2), box(2,3)
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, box(3,1), box(3,2), box(3,3)

!        /*   close file   */
         close( iounit )

!        /*   check error   */
         call error_handling ( ierr, 'subroutine restart_box', 22 )

!-----------------------------------------------------------------------
!     /*   read box                                                   */
!-----------------------------------------------------------------------

!     /*   option: irw == 3   */
      else if ( irw .eq. 3 ) then

!        /*   open file   */
         open ( iounit, file = 'box.ini', status = 'unknown' )

!        /*   read data   */
         read(iounit,*,iostat=ierr) &
     &      k, box(1,1), box(1,2), box(1,3)
         read(iounit,*,iostat=ierr) &
     &      k, box(2,1), box(2,2), box(2,3)
         read(iounit,*,iostat=ierr) &
     &      k, box(3,1), box(3,2), box(3,3)

!        /*   read data   */
         read(iounit,*,iostat=ierr) &
     &      k, boxdot(1,1), boxdot(1,2), boxdot(1,3)
         read(iounit,*,iostat=ierr) &
     &      k, boxdot(2,1), boxdot(2,2), boxdot(2,3)
         read(iounit,*,iostat=ierr) &
     &      k, boxdot(3,1), boxdot(3,2), boxdot(3,3)

!        /*   close file   */
         close(iounit)

!        /*   check error   */
         call error_handling ( ierr, 'subroutine restart_box', 22 )

!        /*   inverse matrix of cell matrix   */
         call inv3 ( box, boxinv )

!        /*   volume   */
         volume  = det3( box )

!        /*   volume velocity   */
         vvol   = boxdot(1,1) * ( 3.d0*volume**(2.d0/3.d0) )

!        /*   logarithm of box velocity   */
         vlog   = boxdot(1,1) / volume**(1.d0/3.d0)

!        /*   logarithm of box velocity   */
         vbox(:,:) = 0.d0

!        /*   logarithm of box velocity   */
         do k = 1, 3
         do j = 1, 3
         do i = 1, 3
            vbox(i,j) = vbox(i,j) + boxdot(i,k)*boxinv(k,j)
         end do
         end do
         end do

!        /*   logarithm of box velocity   */
         do j = 1, 3-1
         do i = j+1, 3
            vbox(i,j) = 0.5d0 * ( vbox(i,j) + vbox(j,i) )
            vbox(j,i) = vbox(i,j)
         end do
         end do

!-----------------------------------------------------------------------
!     /*   write box                                                  */
!-----------------------------------------------------------------------

!     /*   option: irw == 4   */
      else if ( irw .eq. 4 ) then

!        /*   last step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'box.ini', status = 'unknown' )

!        /*   write data   */
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, box(1,1), box(1,2), box(1,3)
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, box(2,1), box(2,2), box(2,3)
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, box(3,1), box(3,2), box(3,3)

!        /*   write data   */
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, boxdot(1,1), boxdot(1,2), boxdot(1,3)
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, boxdot(2,1), boxdot(2,2), boxdot(2,3)
         write(iounit,'(i8,3e24.16)',iostat=ierr) &
     &      k, boxdot(3,1), boxdot(3,2), boxdot(3,3)

!        /*   close file   */
         close( iounit )

!        /*   check error   */
         call error_handling ( ierr, 'subroutine restart_box', 22 )

!-----------------------------------------------------------------------
!        /*   error handling                                          */
!-----------------------------------------------------------------------

!     /*   otherwise   */
      else

!        /*   check error   */
         call error_handling ( 1, 'subroutine restart_box', 22 )

!     /*   option   */
      end if

      return
      end
