!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     extensive MPI parallelization
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine restart_bath_mnhc_XMPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :   write
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, qmass, &
     &   xbath_cent, ybath_cent, zbath_cent, vxbath_cent, vybath_cent, &
     &   vzbath_cent, qmass_cent, iounit, nbead, nnhc, natom, ncolor, &
     &   myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k, m

      real(8):: pxbath, pybath, pzbath, &
     &          pxbath_cent, pybath_cent, pzbath_cent

!-----------------------------------------------------------------------
!     /*   read position and momentum                                 */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               read(iounit,*) &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               read(iounit,*) &
     &         pxbath,       pybath,       pzbath
               vxbath(k,j,i) = pxbath/sqrt(qmass(i))
               vybath(k,j,i) = pybath/sqrt(qmass(i))
               vzbath(k,j,i) = pzbath/sqrt(qmass(i))
            end do
            end do
            end do

!           /*   centroid   */

            do j = 1, natom
            do m = 1, ncolor
            do i = 1, nnhc
               read(iounit,*)  xbath_cent(j,i,m), ybath_cent(j,i,m), &
     &                         zbath_cent(j,i,m)
               read(iounit,*) &
     &         pxbath_cent, pybath_cent, pzbath_cent
               vxbath_cent(j,i,m) = pxbath_cent/sqrt(qmass_cent(i,m))
               vybath_cent(j,i,m) = pybath_cent/sqrt(qmass_cent(i,m))
               vzbath_cent(j,i,m) = pzbath_cent/sqrt(qmass_cent(i,m))
            end do
            end do
            end do

            close( iounit )

         end if

         call my_mpi_bcast_real_3 ( xbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( ybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( zbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

         call my_mpi_bcast_real_3 ( xbath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( ybath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( zbath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( vxbath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( vybath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( vzbath_cent, natom, nnhc, ncolor )

!-----------------------------------------------------------------------
!     /*   write                                                      */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

         call my_mpi_bcast_mnhc_mode_XMPI ( xbath, ybath, zbath, 1 )
         call my_mpi_bcast_mnhc_mode_XMPI ( xbath, ybath, zbath, 4 )

         call my_mpi_bcast_mnhc_mode_XMPI ( vxbath, vybath, vzbath, 1 )
         call my_mpi_bcast_mnhc_mode_XMPI ( vxbath, vybath, vzbath, 4 )

         call my_mpi_bcast_mnhc_cent_XMPI &
     &      ( xbath_cent, ybath_cent, zbath_cent, 1 )

         call my_mpi_bcast_mnhc_cent_XMPI &
     &      ( vxbath_cent, vybath_cent, vzbath_cent, 1 )

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               pxbath = sqrt(qmass(i))*vxbath(k,j,i)
               pybath = sqrt(qmass(i))*vybath(k,j,i)
               pzbath = sqrt(qmass(i))*vzbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         pxbath, pybath, pzbath
            end do
            end do
            end do

!           /*   centroid   */

            do j = 1, natom
            do m = 1, ncolor
            do i = 1, nnhc
               pxbath_cent = sqrt(qmass_cent(i,m))*vxbath_cent(j,i,m)
               pybath_cent = sqrt(qmass_cent(i,m))*vybath_cent(j,i,m)
               pzbath_cent = sqrt(qmass_cent(i,m))*vzbath_cent(j,i,m)
               write(iounit,'(3e24.16)') &
     &         xbath_cent(j,i,m), ybath_cent(j,i,m), zbath_cent(j,i,m)
               write(iounit,'(3e24.16)') &
     &         pxbath_cent, pybath_cent, pzbath_cent
            end do
            end do
            end do

            close( iounit )

         end if

         call my_mpi_barrier

      else

!        /*   error handling   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_bath_mnhc_MPI', 32 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_bath_nhc_XMPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :  write position and momentum
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, qmass, &
     &   rbath_cent, vbath_cent, qmass_cent, &
     &   iounit, nbead, nnhc, natom, ncolor, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k, m

      real(8):: pxbath, pybath, pzbath, pbath_cent

!-----------------------------------------------------------------------
!     /*   read position and momentum                                 */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               read(iounit,*) &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               read(iounit,*) &
     &         pxbath,       pybath,       pzbath
               vxbath(k,j,i) = pxbath/sqrt(qmass(i))
               vybath(k,j,i) = pybath/sqrt(qmass(i))
               vzbath(k,j,i) = pzbath/sqrt(qmass(i))
            end do
            end do
            end do

!           /*   centroid   */

            do m = 1, ncolor
            do i = 1, nnhc
               read(iounit,*) rbath_cent(i,m)
               read(iounit,*) pbath_cent
               vbath_cent(i,m) = pbath_cent/sqrt(qmass_cent(i,m))
            end do
            end do

            close( iounit )

         end if

         call my_mpi_bcast_real_3 ( xbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( ybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( zbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_2 ( rbath_cent, nnhc, ncolor )

         call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_2 ( vbath_cent, nnhc, ncolor )

!-----------------------------------------------------------------------
!     /*   write all                                                  */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

         call my_mpi_bcast_mnhc_mode_XMPI ( xbath, ybath, zbath, 1 )
         call my_mpi_bcast_mnhc_mode_XMPI ( xbath, ybath, zbath, 4 )

         call my_mpi_bcast_mnhc_mode_XMPI ( vxbath, vybath, vzbath, 1 )
         call my_mpi_bcast_mnhc_mode_XMPI ( vxbath, vybath, vzbath, 4 )

!!!      /*   WARNING - INCOMPLETE: So far only valid for NVE CMD   */
!!!      call my_mpi_bcast_nhc_cent_XMPI ( rbath_cent, 1 )
!!!      call my_mpi_bcast_nhc_cent_XMPI ( vbath_cent, 1 )

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

         open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               pxbath = sqrt(qmass(i))*vxbath(k,j,i)
               pybath = sqrt(qmass(i))*vybath(k,j,i)
               pzbath = sqrt(qmass(i))*vzbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         pxbath,       pybath,       pzbath
            end do
            end do
            end do

!           /*   centroid   */

            do m = 1, ncolor
            do i = 1, nnhc
               pbath_cent = sqrt(qmass_cent(i,m))*vbath_cent(i,m)
               write(iounit,'(e24.16)') rbath_cent(i,m)
               write(iounit,'(e24.16)') pbath_cent
            end do
            end do

            close( iounit )

         end if

         call my_mpi_barrier

      else

!        /*   error handling   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_bath_nhc_XMPI', 32 )

      end if

      return
      end
