!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     restart thermostats in path integrals
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine restart_bath_nhc_MPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :  write position and momentum
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, qmass, &
     &   rbath_cent, vbath_cent, qmass_cent, &
     &   iounit, nbead, nnhc, natom, ncolor, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k, m

      real(8):: pxbath, pybath, pzbath, pbath_cent

!-----------------------------------------------------------------------
!     /*   read position and momentum                                 */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               read(iounit,*) &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               read(iounit,*) &
     &         pxbath,       pybath,       pzbath
               vxbath(k,j,i) = pxbath/sqrt(qmass(i))
               vybath(k,j,i) = pybath/sqrt(qmass(i))
               vzbath(k,j,i) = pzbath/sqrt(qmass(i))
            end do
            end do
            end do

!           /*   centroid   */

            do m = 1, ncolor
            do i = 1, nnhc
               read(iounit,*) rbath_cent(i,m)
               read(iounit,*) pbath_cent
               vbath_cent(i,m) = pbath_cent/sqrt(qmass_cent(i,m))
            end do
            end do

            close( iounit )

         end if

         call my_mpi_bcast_real_3 ( xbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( ybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( zbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_2 ( rbath_cent, nnhc, ncolor )

         call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_2 ( vbath_cent, nnhc, ncolor )

!-----------------------------------------------------------------------
!     /*   write all                                                  */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

         open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               pxbath = sqrt(qmass(i))*vxbath(k,j,i)
               pybath = sqrt(qmass(i))*vybath(k,j,i)
               pzbath = sqrt(qmass(i))*vzbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         pxbath,       pybath,       pzbath
            end do
            end do
            end do

!           /*   centroid   */

            do m = 1, ncolor
            do i = 1, nnhc
               pbath_cent = sqrt(qmass_cent(i,m))*vbath_cent(i,m)
               write(iounit,'(e24.16)') rbath_cent(i,m)
               write(iounit,'(e24.16)') pbath_cent
            end do
            end do

            close( iounit )

         end if

         call my_mpi_barrier

      else

!        /*   error handling   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_bath_nhc_MPI', 31 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_bath_nhcs_MPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :   write
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, qmass, &
     &   xbath_cent, ybath_cent, zbath_cent, vxbath_cent, vybath_cent, &
     &   vzbath_cent, qmass_cent, iounit, nbead, nnhc, natom, ncolor, &
     &   myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k, m

      real(8):: pxbath, pybath, pzbath, &
     &          pxbath_cent, pybath_cent, pzbath_cent

!-----------------------------------------------------------------------
!     /*   read position and momentum                                 */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               read(iounit,*) &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               read(iounit,*) &
     &         pxbath,       pybath,       pzbath
               vxbath(k,j,i) = pxbath/sqrt(qmass(i))
               vybath(k,j,i) = pybath/sqrt(qmass(i))
               vzbath(k,j,i) = pzbath/sqrt(qmass(i))
            end do
            end do
            end do

!           /*   centroid   */

            do m = 1, ncolor
            do i = 1, nnhc
               read(iounit,*)  xbath_cent(1,i,m), ybath_cent(1,i,m), &
     &                         zbath_cent(1,i,m)
               read(iounit,*) &
     &         pxbath_cent,       pybath_cent,       pzbath_cent
               vxbath_cent(1,i,m) = pxbath_cent/sqrt(qmass_cent(i,m))
               vybath_cent(1,i,m) = pybath_cent/sqrt(qmass_cent(i,m))
               vzbath_cent(1,i,m) = pzbath_cent/sqrt(qmass_cent(i,m))
            end do
            end do

            close( iounit )

         end if

         call my_mpi_bcast_real_3 ( xbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( ybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( zbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( xbath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( ybath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( zbath_cent, natom, nnhc, ncolor )

         call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vxbath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( vybath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( vzbath_cent, natom, nnhc, ncolor )

!-----------------------------------------------------------------------
!     /*   write                                                      */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               pxbath = sqrt(qmass(i))*vxbath(k,j,i)
               pybath = sqrt(qmass(i))*vybath(k,j,i)
               pzbath = sqrt(qmass(i))*vzbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         pxbath,       pybath,       pzbath
            end do
            end do
            end do

!           /*   centroid   */

            do m = 1, ncolor
            do i = 1, nnhc
               pxbath_cent = sqrt(qmass_cent(i,m))*vxbath_cent(1,i,m)
               pybath_cent = sqrt(qmass_cent(i,m))*vybath_cent(1,i,m)
               pzbath_cent = sqrt(qmass_cent(i,m))*vzbath_cent(1,i,m)
               write(iounit,'(3e24.16)') &
     &         xbath_cent(1,i,m), ybath_cent(1,i,m), zbath_cent(1,i,m)
               write(iounit,'(3e24.16)') &
     &         pxbath_cent,       pybath_cent,       pzbath_cent
            end do
            end do

            close( iounit )

         end if

         call my_mpi_barrier

      else

!        /*   error handling   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_bath_nhcs_MPI', 32 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_bath_mnhc_MPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :   write
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, qmass, &
     &   xbath_cent, ybath_cent, zbath_cent, vxbath_cent, vybath_cent, &
     &   vzbath_cent, qmass_cent, qmass_box, rbath_box, xbath_box, &
     &   ybath_box, zbath_box, vbath_box, vxbath_box, vybath_box, &
     &   vzbath_box, iounit, nbead, nnhc, natom, ncolor, &
     &   myrank, npt_type, ntt_type, ensemble

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k, l, m

      real(8):: pxbath, pybath, pzbath, pbath_box, &
     &          pxbath_cent, pybath_cent, pzbath_cent, &
     &          pxbath_box, pybath_box, pzbath_box

!-----------------------------------------------------------------------
!     /*   read position and momentum                                 */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               read(iounit,*) &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               read(iounit,*) &
     &         pxbath,       pybath,       pzbath
               vxbath(k,j,i) = pxbath/sqrt(qmass(i))
               vybath(k,j,i) = pybath/sqrt(qmass(i))
               vzbath(k,j,i) = pzbath/sqrt(qmass(i))
            end do
            end do
            end do

!           /*   centroid   */

            do j = 1, natom
            do m = 1, ncolor
            do i = 1, nnhc
               read(iounit,*)  xbath_cent(j,i,m), ybath_cent(j,i,m), &
     &                         zbath_cent(j,i,m)
               read(iounit,*) &
     &         pxbath_cent, pybath_cent, pzbath_cent
               vxbath_cent(j,i,m) = pxbath_cent/sqrt(qmass_cent(i,m))
               vybath_cent(j,i,m) = pybath_cent/sqrt(qmass_cent(i,m))
               vzbath_cent(j,i,m) = pzbath_cent/sqrt(qmass_cent(i,m))
            end do
            end do
            end do

!           /*   box   */

            if ( ( ( ensemble(1:3) .eq. 'NPT' ) .and. &
     &             ( npt_type(1:6) .eq. 'CUBIC2' ) ) .or. &
     &           ( ( ensemble(1:3) .eq. 'NTT' ) .and. &
     &             ( ntt_type(1:6) .eq. 'CUBIC2' ) ) ) then

               do m = 1, ncolor
               do i = 1, nnhc
                  read(iounit,*) rbath_box(i,m)
                  read(iounit,*) pbath_box
                  vbath_box(i,m) = pbath_box/sqrt(qmass_box(i,m))
               end do
               end do

            end if

            if ( ( ( ensemble(1:3) .eq. 'NPT' ) .and. &
     &             ( npt_type(1:6) .eq. 'PPHEX ' ) ) .or. &
     &           ( ( ensemble(1:3) .eq. 'NTT' ) .and. &
     &             ( ntt_type(1:6) .eq. 'PPHEX ' ) ) ) then

               do m = 1, ncolor
               do i = 1, nnhc
               do j = 1, 3
                  read(iounit,*) &
     &            xbath_box(j,i,m), ybath_box(j,i,m), zbath_box(j,i,m)
                  read(iounit,*) &
     &            pxbath_box, pybath_box, pzbath_box
                  vxbath_box(j,i,m) = pxbath_box/sqrt(qmass_box(i,m))
                  vybath_box(j,i,m) = pybath_box/sqrt(qmass_box(i,m))
                  vzbath_box(j,i,m) = pzbath_box/sqrt(qmass_box(i,m))
               end do
               end do
               end do

               do l = 1, ncolor
               do i = 1, nnhc
                  vxbath_box(2,i,l) &
     &               = 0.5d0 * ( vxbath_box(2,i,l) + vybath_box(1,i,l) )
                  vybath_box(1,i,l) = vxbath_box(2,i,l)
                  vybath_box(3,i,l) &
     &               = 0.5d0 * ( vybath_box(3,i,l) + vzbath_box(2,i,l) )
                  vzbath_box(2,i,l) = vybath_box(3,i,l)
                  vzbath_box(1,i,l) &
     &               = 0.5d0 * ( vzbath_box(1,i,l) + vxbath_box(3,i,l) )
                  vxbath_box(3,i,l) = vzbath_box(1,i,l)
               end do
               end do

            end if

            close( iounit )

         end if

         call my_mpi_bcast_real_3 ( xbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( ybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( zbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

         call my_mpi_bcast_real_3 ( xbath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( ybath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( zbath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( vxbath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( vybath_cent, natom, nnhc, ncolor )
         call my_mpi_bcast_real_3 ( vzbath_cent, natom, nnhc, ncolor )

         if ( ( ( ensemble(1:3) .eq. 'NPT' ) .and. &
     &          ( npt_type(1:6) .eq. 'CUBIC2' ) ) .or. &
     &        ( ( ensemble(1:3) .eq. 'NTT' ) .and. &
     &          ( ntt_type(1:6) .eq. 'CUBIC2' ) ) ) then
            call my_mpi_bcast_real_2 ( rbath_box, nnhc, ncolor )
            call my_mpi_bcast_real_2 ( vbath_box, nnhc, ncolor )
         end if

         if ( ( ( ensemble(1:3) .eq. 'NPT' ) .and. &
     &          ( npt_type(1:6) .eq. 'PPHEX ' ) ) .or. &
     &        ( ( ensemble(1:3) .eq. 'NTT' ) .and. &
     &          ( ntt_type(1:6) .eq. 'PPHEX ' ) ) ) then
            call my_mpi_bcast_real_3 ( xbath_box, 3, nnhc, ncolor )
            call my_mpi_bcast_real_3 ( ybath_box, 3, nnhc, ncolor )
            call my_mpi_bcast_real_3 ( zbath_box, 3, nnhc, ncolor )
            call my_mpi_bcast_real_3 ( vxbath_box, 3, nnhc, ncolor )
            call my_mpi_bcast_real_3 ( vybath_box, 3, nnhc, ncolor )
            call my_mpi_bcast_real_3 ( vzbath_box, 3, nnhc, ncolor )
         end if

!-----------------------------------------------------------------------
!     /*   write                                                      */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 2, nbead
            do j = 1, nnhc
            do k = 1, natom
               pxbath = sqrt(qmass(i))*vxbath(k,j,i)
               pybath = sqrt(qmass(i))*vybath(k,j,i)
               pzbath = sqrt(qmass(i))*vzbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         pxbath, pybath, pzbath
            end do
            end do
            end do

!           /*   centroid   */

            do j = 1, natom
            do m = 1, ncolor
            do i = 1, nnhc
               pxbath_cent = sqrt(qmass_cent(i,m))*vxbath_cent(j,i,m)
               pybath_cent = sqrt(qmass_cent(i,m))*vybath_cent(j,i,m)
               pzbath_cent = sqrt(qmass_cent(i,m))*vzbath_cent(j,i,m)
               write(iounit,'(3e24.16)') &
     &         xbath_cent(j,i,m), ybath_cent(j,i,m), zbath_cent(j,i,m)
               write(iounit,'(3e24.16)') &
     &         pxbath_cent, pybath_cent, pzbath_cent
            end do
            end do
            end do

!           /*   box   */

            if ( ( ( ensemble(1:3) .eq. 'NPT' ) .and. &
     &             ( npt_type(1:6) .eq. 'CUBIC2' ) ) .or. &
     &           ( ( ensemble(1:3) .eq. 'NTT' ) .and. &
     &             ( ntt_type(1:6) .eq. 'CUBIC2' ) ) ) then

               do m = 1, ncolor
               do i = 1, nnhc
                  pbath_box = vbath_box(i,m)*sqrt(qmass_box(i,m))
                  write(iounit,'(e24.16)') rbath_box(i,m)
                  write(iounit,'(e24.16)') pbath_box
               end do
               end do

            end if

            if ( ( ( ensemble(1:3) .eq. 'NPT' ) .and. &
     &             ( npt_type(1:6) .eq. 'PPHEX ' ) ) .or. &
     &           ( ( ensemble(1:3) .eq. 'NTT' ) .and. &
     &             ( ntt_type(1:6) .eq. 'PPHEX ' ) ) ) then

               do m = 1, ncolor
               do i = 1, nnhc
               do j = 1, 3
                  pxbath_box = vxbath_box(j,i,m)*sqrt(qmass_box(i,m))
                  pybath_box = vybath_box(j,i,m)*sqrt(qmass_box(i,m))
                  pzbath_box = vzbath_box(j,i,m)*sqrt(qmass_box(i,m))
                  write(iounit,'(3e24.16)') &
     &            xbath_box(j,i,m), ybath_box(j,i,m), zbath_box(j,i,m)
                  write(iounit,'(3e24.16)') &
     &            pxbath_box, pybath_box, pzbath_box
               end do
               end do
               end do

            end if

            close( iounit )

         end if

         call my_mpi_barrier

      else

!        /*   error handling   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_bath_mnhc_MPI', 32 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_bath_mnhc_om_MPI( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file    irw = 1 :   read position and momentum
!                                irw = 2 :   write
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, qmass, &
     &   iounit, nbead, nnhc, natom, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k

      real(8):: pxbath, pybath, pzbath

!-----------------------------------------------------------------------
!     /*   read position and momentum                                 */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 1, nbead
            do j = 1, nnhc
            do k = 1, natom
               read(iounit,*) &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               read(iounit,*) &
     &         pxbath,       pybath,       pzbath
               vxbath(k,j,i) = pxbath/sqrt(qmass(i))
               vybath(k,j,i) = pybath/sqrt(qmass(i))
               vzbath(k,j,i) = pzbath/sqrt(qmass(i))
            end do
            end do
            end do

            close( iounit )

         end if

         call my_mpi_bcast_real_3 ( xbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( ybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( zbath, natom, nnhc, nbead )

         call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
         call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

!-----------------------------------------------------------------------
!     /*   write                                                      */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   only myrank = 0   */
         if ( myrank .eq. 0 ) then

            open ( iounit, file = 'bath.ini', status = 'unknown' )

!           /*   non-centroid   */

            do i = 1, nbead
            do j = 1, nnhc
            do k = 1, natom
               pxbath = sqrt(qmass(i))*vxbath(k,j,i)
               pybath = sqrt(qmass(i))*vybath(k,j,i)
               pzbath = sqrt(qmass(i))*vzbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         xbath(k,j,i), ybath(k,j,i), zbath(k,j,i)
               write(iounit,'(3e24.16)') &
     &         pxbath, pybath, pzbath
            end do
            end do
            end do

            close( iounit )

         end if

         call my_mpi_barrier

      else

!        /*   error handling   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_bath_mnhc_MPI', 32 )

      end if

      return
      end
