!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     extensive MPI parallelization
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine restart_velocity_XMPI ( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file     irw = 1 :   read normal mode momentum
!
!                                 irw = 2 :   read Cartesian velocity
!
!                                 irw = 3 :   write normal mode momentum
!
!                                 irw = 4 :   write Cartesian velocity
!
!     NOTE:  Only normal modes are updated inside mdcycle;
!            Cartesian velocity update is not ensured.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, vx, vy, vz, pux, puy, puz, fictmass, &
     &   x, y, z, ux, uy, uz, iounit, natom, nbead, istep_end, &
     &   mbox, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k

!-----------------------------------------------------------------------
!     /*   read normal mode coordinates and momenta                   */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

         if ( myrank .eq. 0 ) then

!           /*   open file   */
            open ( iounit, file = 'geometry.ini', status = 'unknown' )

!           /*   read   */
            do j = 1, nbead
            do i = 1, natom
               read( iounit, * ) &
     &            k,  ux(i,j),  uy(i,j),  uz(i,j), &
     &               pux(i,j), puy(i,j), puz(i,j), mbox(1:3,i,j)
            end do
            end do

!           /*   open file   */
            close( iounit )

!           /*   normal mode momentum -> normal mode velocity   */
            do j = 1, nbead
            do i = 1, natom
               vux(i,j) = pux(i,j)/sqrt(fictmass(i,j))
               vuy(i,j) = puy(i,j)/sqrt(fictmass(i,j))
               vuz(i,j) = puz(i,j)/sqrt(fictmass(i,j))
            end do
            end do

         end if

!        /*   mpi communication   */
         call my_mpi_bcast_real_2 ( vux, natom, nbead )
         call my_mpi_bcast_real_2 ( vuy, natom, nbead )
         call my_mpi_bcast_real_2 ( vuz, natom, nbead )
         call my_mpi_bcast_int_3  ( mbox, 3, natom, nbead )

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI( 0 )

!-----------------------------------------------------------------------
!     /*   read Cartesian coordinates and velocities                  */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

         if ( myrank .eq. 0 ) then

!           /*   open file   */
            open ( iounit, file = 'geometry.ini', status = 'unknown' )

!           /*   read   */
            do j = 1, nbead
            do i = 1, natom
               read( iounit, * ) &
     &            k,  x(i,j),  y(i,j),  z(i,j), &
     &               vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
            end do
            end do

!           /*   close file   */
            close( iounit )

         end if

!        /*   mpi communication   */
         call my_mpi_bcast_real_2 ( vx, natom, nbead )
         call my_mpi_bcast_real_2 ( vy, natom, nbead )
         call my_mpi_bcast_real_2 ( vz, natom, nbead )
         call my_mpi_bcast_int_3  ( mbox, 3, natom, nbead )

!        /*   Cartesian velocity -> normal mode velocity   */
         call nm_trans_velocity_MPI( 1 )

!-----------------------------------------------------------------------
!     /*   write normal mode coordinates and momenta                  */
!-----------------------------------------------------------------------

      else if ( irw .eq. 3 ) then

         call my_mpi_bcast_xyz_XMPI  ( ux, uy, uz, 1 )
         call my_mpi_bcast_xyz_XMPI  ( ux, uy, uz, 4 )

         call my_mpi_bcast_xyz_XMPI  ( vux, vuy, vuz, 1 )
         call my_mpi_bcast_xyz_XMPI  ( vux, vuy, vuz, 4 )

         if ( myrank .eq. 0 ) then

!           /*   current step   */
            k = istep_end

!           /*   normal mode velocity -> normal mode momentum   */
            do j = 1, nbead
            do i = 1, natom
               pux(i,j) = vux(i,j)*sqrt(fictmass(i,j))
               puy(i,j) = vuy(i,j)*sqrt(fictmass(i,j))
               puz(i,j) = vuz(i,j)*sqrt(fictmass(i,j))
            end do
            end do

!           /*   open file   */
            open ( iounit, file = 'geometry.ini', status = 'unknown' )

!           /*   write   */
            do j = 1, nbead
            do i = 1, natom
               write( iounit, '(i8,6e24.16,3i4)' ) &
     &            k,  ux(i,j),  uy(i,j),  uz(i,j), &
     &               pux(i,j), puy(i,j), puz(i,j), mbox(1:3,i,1)
!               write( iounit, '(i8,6e24.16,3i4)' ) &
!     &            k,  ux(i,j),  uy(i,j),  uz(i,j), &
!     &               pux(i,j), puy(i,j), puz(i,j), mbox(1:3,i,j)
            end do
            end do

!           /*   close file   */
            close( iounit )

         end if

!-----------------------------------------------------------------------
!     /*   write Cartesian coordinates and velocities                 */
!-----------------------------------------------------------------------

      else if ( irw .eq. 4 ) then

         call my_mpi_bcast_xyz_XMPI  ( ux, uy, uz, 1 )
         call my_mpi_bcast_xyz_XMPI  ( ux, uy, uz, 4 )

         call my_mpi_bcast_xyz_XMPI  ( vux, vuy, vuz, 1 )
         call my_mpi_bcast_xyz_XMPI  ( vux, vuy, vuz, 4 )

         if ( myrank .eq. 0 ) then

!           /*   normal mode position -> Cartesian position   */
            call nm_trans_MPI( 0 )

!           /*   normal mode velocity -> Cartesian velocity   */
            call nm_trans_velocity_MPI( 0 )

!           /*   current step   */
            k = istep_end

!           /*   open file   */
            open ( iounit, file = 'geometry.ini', status = 'unknown' )

            do j = 1, nbead
            do i = 1, natom
!              /*   write   */
               write( iounit, '(i8,6e24.16,3i4)' ) &
     &            k,  x(i,j),  y(i,j),  z(i,j), &
     &               vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,1)
!               write( iounit, '(i8,6e24.16,3i4)' ) &
!    &            k,  x(i,j),  y(i,j),  z(i,j), &
!    &               vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
            end do
            end do

!           /*   close file   */
            close( iounit )

         end if

      else

!        /*   error handling   */
         call error_handling_MPI &
     &      ( 1, 'subroutine restart_velocity_XMPI', 32 )

      end if

      return
      end
