!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 7, 2019 by M. Shiga
!      Description:     restart position and velocity
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine restart_position ( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file     irw = 1 :   read normal mode position
!
!                                 irw = 2 :   read Cartesian position
!
!                                 irw = 3 :   write normal mode position
!
!                                 irw = 4 :   write Cartesian position
!
!     NOTE:  Only normal modes are updated inside mdcycle;
!            Cartesian position updated is not ensured.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, x, y, z, iounit, natom, nbead, istep_end, mbox

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k

      real(8) :: dummy

!-----------------------------------------------------------------------
!     /*   read normal mode coordinate                                */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do j = 1, nbead
         do i = 1, natom
            read( iounit, * ) k, ux(i,j), uy(i,j), uz(i,j), &
     &                        dummy, dummy, dummy, mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!        /*   normal mode position -> Cartesian position   */
         call nm_trans( 0 )

!-----------------------------------------------------------------------
!     /*   read Cartesian coordinate                                  */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do j = 1, nbead
         do i = 1, natom
            read( iounit, * ) k, x(i,j), y(i,j), z(i,j), &
     &                        dummy, dummy, dummy, mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!        /*   Cartesian position -> normal mode position   */
         call nm_trans( 1 )

!-----------------------------------------------------------------------
!     /*   write normal mode coordinate                               */
!-----------------------------------------------------------------------

      else if ( irw .eq. 3 ) then

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   write   */
         do j = 1, nbead
         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         k, ux(i,j), uy(i,j), uz(i,j), &
     &         0.d0, 0.d0, 0.d0, mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   write Cartesian coordinate                                 */
!-----------------------------------------------------------------------

      else if ( irw .eq. 4 ) then

!        /*   normal mode position -> Cartesian position   */
         call nm_trans( 0 )

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   write   */
         do j = 1, nbead
         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         k, x(i,j), y(i,j), z(i,j), &
     &         0.d0, 0.d0, 0.d0, mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

      else

!        /*   error handling   */
         call error_handling ( 1, 'subroutine restart_position', 27 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_velocity ( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file     irw = 1 :   read normal mode momentum
!
!                                 irw = 2 :   read Cartesian velocity
!
!                                 irw = 3 :   write normal mode momentum
!
!                                 irw = 4 :   write Cartesian velocity
!
!     NOTE:  Only normal modes are updated inside mdcycle;
!            Cartesian velocity update is not ensured.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, vx, vy, vz, pux, puy, puz, fictmass, &
     &   x, y, z, ux, uy, uz, iounit, natom, nbead, istep_end, mbox

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k

!-----------------------------------------------------------------------
!     /*   read normal mode coordinates and velocities                */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do j = 1, nbead
         do i = 1, natom
            read( iounit, * ) &
     &         k,  ux(i,j),  uy(i,j),  uz(i,j), &
     &            pux(i,j), puy(i,j), puz(i,j), mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!        /*   normal mode momentum -> normal mode velocity   */
         do j = 1, nbead
         do i = 1, natom
            vux(i,j) = pux(i,j)/sqrt(fictmass(i,j))
            vuy(i,j) = puy(i,j)/sqrt(fictmass(i,j))
            vuz(i,j) = puz(i,j)/sqrt(fictmass(i,j))
         end do
         end do

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity( 0 )

!-----------------------------------------------------------------------
!     /*   read Cartesian coordinates and velocities                  */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do j = 1, nbead
         do i = 1, natom
            read( iounit, * ) &
     &         k,  x(i,j),  y(i,j),  z(i,j), &
     &            vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!        /*   Cartesian velocity -> normal mode velocity   */
         call nm_trans_velocity( 1 )

!-----------------------------------------------------------------------
!     /*   write normal mode coordinates and momenta                  */
!-----------------------------------------------------------------------

      else if ( irw .eq. 3 ) then

!        /*   current step   */
         k = istep_end

!        /*   normal mode velocity -> normal mode momentum   */
         do j = 1, nbead
         do i = 1, natom
            pux(i,j) = vux(i,j)*sqrt(fictmass(i,j))
            puy(i,j) = vuy(i,j)*sqrt(fictmass(i,j))
            puz(i,j) = vuz(i,j)*sqrt(fictmass(i,j))
         end do
         end do

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   write   */
         do j = 1, nbead
         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         k,  ux(i,j),  uy(i,j),  uz(i,j), &
     &            pux(i,j), puy(i,j), puz(i,j), mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   write Cartesian coordinates and velocities                 */
!-----------------------------------------------------------------------

      else if ( irw .eq. 4 ) then

!        /*   normal mode position -> Cartesian position   */
         call nm_trans( 0 )

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity( 0 )

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   write   */
         do j = 1, nbead
         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         k,  x(i,j),  y(i,j),  z(i,j), &
     &            vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

      else

!        /*   error handling   */
         call error_handling ( 1, 'subroutine restart_velocity', 27 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_cartesian ( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file     irw = 1 :   read Cartesian position
!
!                                 irw = 2 :   write Cartesian position
!
!                                 irw = 3 :   read Cartesian velocity
!
!                                 irw = 4 :   write Cartesian velocity
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, vx, vy, vz, iounit, natom, nbead, istep_end, mbox

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k

      real(8) :: dummy

!-----------------------------------------------------------------------
!     /*   read Cartesian coordinate                                  */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do j = 1, nbead
         do i = 1, natom
            read( iounit, * ) k, x(i,j), y(i,j), z(i,j), &
     &                        dummy, dummy, dummy, mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   write Cartesian coordinate                                 */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   write   */
         do j = 1, nbead
         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         k, x(i,j), y(i,j), z(i,j), &
     &         0.d0, 0.d0, 0.d0, mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   read Cartesian coordinate and velocity                     */
!-----------------------------------------------------------------------

      else if ( irw .eq. 3 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do j = 1, nbead
         do i = 1, natom
            read( iounit, * ) k, x(i,j), y(i,j), z(i,j), &
     &                        vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   write Cartesian coordinate                                 */
!-----------------------------------------------------------------------

      else if ( irw .eq. 4 ) then

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   write   */
         do j = 1, nbead
         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         k, x(i,j), y(i,j), z(i,j), &
     &         vx(i,j), vy(i,j), vz(i,j), mbox(1:3,i,j)
         end do
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!     /*   read Cartesian coordinate                                  */
!-----------------------------------------------------------------------

      else if ( irw .eq. 5 ) then

!        /*   open file   */
         open ( iounit, file = 'geometry.ini', status = 'unknown' )

!        /*   read   */
         do i = 1, natom
            read( iounit, * ) k, x(i,1), y(i,1), z(i,1), &
     &                        dummy, dummy, dummy, mbox(1:3,i,1)
         end do

!        /*   close file   */
         close( iounit )

      else

!        /*   error handling   */
         call error_handling ( 1, 'subroutine restart_cartesian', 28 )

      end if

      return
      end





!***********************************************************************
      subroutine restart_string ( irw )
!***********************************************************************
!=======================================================================
!
!     read/write restart file     irw = 1 :   read Cartesian position
!
!                                 irw = 2 :   write Cartesian position
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, dt, iounit, natom, nbead, istep_end, mbox

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: irw, i, j, k, ierr

      real(8) :: dummy

!-----------------------------------------------------------------------
!     /*   read Cartesian coordinate                                  */
!-----------------------------------------------------------------------

      if ( irw .eq. 1 ) then

!        /*   open file   */
         open ( iounit, file = 'string.ini', status = 'unknown' )

!        /*   read   */
         do j = 1, nbead
         do i = 1, natom
            read( iounit, *, iostat=ierr ) &
     &         k, x(i,j), y(i,j), z(i,j), &
     &         dummy, dummy, dummy, mbox(1:3,i,j)
         end do
         end do

!        /*   read step   */
         read( iounit, *, iostat=ierr ) dt

!        /*   close file   */
         close( iounit )

!        /*   error handling   */
         call error_handling ( ierr, 'subroutine restart_string', 25 )

!-----------------------------------------------------------------------
!     /*   write Cartesian coordinate                                 */
!-----------------------------------------------------------------------

      else if ( irw .eq. 2 ) then

!        /*   current step   */
         k = istep_end

!        /*   open file   */
         open ( iounit, file = 'string.ini', status = 'unknown' )

!        /*   write   */
         do j = 1, nbead
         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         k, x(i,j), y(i,j), z(i,j), &
     &         0.d0, 0.d0, 0.d0, mbox(1:3,i,j)
         end do
         end do

!        /*   write step   */
         write( iounit, '(e24.16)' ) dt

!        /*   close file   */
         close( iounit )

      else

!        /*   error handling   */
         call error_handling ( 1, 'subroutine restart_string', 25 )

      end if

      return
      end
