!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 18, 2020 by M. Shiga
!      Description:     molecular reorientation in the string method
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine reorient_string_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, x, y, z, natom, nbead, iboundary, atom_change

#ifdef nolapack
      use common_variables, only : myrank
#endif

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8), dimension(:,:), allocatable :: a
      real(8), dimension(:,:), allocatable :: u
      real(8), dimension(:,:), allocatable :: vt
      real(8), dimension(:),   allocatable :: s
      real(8), dimension(:),   allocatable :: work
      real(8), dimension(:,:), allocatable :: b

      integer :: n = 3
      integer :: info = 0
      integer, save :: lwork = -1

      integer :: i, j, k, ibead, jbead, kbead, ix, iy, iz

      real(8) :: xgj, ygj, zgj, xgk, ygk, zgk, pg, pmi, &
     &           rxj, ryj, rzj, rxk, ryk, rzk, dummy(1,1)

!-----------------------------------------------------------------------
!     /*   monoatomic system                                          */
!-----------------------------------------------------------------------

      if ( natom .eq. 1 ) return

!-----------------------------------------------------------------------
!     /*   overwrite to bead 1 positions for fixed atoms              */
!-----------------------------------------------------------------------

      ix = 0
      iy = 0
      iz = 0

      do i = 1, natom
         if ( atom_change(i)(1:6) .eq. 'FREEZE' ) ix = 1
         if ( atom_change(i)(1:6) .eq. 'HEAVY ' ) ix = 1
         if ( atom_change(i)(1:6) .eq. 'FIXX  ' ) ix = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXY ' ) ix = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXZ ' ) ix = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXYZ' ) ix = 1
         if ( atom_change(i)(1:6) .eq. 'FREEZE' ) iy = 1
         if ( atom_change(i)(1:6) .eq. 'HEAVY ' ) iy = 1
         if ( atom_change(i)(1:6) .eq. 'FIXY  ' ) iy = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXY ' ) iy = 1
         if ( atom_change(i)(1:6) .eq. 'FIXYZ ' ) iy = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXYZ' ) iy = 1
         if ( atom_change(i)(1:6) .eq. 'FREEZE' ) iz = 1
         if ( atom_change(i)(1:6) .eq. 'HEAVY ' ) iz = 1
         if ( atom_change(i)(1:6) .eq. 'FIXZ  ' ) iz = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXZ ' ) iz = 1
         if ( atom_change(i)(1:6) .eq. 'FIXYZ ' ) iz = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXYZ' ) iz = 1
      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated(a)  ) allocate( a(n,n)  )
      if ( .not. allocated(u)  ) allocate( u(n,n)  )
      if ( .not. allocated(vt) ) allocate( vt(n,n) )
      if ( .not. allocated(s)  ) allocate( s(n)    )
      if ( .not. allocated(b)  ) allocate( b(n,n)  )

!-----------------------------------------------------------------------
!     /*   singular value decomposition: set lwork                    */
!-----------------------------------------------------------------------

#ifdef nolapack
      write( 6, '(a)' ) &
     &   'Error - Singular value decomposition routine not linked.'
      call error_handling_MPI &
     &   ( 1, 'subroutine reorient_string_MPI', 30 )
#else
      if ( lwork .eq. -1 ) then
         call dgesvd( 'A', 'A', &
     &      n, n, a, n, s, u, n, vt, n, dummy, lwork, info )
         lwork = nint( dummy(1,1) )
      end if
      if ( .not. allocated(work) ) allocate( work(lwork) )
#endif

!-----------------------------------------------------------------------
!     /*   start main loop                                           */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do ibead = 1, nbead-1

!-----------------------------------------------------------------------
!        /*   set reference bead                                      */
!-----------------------------------------------------------------------

!        /*  j is the reference   */
         jbead = ibead

!        /*  k will be shifted   */
         kbead = ibead + 1

!-----------------------------------------------------------------------
!        /*   center of masses                                        */
!-----------------------------------------------------------------------

!        /*   center of mass of bead j   */
         xgj = 0.d0
         ygj = 0.d0
         zgj = 0.d0

!        /*   center of mass of bead k   */
         xgk = 0.d0
         ygk = 0.d0
         zgk = 0.d0

!        /*   total mass   */
         pg  = 0.d0

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   mass of atom i   */
            pmi = fictmass(i,1)

!           /*   center of mass of bead j   */
            xgj = xgj + pmi * x(i,jbead)
            ygj = ygj + pmi * y(i,jbead)
            zgj = zgj + pmi * z(i,jbead)

!           /*   center of mass of bead k   */
            xgk = xgk + pmi * x(i,kbead)
            ygk = ygk + pmi * y(i,kbead)
            zgk = zgk + pmi * z(i,kbead)

!           /*   total mass   */
            pg  = pg  + pmi

!        /*   loop of atoms   */
         end do

!        /*   center of mass of bead j   */
         xgj = xgj / pg
         ygj = ygj / pg
         zgj = zgj / pg

!        /*   center of mass of bead k   */
         xgk = xgk / pg
         ygk = ygk / pg
         zgk = zgk / pg

!-----------------------------------------------------------------------
!        /*   periodic boundaries: make b matrix                      */
!-----------------------------------------------------------------------

         if ( iboundary .ne. 0 ) then

            b(1,1) = 1.d0
            b(1,2) = 0.d0
            b(1,3) = 0.d0
            b(2,1) = 0.d0
            b(2,2) = 1.d0
            b(2,3) = 0.d0
            b(3,1) = 0.d0
            b(3,2) = 0.d0
            b(3,3) = 1.d0

!-----------------------------------------------------------------------
!        /*   free boundaries: make a matrix                          */
!-----------------------------------------------------------------------

         else

!           /*   initialize   */
            a(:,:) = 0.d0

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   mass of atom i   */
               pmi = fictmass(i,1)

!              /*   relative position of bead j   */
               rxj = x(i,jbead) - xgj
               ryj = y(i,jbead) - ygj
               rzj = z(i,jbead) - zgj

!              /*   relative position of bead k   */
               rxk = x(i,kbead) - xgk
               ryk = y(i,kbead) - ygk
               rzk = z(i,kbead) - zgk

!              /*   a matrix   */
               a(1,1) = a(1,1) + pmi * rxj * rxk
               a(1,2) = a(1,2) + pmi * rxj * ryk
               a(1,3) = a(1,3) + pmi * rxj * rzk
               a(2,1) = a(2,1) + pmi * ryj * rxk
               a(2,2) = a(2,2) + pmi * ryj * ryk
               a(2,3) = a(2,3) + pmi * ryj * rzk
               a(3,1) = a(3,1) + pmi * rzj * rxk
               a(3,2) = a(3,2) + pmi * rzj * ryk
               a(3,3) = a(3,3) + pmi * rzj * rzk

!           /*   loop of atoms   */
            end do

!-----------------------------------------------------------------------
!     /*   free boundaries: singular value decomposition              */
!-----------------------------------------------------------------------

#ifdef nolapack
            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - Singular value decomposition routine not' // &
     &            ' linked.'
               call error_handling_MPI &
     &            ( 1, 'subroutine reorient_string', 30 )
            end if
#else
            call dgesvd( 'A', 'A', &
     &         n, n, a, n, s, u, n, vt, n, work, lwork, info )
#endif

!-----------------------------------------------------------------------
!     /*   free boundaries: make b matrix                             */
!-----------------------------------------------------------------------

            b(:,:) = 0.d0

            do i = 1, n
            do j = 1, n
            do k = 1, n
               b(i,j) = b(i,j) + u(i,k) * vt(k,j)
            end do
            end do
            end do

         end if

!-----------------------------------------------------------------------
!     /*   shift bead position                                        */
!-----------------------------------------------------------------------

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   relative position of bead k   */
            rxk = x(i,kbead) - xgk
            ryk = y(i,kbead) - ygk
            rzk = z(i,kbead) - zgk

!           /*   rotate and translate   */
            if ( ix .eq. 0 ) then
               x(i,kbead) = b(1,1)*rxk + b(1,2)*ryk + b(1,3)*rzk + xgj
            end if
            if ( iy .eq. 0 ) then
               y(i,kbead) = b(2,1)*rxk + b(2,2)*ryk + b(2,3)*rzk + ygj
            end if
            if ( iz .eq. 0 ) then
               z(i,kbead) = b(3,1)*rxk + b(3,2)*ryk + b(3,3)*rzk + zgj
            end if

!        /*   loop of atoms   */
         end do

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   overwrite to bead 1 positions for fixed atoms              */
!-----------------------------------------------------------------------

      do i = 1, natom
         k = 0
         if ( atom_change(i)(1:6) .eq. 'FREEZE' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'HEAVY ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXX  ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXY ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXZ ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXYZ' ) k = 1
         if ( k .eq. 1 ) then
            do j = 2, nbead
               x(i,j) = x(i,1)
            end do
         end if
      end do

      do i = 1, natom
         k = 0
         if ( atom_change(i)(1:6) .eq. 'FREEZE' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'HEAVY ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXY  ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXY ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXYZ ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXYZ' ) k = 1
         if ( k .eq. 1 ) then
            do j = 2, nbead
               y(i,j) = y(i,1)
            end do
         end if
      end do

      do i = 1, natom
         k = 0
         if ( atom_change(i)(1:6) .eq. 'FREEZE' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'HEAVY ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXZ  ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXZ ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXYZ ' ) k = 1
         if ( atom_change(i)(1:6) .eq. 'FIXXYZ' ) k = 1
         if ( k .eq. 1 ) then
            do j = 2, nbead
               z(i,j) = z(i,1)
            end do
         end if
      end do

!-----------------------------------------------------------------------
!     /*   memory deallocation                                        */
!-----------------------------------------------------------------------

      if ( allocated(a)    ) deallocate( a    )
      if ( allocated(u)    ) deallocate( u    )
      if ( allocated(vt)   ) deallocate( vt   )
      if ( allocated(s)    ) deallocate( s    )
      if ( allocated(b)    ) deallocate( b    )
      if ( allocated(work) ) deallocate( work )

      return
      end
