!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     replica exchange Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine remccycle_t_MPI
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     no exchange between parallel temperatures.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_end, istep_start, nstep, iexit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   save data   */
      call save_remc

!     /*   standard output   */
      call standard_remc_t_MPI

!     /*   do some analysis   */
      call analysis_remc_MPI( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the coordinates   */
         call update_remc_MPI

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   judge accept or reject: mc   */
         call judge_remc_t_MPI

!        /*   save data   */
         call save_remc

!        /*   standard output   */
         call standard_remc_t_MPI

!        /*   do some analysis   */
         call analysis_remc_MPI( 2 )

!        /*   output restart   */
         call backup_remc_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine remccycle_t_dual_MPI
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     no exchange between parallel temperatures.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_end, istep_start, nstep, iexit

      use rehmc_variables, only : &
     &   jstep, nsubstep_ax

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_dual_lo_MPI

!     /*   get interatomic forces   */
      call getforce_dual_hi_MPI

!     /*   save data   */
      call save_remc

!     /*   standard output   */
      call standard_remc_t_MPI

!     /*   do some analysis   */
      call analysis_remc_MPI( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo_MPI

!        /*   start dual subcycle   */
         do jstep = 1, nsubstep_ax

!           /*   update the coordinates   */
            call update_remc_sub_MPI

!           /*   get interatomic forces   */
            call getforce_dual_lo_MPI

!           /*   judge accept or reject: mc   */
            call judge_remc_sub_MPI

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi_MPI

!        /*   judge accept or reject: mc   */
         call judge_remc_t_dual_MPI

!        /*   save data   */
         call save_remc

!        /*   standard output   */
         call standard_remc_t_MPI

!        /*   do some analysis   */
         call analysis_remc_MPI( 2 )

!        /*   output restart   */
         call backup_remc_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine remccycle_tx_MPI
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     temperature exchange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_end, istep_start, nstep, iexit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   save data   */
      call save_remc

!     /*   standard output   */
      call standard_remc_tx_MPI

!     /*   do some analysis   */
      call analysis_remc_MPI( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the coordinates   */
         call update_remc_MPI

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   judge accept or reject: mc   */
         call judge_remc_t_MPI

!        /*   judge accept or reject: rem   */
         call judge_rem_remc_MPI

!        /*   save data   */
         call save_remc

!        /*   standard output   */
         call standard_remc_tx_MPI

!        /*   do some analysis   */
         call analysis_remc_MPI( 2 )

!        /*   output restart   */
         call backup_remc_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine remccycle_tx_dual_MPI
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     temperature exchange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_end, istep_start, nstep, iexit

      use rehmc_variables, only : &
     &   jstep, nsubstep_ax

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_dual_lo_MPI

!     /*   get interatomic forces   */
      call getforce_dual_hi_MPI

!     /*   save data   */
      call save_remc

!     /*   standard output   */
      call standard_remc_tx_MPI

!     /*   do some analysis   */
      call analysis_remc_MPI( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo_MPI

!        /*   start dual subcycle   */
         do jstep = 1, nsubstep_ax

!           /*   update the coordinates   */
            call update_remc_sub_MPI

!           /*   get interatomic forces   */
            call getforce_dual_lo_MPI

!           /*   judge accept or reject: mc   */
            call judge_remc_sub_MPI

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi_MPI

!        /*   judge accept or reject: mc   */
         call judge_remc_t_dual_MPI

!        /*   judge accept or reject: rem   */
         call judge_rem_remc_MPI

!        /*   save data   */
         call save_remc

!        /*   standard output   */
         call standard_remc_tx_MPI

!        /*   do some analysis   */
         call analysis_remc_MPI( 2 )

!        /*   output restart   */
         call backup_remc_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine exchange_rehmc_MPI
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     no exchange between parallel temperatures.
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : istep

      use rehmc_variables, only : istep_ax_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!-----------------------------------------------------------------------
!     /*   condition for atom exchange                                */
!-----------------------------------------------------------------------

      if ( istep_ax_hmc .le. 0 ) return

      if ( mod(istep,istep_ax_hmc) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   save data   */
      call save_remc

!     /*   update the coordinates   */
      call update_remc_MPI

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   judge accept or reject: mc   */
      call judge_remc_t_MPI

!     /*   standard output   */
      call standard_remc_t_MPI

      return
      end





!***********************************************************************
      subroutine exchange_rehmc_dual_MPI
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     no exchange between parallel temperatures.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : istep

      use rehmc_variables, only : istep_ax_hmc, jstep, nsubstep_ax

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!-----------------------------------------------------------------------
!     /*   condition for atom exchange                                */
!-----------------------------------------------------------------------

      if ( istep_ax_hmc .le. 0 ) return

      if ( mod(istep,istep_ax_hmc) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   save data   */
      call save_remc

!     /*   get interatomic forces   */
      call getforce_dual_lo_MPI

!     /*   start dual subcycle   */
      do jstep = 1, nsubstep_ax

!        /*   update the coordinates   */
         call update_remc_sub_MPI

!        /*   get interatomic forces   */
         call getforce_dual_lo_MPI

!        /*   judge accept or reject: mc   */
         call judge_remc_sub_MPI

!     /*   end of molecular dynamics cycle   */
      end do

!     /*   get interatomic forces   */
      call getforce_dual_hi_MPI

!     /*   judge accept or reject: mc   */
      call judge_remc_t_dual_MPI

!     /*   standard output   */
      call standard_remc_t_MPI

      return
      end
