!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     replica exchange Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine remccycle_t
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     no exchange between parallel temperatures.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_end, istep_start, nstep, iexit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce

!     /*   save data   */
      call save_remc

!     /*   standard output   */
      call standard_remc_t

!     /*   do some analysis   */
      call analysis_remc( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the coordinates   */
         call update_remc

!        /*   get interatomic forces   */
         call getforce

!        /*   judge accept or reject: mc   */
         call judge_remc_t

!        /*   save data   */
         call save_remc

!        /*   standard output   */
         call standard_remc_t

!        /*   do some analysis   */
         call analysis_remc( 2 )

!        /*   output restart   */
         call backup_remc

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine remccycle_t_dual
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     no exchange between parallel temperatures.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_end, istep_start, nstep, iexit

      use rehmc_variables, only : &
     &   jstep, nsubstep_ax

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   save data   */
      call save_remc

!     /*   standard output   */
      call standard_remc_t

!     /*   do some analysis   */
      call analysis_remc( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   start dual subcycle   */
         do jstep = 1, nsubstep_ax

!           /*   update the coordinates   */
            call update_remc_sub

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   judge accept or reject: mc   */
            call judge_remc_sub

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   judge accept or reject: mc   */
         call judge_remc_t_dual

!        /*   save data   */
         call save_remc

!        /*   standard output   */
         call standard_remc_t

!        /*   do some analysis   */
         call analysis_remc( 2 )

!        /*   output restart   */
         call backup_remc

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine remccycle_tx
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     temperature exchange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_end, istep_start, nstep, iexit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce

!     /*   save data   */
      call save_remc

!     /*   standard output   */
      call standard_remc_tx

!     /*   do some analysis   */
      call analysis_remc( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the coordinates   */
         call update_remc

!        /*   get interatomic forces   */
         call getforce

!        /*   judge accept or reject: mc   */
         call judge_remc_t

!        /*   judge accept or reject: rem   */
         call judge_rem_rehmc

!        /*   save data   */
         call save_remc

!        /*   standard output   */
         call standard_remc_tx

!        /*   do some analysis   */
         call analysis_remc( 2 )

!        /*   output restart   */
         call backup_remc

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine remccycle_tx_dual
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     temperature exchange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_end, istep_start, nstep, iexit

      use rehmc_variables, only : &
     &   jstep, nsubstep_ax

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   save data   */
      call save_remc

!     /*   standard output   */
      call standard_remc_tx

!     /*   do some analysis   */
      call analysis_remc( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   start dual subcycle   */
         do jstep = 1, nsubstep_ax

!           /*   update the coordinates   */
            call update_remc_sub

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   judge accept or reject: mc   */
            call judge_remc_sub

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   judge accept or reject: mc   */
         call judge_remc_t_dual

!        /*   judge accept or reject: rem   */
         call judge_rem_rehmc

!        /*   save data   */
         call save_remc

!        /*   standard output   */
         call standard_remc_tx

!        /*   do some analysis   */
         call analysis_remc( 2 )

!        /*   output restart   */
         call backup_remc

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine exchange_rehmc
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     no exchange between parallel temperatures.
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : istep

      use rehmc_variables, only : istep_ax_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!-----------------------------------------------------------------------
!     /*   condition for atom exchange                                */
!-----------------------------------------------------------------------

      if ( istep_ax_hmc .le. 0 ) return

      if ( mod(istep,istep_ax_hmc) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   save data   */
      call save_remc

!     /*   update the coordinates   */
      call update_remc

!     /*   get interatomic forces   */
      call getforce

!     /*   judge accept or reject: mc   */
      call judge_remc_t

!     /*   standard output   */
      call standard_remc_t

      return
      end





!***********************************************************************
      subroutine exchange_rehmc_dual
!***********************************************************************
!=======================================================================
!
!     replica exchange monte carlo for atomic exchange:
!     no exchange between parallel temperatures.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : istep

      use rehmc_variables, only : istep_ax_hmc, jstep, nsubstep_ax

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   flag
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   condition for atom exchange                                */
!-----------------------------------------------------------------------

      if ( istep_ax_hmc .le. 0 ) return

      if ( mod(istep,istep_ax_hmc) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     //   setup remc
      if ( iset .eq. 0 ) call setup_remc

!     //   setup complete
      iset = 1

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   save data   */
      call save_remc

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   start dual subcycle   */
      do jstep = 1, nsubstep_ax

!        /*   update the coordinates   */
         call update_remc_sub

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   judge accept or reject: mc   */
         call judge_remc_sub

!     /*   end of molecular dynamics cycle   */
      end do

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   judge accept or reject: mc   */
      call judge_remc_t_dual

!     /*   standard output   */
      call standard_remc_t

      return
      end
