!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     replica exchange hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine rehmccycle_t_npt
!***********************************************************************
!=======================================================================
!
!     replica exchange hybrid monte carlo:
!     no exchange between parallel temperatures.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iref, nref, istep, istep_start, istep_end, iexit, istep_hmc, &
     &   nstep

      use rehmc_variables, only : &
     &   jstep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_rem

!     /*   force multiplied by nbead   */
      call nm_trans_force_rehmc

!     /*   energy   */
      call getenergy_rehmc_npt

!     /*   save data   */
      call save_rehmc_npt

!     /*   adjust HMC step   */
      call adjust_step_rehmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_rehmc_t

!     /*   do some analysis   */
      call analysis_rehmc_npt( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_rehmc_npt

!           /*   update the velocities by interatomic forces   */
            call update_vel_xyz_rehmc

!           /*   respa   */
            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_rehmc_npt

!              /*   update atom velocities   */
               call update_vel_all_rehmc_npt

!              /*   update atom position   */
               call update_pos_all_rehmc_npt

!              /*   update atom velocities   */
               call update_vel_all_rehmc_npt

!              /*   update volume   */
               call update_pos_box_rehmc_npt

!           /*   respa   */
            end do

!           /*   get interatomic forces   */
            call getforce_rem

!           /*   force multiplied by nbead   */
            call nm_trans_force_rehmc

!           /*   update the velocities by interatomic forces   */
            call update_vel_xyz_rehmc

!           /*   update volume velocity   */
            call update_vel_box_rehmc_npt

!           /*   energy   */
            call getenergy_rehmc_npt

!        /*   end of molecular dynamics cycle   */
         end do

!c        /*   energy   */
!         call getenergy_rehmc_npt

!        /*   judge accept or reject: hmc   */
         call judge_hmc_rehmc_npt

!        /*   adjust HMC step   */
         call adjust_step_rehmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_rehmc_t

!        /*   atom exchange   */
         call exchange_rehmc

!        /*   do some analysis   */
         call analysis_rehmc_npt( 2 )

!        /*   output restart   */
         call backup_rehmc_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine rehmccycle_tx_npt
!***********************************************************************
!=======================================================================
!
!     replica exchange hybrid monte carlo:
!     temperature exchange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iref, nref, istep, istep_end, istep_start, nstep, iexit, &
     &   istep_hmc

      use rehmc_variables, only : &
     &   jstep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_rem

!     /*   force multiplied by nbead   */
      call nm_trans_force_rehmc

!     /*   energy   */
      call getenergy_rehmc_npt

!     /*   save data   */
      call save_rehmc_npt

!     /*   adjust HMC step   */
      call adjust_step_rehmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_rehmc_tx

!     /*   do some analysis   */
      call analysis_rehmc_npt( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_rehmc_npt

!           /*   update the velocities by interatomic forces   */
            call update_vel_xyz_rehmc

!           /*   respa   */
            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_rehmc_npt

!              /*   update atom velocities   */
               call update_vel_all_rehmc_npt

!              /*   update atom position   */
               call update_pos_all_rehmc_npt

!              /*   update atom velocities   */
               call update_vel_all_rehmc_npt

!              /*   update volume   */
               call update_pos_box_rehmc_npt

!           /*   respa   */
            end do

!           /*   get interatomic forces   */
            call getforce_rem

!           /*   force multiplied by nbead   */
            call nm_trans_force_rehmc

!           /*   update the velocities by interatomic forces   */
            call update_vel_xyz_rehmc

!           /*   update volume velocity   */
            call update_vel_box_rehmc_npt

!           /*   energy   */
            call getenergy_rehmc_npt

!        /*   end of molecular dynamics cycle   */
         end do

!c        /*   energy   */
!         call getenergy_rehmc_npt

!        /*   judge accept or reject: hmc   */
         call judge_hmc_rehmc_npt

!        /*   judge accept or reject: rem   */
         call judge_rem_rehmc_npt

!        /*   adjust HMC step   */
         call adjust_step_rehmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_rehmc_tx

!        /*   atom exchange   */
         call exchange_rehmc

!        /*   do some analysis   */
         call analysis_rehmc_npt( 2 )

!        /*   output restart   */
         call backup_rehmc_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine rehmccycle_t_dual_npt
!***********************************************************************
!=======================================================================
!
!     replica exchange hybrid monte carlo:
!     no exchange between parallel temperatures.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iref, nref, istep, istep_start, istep_end, iexit, istep_hmc, &
     &   nstep

      use rehmc_variables, only : &
     &   jstep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_dual_lo_rem

!     /*   get interatomic forces   */
      call getforce_dual_hi_rem

!     /*   force multiplied by nbead   */
      call nm_trans_force_rehmc

!     /*   energy   */
      call getenergy_rehmc_npt

!     /*   save data   */
      call save_rehmc_npt

!     /*   adjust HMC step   */
      call adjust_step_rehmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_rehmc_t

!     /*   do some analysis   */
      call analysis_rehmc_npt( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo_rem

!        /*   force multiplied by nbead   */
         call nm_trans_force_rehmc

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_rehmc_npt

!           /*   update the velocities by interatomic forces   */
            call update_vel_xyz_rehmc

!           /*   respa   */
            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_rehmc_npt

!              /*   update atom velocities   */
               call update_vel_all_rehmc_npt

!              /*   update atom position   */
               call update_pos_all_rehmc_npt

!              /*   update atom velocities   */
               call update_vel_all_rehmc_npt

!              /*   update volume   */
               call update_pos_box_rehmc_npt

!           /*   respa   */
            end do

!           /*   get interatomic forces   */
            call getforce_dual_lo_rem

!           /*   force multiplied by nbead   */
            call nm_trans_force_rehmc

!           /*   update the velocities by interatomic forces   */
            call update_vel_xyz_rehmc

!           /*   update volume velocity   */
            call update_vel_box_rehmc_npt

!           /*   energy   */
            call getenergy_rehmc_npt

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi_rem

!        /*   energy   */
         call getenergy_rehmc_npt

!        /*   judge accept or reject: hmc   */
         call judge_hmc_rehmc_npt

!        /*   adjust HMC step   */
         call adjust_step_rehmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_rehmc_t

!        /*   atom exchange   */
         call exchange_rehmc_dual

!        /*   do some analysis   */
         call analysis_rehmc_npt( 2 )

!        /*   output restart   */
         call backup_rehmc_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine rehmccycle_tx_dual_npt
!***********************************************************************
!=======================================================================
!
!     replica exchange hybrid monte carlo:
!     temperature exchange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iref, nref, istep, istep_end, istep_start, nstep, iexit, &
     &   istep_hmc

      use rehmc_variables, only : &
     &   jstep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_dual_lo_rem

!     /*   get interatomic forces   */
      call getforce_dual_hi_rem

!     /*   force multiplied by nbead   */
      call nm_trans_force_rehmc

!     /*   energy   */
      call getenergy_rehmc_npt

!     /*   save data   */
      call save_rehmc_npt

!     /*   adjust HMC step   */
      call adjust_step_rehmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_rehmc_tx

!     /*   do some analysis   */
      call analysis_rehmc_npt( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo_rem

!        /*   force multiplied by nbead   */
         call nm_trans_force_rehmc

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_rehmc_npt

!           /*   update the velocities by interatomic forces   */
            call update_vel_xyz_rehmc

!           /*   respa   */
            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_rehmc_npt

!              /*   update atom velocities   */
               call update_vel_all_rehmc_npt

!              /*   update atom position   */
               call update_pos_all_rehmc_npt

!              /*   update atom velocities   */
               call update_vel_all_rehmc_npt

!              /*   update volume   */
               call update_pos_box_rehmc_npt

!           /*   respa   */
            end do

!           /*   get interatomic forces   */
            call getforce_dual_lo_rem

!           /*   force multiplied by nbead   */
            call nm_trans_force_rehmc

!           /*   update the velocities by interatomic forces   */
            call update_vel_xyz_rehmc

!           /*   update volume velocity   */
            call update_vel_box_rehmc_npt

!           /*   energy   */
            call getenergy_rehmc_npt

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi_rem

!        /*   energy   */
         call getenergy_rehmc_npt

!        /*   judge accept or reject: hmc   */
         call judge_hmc_rehmc_npt

!        /*   judge accept or reject: rem   */
         call judge_rem_rehmc_npt

!        /*   adjust HMC step   */
         call adjust_step_rehmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_rehmc_tx

!        /*   atom exchange   */
         call exchange_rehmc_dual

!        /*   do some analysis   */
         call analysis_rehmc_npt( 2 )

!        /*   output restart   */
         call backup_rehmc_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine update_vel_box_rehmc_npt
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, volume_bead, boxmass, pressure, cmtk, dt, physmass, &
     &   vbox_bead, vir_bead, boxdot_bead, box_bead, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ckin(3,3), factor(3,3), dkin, pres(3,3), vbox(3,3), &
     &           vir(3,3), boxdot(3,3), box(3,3), volume

      integer :: i, j, k, l

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do l = 1, nbead

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      vir(:,:)  = vir_bead(:,:,l)
      vbox(:,:) = vbox_bead(:,:,l)
      box(:,:)  = box_bead(:,:,l)
      volume    = volume_bead(l)

!-----------------------------------------------------------------------
!     /*   kinetic energy matrix of centroids                         */
!-----------------------------------------------------------------------

      ckin(:,:) = 0.d0

      do i = 1, natom

         ckin(1,1) = ckin(1,1) + physmass(i)*vx(i,l)*vx(i,l)
         ckin(1,2) = ckin(1,2) + physmass(i)*vx(i,l)*vy(i,l)
         ckin(1,3) = ckin(1,3) + physmass(i)*vx(i,l)*vz(i,l)
         ckin(2,1) = ckin(2,1) + physmass(i)*vy(i,l)*vx(i,l)
         ckin(2,2) = ckin(2,2) + physmass(i)*vy(i,l)*vy(i,l)
         ckin(2,3) = ckin(2,3) + physmass(i)*vy(i,l)*vz(i,l)
         ckin(3,1) = ckin(3,1) + physmass(i)*vz(i,l)*vx(i,l)
         ckin(3,2) = ckin(3,2) + physmass(i)*vz(i,l)*vy(i,l)
         ckin(3,3) = ckin(3,3) + physmass(i)*vz(i,l)*vz(i,l)

      end do

      ckin(:,:) = 0.5d0*ckin(:,:)

!-----------------------------------------------------------------------
!     /*   kinetic energy of centroids                                */
!-----------------------------------------------------------------------

      dkin = ckin(1,1) + ckin(2,2) + ckin(3,3)

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres(:,:) = ( 2.d0*ckin(:,:) + vir(:,:) ) / volume

!-----------------------------------------------------------------------
!     /*   box force                                                  */
!-----------------------------------------------------------------------

      factor(:,:) = volume * pres(:,:)

      factor(1,1) = factor(1,1) - volume * pressure
      factor(2,2) = factor(2,2) - volume * pressure
      factor(3,3) = factor(3,3) - volume * pressure

      factor(1,1) = factor(1,1) + cmtk / dble(3.d0*natom) * 2.d0*dkin
      factor(2,2) = factor(2,2) + cmtk / dble(3.d0*natom) * 2.d0*dkin
      factor(3,3) = factor(3,3) + cmtk / dble(3.d0*natom) * 2.d0*dkin

!-----------------------------------------------------------------------
!     /*   update logarithm of box velocity                           */
!-----------------------------------------------------------------------

      vbox(:,:) = vbox(:,:) + factor(:,:) / boxmass(:,:) * 0.5d0*dt

!-----------------------------------------------------------------------
!     /*   apply rotational correction                                */
!-----------------------------------------------------------------------

      do i = 1, 3
      do j = i, 3
         vbox(i,j) = 0.5d0 * ( vbox(j,i) + vbox(i,j) )
         vbox(j,i) = vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         boxdot(i,j) = boxdot(i,j) + vbox(i,k)*box(k,j)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      vbox_bead(:,:,l)   = vbox(:,:)
      boxdot_bead(:,:,l) = boxdot(:,:)

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine update_pos_box_rehmc_npt
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   volume_bead, vbox_bead, dt_ref, box_bead, boxinv_bead, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l

      real(8) :: b(3,3), a(3,3), e(3), factor, det3, volume, vbox(3,3), &
     &           box(3,3), boxinv(3,3)

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do l = 1, nbead

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      vbox(:,:)  = vbox_bead(:,:,l)
      box(:,:)   = box_bead(:,:,l)

!-----------------------------------------------------------------------
!     /*   diagonalize box velocity                                   */
!-----------------------------------------------------------------------

      call ddiag ( vbox, e, b, 3 )

!-----------------------------------------------------------------------
!     /*   transform matrix                                           */
!-----------------------------------------------------------------------

      a(:,:) = 0.d0

      do k = 1, 3

         factor = exp( e(k)*0.5d0*dt_ref )

         do j = 1, 3
         do i = 1, 3
            a(i,j) = a(i,j) + b(i,k) * factor * b(j,k)
         end do
         end do

      end do

      do j = 1, 3-1
      do i = j+1, 3
         a(i,j) = 0.5d0 * ( a(i,j) + a(j,i) )
         a(j,i) = a(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   update box due to box transform                            */
!-----------------------------------------------------------------------

      b(1,:) = a(1,1)*box(1,:) + a(1,2)*box(2,:) + a(1,3)*box(3,:)
      b(2,:) = a(2,1)*box(1,:) + a(2,2)*box(2,:) + a(2,3)*box(3,:)
      b(3,:) = a(3,1)*box(1,:) + a(3,2)*box(2,:) + a(3,3)*box(3,:)

      box(:,:) = b(:,:)

!-----------------------------------------------------------------------
!     /*   update volume                                              */
!-----------------------------------------------------------------------

      volume = det3( box )

!-----------------------------------------------------------------------
!     /*   inverse matrix of cell matrix                              */
!-----------------------------------------------------------------------

      call inv3 ( box, boxinv )

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      box_bead(:,:,l)    = box(:,:)
      boxinv_bead(:,:,l) = boxinv(:,:)
      volume_bead(l)     = volume

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine update_vel_all_rehmc_npt
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, vbox_bead, dt_ref, cmtk, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l

      real(8) :: factor, b(3,3), a(3,3), e(3), vbox(3,3)

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do l = 1, nbead

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      vbox(:,:)  = vbox_bead(:,:,l)

!-----------------------------------------------------------------------
!     /*   friction term                                              */
!-----------------------------------------------------------------------

      factor = ( vbox(1,1)+vbox(2,2)+vbox(3,3) ) /dble(3*natom) * cmtk

      a(1,1) = + vbox(1,1) + factor
      a(1,2) = + vbox(1,2)
      a(1,3) = + vbox(1,3)
      a(2,1) = + vbox(2,1)
      a(2,2) = + vbox(2,2) + factor
      a(2,3) = + vbox(2,3)
      a(3,1) = + vbox(3,1)
      a(3,2) = + vbox(3,2)
      a(3,3) = + vbox(3,3) + factor

!-----------------------------------------------------------------------
!     /*   diagonalize friction                                       */
!-----------------------------------------------------------------------

      call ddiag ( a, e, b, 3 )

!-----------------------------------------------------------------------
!     /*   transform matrix                                           */
!-----------------------------------------------------------------------

      a(:,:) = 0.d0

      do i = 1, 3
      do j = 1, 3
      do k = 1, 3
         a(i,j) = a(i,j) + b(i,k) * exp( - e(k)*0.5d0*dt_ref ) * b(j,k)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   update velocity due to box transform                       */
!-----------------------------------------------------------------------

      do i = 1, natom

         e(1) = a(1,1)*vx(i,l) + a(1,2)*vy(i,l) + a(1,3)*vz(i,l)
         e(2) = a(2,1)*vx(i,l) + a(2,2)*vy(i,l) + a(2,3)*vz(i,l)
         e(3) = a(3,1)*vx(i,l) + a(3,2)*vy(i,l) + a(3,3)*vz(i,l)

         vx(i,l) = e(1)
         vy(i,l) = e(2)
         vz(i,l) = e(3)

      end do

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine update_pos_all_rehmc_npt
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, vx, vy, vz, dt_ref, vbox_bead, natom, &
     &   nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l

      real(8) :: b(3,3), a(3,3), e(3), factor, vbox(3,3)

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do l = 1, nbead

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      vbox(:,:) = vbox_bead(:,:,l)

!-----------------------------------------------------------------------
!     /*   diagonalize box velocity                                   */
!-----------------------------------------------------------------------

      call ddiag ( vbox, e, b, 3 )

!-----------------------------------------------------------------------
!     /*   transform matrix                                           */
!-----------------------------------------------------------------------

      a(:,:) = 0.d0

      do k = 1, 3

         factor = exp( e(k)*0.5d0*dt_ref )

         do i = 1, 3
         do j = 1, 3
            a(i,j) = a(i,j) + b(i,k) * factor * b(j,k)
         end do
         end do

      end do

!-----------------------------------------------------------------------
!     /*   update centroid due to box transform                       */
!-----------------------------------------------------------------------

      do i = 1, natom

         e(1) = a(1,1)*x(i,l) + a(1,2)*y(i,l) + a(1,3)*z(i,l)
         e(2) = a(2,1)*x(i,l) + a(2,2)*y(i,l) + a(2,3)*z(i,l)
         e(3) = a(3,1)*x(i,l) + a(3,2)*y(i,l) + a(3,3)*z(i,l)

         x(i,l) = e(1)
         y(i,l) = e(2)
         z(i,l) = e(3)

      end do

!-----------------------------------------------------------------------
!     /*   update centroid due to atom velocity                       */
!-----------------------------------------------------------------------

      x(:,l) = x(:,l) + vx(:,l) * dt_ref
      y(:,l) = y(:,l) + vy(:,l) * dt_ref
      z(:,l) = z(:,l) + vz(:,l) * dt_ref

!-----------------------------------------------------------------------
!     /*   update centroid due to box transform                       */
!-----------------------------------------------------------------------

      do i = 1, natom

         e(1) = a(1,1)*x(i,l) + a(1,2)*y(i,l) + a(1,3)*z(i,l)
         e(2) = a(2,1)*x(i,l) + a(2,2)*y(i,l) + a(2,3)*z(i,l)
         e(3) = a(3,1)*x(i,l) + a(3,2)*y(i,l) + a(3,3)*z(i,l)

         x(i,l) = e(1)
         y(i,l) = e(2)
         z(i,l) = e(3)

      end do

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      call pbc_rehmc_npt

!-----------------------------------------------------------------------
!     /*   copy geometry                                              */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         ux(i,j) = x(i,j)
         uy(i,j) = y(i,j)
         uz(i,j) = z(i,j)
      end do
      end do

      return
      end
