!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     polymers projections of mean forces
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine projection_polymers_atom_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : natom

      use polymers_variables, only : txc_poly, tyc_poly, tzc_poly, &
     &    fxc_poly, fyc_poly, fzc_poly, npoly, projcmf_poly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   real numbers   */
      real(8) :: sum

!     /*   integers   */
      integer :: i, j

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( projcmf_poly(1:3) .ne. 'ON ' ) return

!-----------------------------------------------------------------------
!     /*   calculate tangent of string                                */
!-----------------------------------------------------------------------

      call tangent_polymers_atom_MPI

!-----------------------------------------------------------------------
!     /*   projection applied except polymer ends                     */
!-----------------------------------------------------------------------

      do j = 2, npoly-1

         sum = 0.d0

         do i = 1, natom

            sum = sum + fxc_poly(i,j) * txc_poly(i,j)
            sum = sum + fyc_poly(i,j) * tyc_poly(i,j)
            sum = sum + fzc_poly(i,j) * tzc_poly(i,j)

         end do

         do i = 1, natom

            fxc_poly(i,j) = fxc_poly(i,j) - sum*txc_poly(i,j)
            fyc_poly(i,j) = fyc_poly(i,j) - sum*tyc_poly(i,j)
            fzc_poly(i,j) = fzc_poly(i,j) - sum*tzc_poly(i,j)

         end do

      end do

      return
      end





!***********************************************************************
      subroutine tangent_polymers_atom_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : physmass, natom

      use polymers_variables, only : &
     &    txc_poly, tyc_poly, tzc_poly, narc, npoly, ngrid_poly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   paramter s from reactant (0) to product (1)   */
      real(8) :: s1, s2, s3, s4

!     /*   real numbers   */
      real(8) :: factor, sum

!     /*   integers   */
      integer :: i, j, k, l

!     /*   mass weighted coordinates   */
      real(8), dimension(3*natom) :: r1, r2, r3, r4

!-----------------------------------------------------------------------
!     /*   initialize spline - s:r                                    */
!-----------------------------------------------------------------------

      call spline_init_crd_poly_atom

!-----------------------------------------------------------------------
!     /*   calculate tangent vector                                   */
!-----------------------------------------------------------------------

!     /*   free energy difference   */
      sum = 0.d0

!     /*   scan from the reactant to the product   */
      do j = 1, narc, ngrid_poly-1

!        /*   parameter s   */

         if ( j .eq. 1 ) then
            s1 = dble(j-1) / dble(narc-1)
            s2 = dble(j-0) / dble(narc-1)
            s3 = dble(j-1) / dble(narc-1)
            s4 = dble(j-0) / dble(narc-1)
         else if ( j .eq. narc ) then
            s1 = dble(j-2) / dble(narc-1)
            s2 = dble(j-1) / dble(narc-1)
            s3 = dble(j-2) / dble(narc-1)
            s4 = dble(j-1) / dble(narc-1)
         else
            s1 = dble(j-2) / dble(narc-1)
            s2 = dble(j-1) / dble(narc-1)
            s3 = dble(j-1) / dble(narc-1)
            s4 = dble(j-0) / dble(narc-1)
         end if

!        /*   get new mass weighted coordinates   */

         call spline_crd_poly_atom( r1, s1 )
         call spline_crd_poly_atom( r2, s2 )
         call spline_crd_poly_atom( r3, s3 )
         call spline_crd_poly_atom( r4, s4 )

!        /*   calculate arc length by linear approximation   */

         l = (j-1)/(ngrid_poly-1) + 1

         k = 0

         do i = 1, natom

            factor = sqrt(physmass(i))

            k = k + 1

            txc_poly(i,l) = 0.5d0 * ( r2(k) - r1(k) ) / factor &
     &                    + 0.5d0 * ( r4(k) - r3(k) ) / factor

            k = k + 1

            tyc_poly(i,l) = 0.5d0 * ( r2(k) - r1(k) ) / factor &
     &                    + 0.5d0 * ( r4(k) - r3(k) ) / factor

            k = k + 1

            tzc_poly(i,l) = 0.5d0 * ( r2(k) - r1(k) ) / factor &
     &                    + 0.5d0 * ( r4(k) - r3(k) ) / factor

         end do

      end do

!-----------------------------------------------------------------------
!     /*   normalize tangent vector                                   */
!-----------------------------------------------------------------------

      do l = 1, npoly

         sum = 0.d0

         do i = 1, natom
            sum = sum + txc_poly(i,l)*txc_poly(i,l) &
     &                + tyc_poly(i,l)*tyc_poly(i,l) &
     &                + tzc_poly(i,l)*tzc_poly(i,l)
         end do

         sum = sqrt(sum)

         txc_poly(:,l) = txc_poly(:,l) / sum
         tyc_poly(:,l) = tyc_poly(:,l) / sum
         tzc_poly(:,l) = tzc_poly(:,l) / sum

      end do

      return
      end

