!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     project out translations and rotations
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine project_out_nma( l )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: x, y, z, physmass, natom, iboundary

      use nma_variables, only: hess

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, i5or6

      real(8) :: suma, sumx, sumy, sumz, sump, xg, yg, zg, dx, dy, dz, &
     &           vxi, vyi, vzi

      real(8) :: p(3*natom,3*natom)

      real(8) :: a(3*natom)

      real(8) :: f(3*natom,3*natom)

      real(8) :: b(3,3)

      real(8) :: d(3,3)

      real(8) :: g(3)

      real(8) :: h3(3*natom,3)

      real(8) :: h5(3*natom,5)

      real(8) :: h6(3*natom,6)

      real(8) :: vx(natom), vy(natom), vz(natom)

!-----------------------------------------------------------------------
!     /*   projection preset to unit matrix                           */
!-----------------------------------------------------------------------

      p(:,:) = 0.d0

      do i = 1, 3*natom
         p(i,i) = 1.d0
      end do

!-----------------------------------------------------------------------
!     /*   translational contribution                                 */
!-----------------------------------------------------------------------

      do k = 1, 3

         a(:) = 0.d0

         do i = k, 3*natom, 3
            a(i) = 1.d0/sqrt(dble(natom))
         end do

         h3(:,k) = a(:)
         h5(:,k) = a(:)
         h6(:,k) = a(:)

      end do

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      sumx = 0.d0
      sumy = 0.d0
      sumz = 0.d0
      sump = 0.d0

      do j = 1, natom
         sumx = sumx + physmass(j)*x(j,l)
         sumy = sumy + physmass(j)*y(j,l)
         sumz = sumz + physmass(j)*z(j,l)
         sump = sump + physmass(j)
      end do

      xg = sumx/sump
      yg = sumy/sump
      zg = sumz/sump

!-----------------------------------------------------------------------
!     /*   moment of inertia                                          */
!-----------------------------------------------------------------------

      b(:,:) = 0.d0

      do i = 1, natom
         b(1,1) = b(1,1) + physmass(i)*(y(i,l)-yg)*(y(i,l)-yg) &
     &                   + physmass(i)*(z(i,l)-zg)*(z(i,l)-zg)
         b(1,2) = b(1,2) - physmass(i)*(x(i,l)-xg)*(y(i,l)-yg)
         b(1,3) = b(1,3) - physmass(i)*(x(i,l)-xg)*(z(i,l)-zg)
         b(2,1) = b(2,1) - physmass(i)*(y(i,l)-yg)*(x(i,l)-xg)
         b(2,2) = b(2,2) + physmass(i)*(z(i,l)-zg)*(z(i,l)-zg) &
     &                   + physmass(i)*(x(i,l)-xg)*(x(i,l)-xg)
         b(2,3) = b(2,3) - physmass(i)*(y(i,l)-yg)*(z(i,l)-zg)
         b(3,1) = b(3,1) - physmass(i)*(z(i,l)-zg)*(x(i,l)-xg)
         b(3,2) = b(3,2) - physmass(i)*(z(i,l)-zg)*(y(i,l)-yg)
         b(3,3) = b(3,3) + physmass(i)*(x(i,l)-xg)*(x(i,l)-xg) &
     &                   + physmass(i)*(y(i,l)-yg)*(y(i,l)-yg)
      end do

!-----------------------------------------------------------------------
!     /*   principal axis:  diagonalize moment of inertia             */
!-----------------------------------------------------------------------

      call ddiag ( b, g, d, 3 )

!-----------------------------------------------------------------------
!     /*   check if linear molecule                                   */
!-----------------------------------------------------------------------

!     /*   lowest value of moment of inertia   */

      if ( iboundary .eq. 0 ) then
         if ( g(1) .le. 1.d-8 ) then
            i5or6 = 5
         else
            i5or6 = 6
         end if
      else if ( iboundary .eq. 1 ) then
         i5or6 = 3
      else if ( iboundary .eq. 2 ) then
         i5or6 = 3
      else
         i5or6 = 3
      end if

!-----------------------------------------------------------------------
!     /*   rotational contribution:  nonlinear molecule               */
!-----------------------------------------------------------------------

      if ( i5or6 .eq. 6 ) then

         do m = 1, 3

!           /*   coordinates in principal axis   */

            do i = 1, natom
               dx = x(i,l) - xg
               dy = y(i,l) - yg
               dz = z(i,l) - zg
               vx(i) = d(1,1)*dx + d(2,1)*dy + d(3,1)*dz
               vy(i) = d(1,2)*dx + d(2,2)*dy + d(3,2)*dz
               vz(i) = d(1,3)*dx + d(2,3)*dy + d(3,3)*dz
            end do

!           /*   rotational momentum in principal axis   */

            do i = 1, natom

               vxi = vx(i)
               vyi = vy(i)
               vzi = vz(i)

               if      ( m .eq. 1 ) then

                  vx(i) = + 0.d0
                  vy(i) = - vzi
                  vz(i) = + vyi

               else if ( m .eq. 2 ) then

                  vx(i) = + vzi
                  vy(i) = + 0.d0
                  vz(i) = - vxi

               else if ( m .eq. 3 ) then

                  vx(i) = - vyi
                  vy(i) = + vxi
                  vz(i) = + 0.d0

               end if

            end do

!           /*   rotational momentum in laboratory frame   */

            k = 0

            do i = 1, natom

               k = k + 1
               a(k) = d(1,1)*vx(i) + d(1,2)*vy(i) + d(1,3)*vz(i)

               k = k + 1
               a(k) = d(2,1)*vx(i) + d(2,2)*vy(i) + d(2,3)*vz(i)

               k = k + 1
               a(k) = d(3,1)*vx(i) + d(3,2)*vy(i) + d(3,3)*vz(i)

            end do

!           /*  normalize   */

            suma = 0.d0

            do i = 1, 3*natom
               suma = suma + a(i)*a(i)
            end do

            suma = sqrt(suma)

            do i = 1, 3*natom
               a(i) = a(i) / suma
            end do

            h6(:,m+3) = a(:)

         end do

!        /*   orthogonalize   */

         call gram_schmidt ( h6, 3*natom, 6 )

!        /*   projection matrix   */

         do j = 1, 3*natom
         do i = 1, 3*natom
         do k = 1, 6
            p(i,j) = p(i,j) - h6(i,k)*h6(j,k)
         end do
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   rotational contribution:  linear molecule                  */
!-----------------------------------------------------------------------

      if ( i5or6 .eq. 5 ) then

         do m = 2, 3

!           /*   coordinates in principal axis   */

            do i = 1, natom
               dx = x(i,l) - xg
               dy = y(i,l) - yg
               dz = z(i,l) - zg
               vx(i) = d(1,1)*dx + d(2,1)*dy + d(3,1)*dz
               vy(i) = d(1,2)*dx + d(2,2)*dy + d(3,2)*dz
               vz(i) = d(1,3)*dx + d(2,3)*dy + d(3,3)*dz
            end do

!           /*   rotational momentum in principal axis   */

            do i = 1, natom

               vxi = vx(i)
               vyi = vy(i)
               vzi = vz(i)

               if      ( m .eq. 1 ) then

                  vx(i) = + 0.d0
                  vy(i) = - vzi
                  vz(i) = + vyi

               else if ( m .eq. 2 ) then

                  vx(i) = + vzi
                  vy(i) = + 0.d0
                  vz(i) = - vxi

               else if ( m .eq. 3 ) then

                  vx(i) = - vyi
                  vy(i) = + vxi
                  vz(i) = + 0.d0

               end if

            end do

!           /*   rotational momentum in laboratory frame   */

            k = 0

            do i = 1, natom

               k = k + 1
               a(k) = d(1,1)*vx(i) + d(1,2)*vy(i) + d(1,3)*vz(i)

               k = k + 1
               a(k) = d(2,1)*vx(i) + d(2,2)*vy(i) + d(2,3)*vz(i)

               k = k + 1
               a(k) = d(3,1)*vx(i) + d(3,2)*vy(i) + d(3,3)*vz(i)

            end do

!           /*  normalize   */

            suma = 0.d0

            do i = 1, 3*natom
               suma = suma + a(i)*a(i)
            end do

            suma = sqrt(suma)

            do i = 1, 3*natom
               a(i) = a(i) / suma
            end do

            h5(:,m+3-1) = a(:)

         end do

!        /*   orthogonalize   */

         call gram_schmidt ( h5, 3*natom, 5 )

!        /*   projection matrix   */

         do j = 1, 3*natom
         do i = 1, 3*natom
         do k = 1, 5
            p(i,j) = p(i,j) - h5(i,k)*h5(j,k)
         end do
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   no rotational contribution                                 */
!-----------------------------------------------------------------------

      if ( i5or6 .eq. 3 ) then

!        /*   orthogonalize   */

         call gram_schmidt ( h3, 3*natom, 3 )

!        /*   projection matrix   */

         do j = 1, 3*natom
         do i = 1, 3*natom
         do k = 1, 3
            p(i,j) = p(i,j) - h3(i,k)*h3(j,k)
         end do
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   correct hessian                                            */
!-----------------------------------------------------------------------

      do j = 1, 3*natom
      do i = 1, 3*natom

         f(i,j) = 0.d0

         do k = 1, 3*natom
            f(i,j) = f(i,j) + p(k,i)*hess(k,j)
         end do

      end do
      end do

      do j = 1, 3*natom
      do i = 1, 3*natom

         hess(i,j) = 0.d0

         do k = 1, 3*natom
            hess(i,j) = hess(i,j) + f(i,k)*p(k,j)
         end do

      end do
      end do

      return
      end
