!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 27, 2018 by M. Shiga
!      Description:     print final geometry
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine print_final_xyz_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, au_length, box, boxinv, iounit, istep, species, &
     &   natom, nbead, mbox, ikind, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m1, m2, m3

!     /*   real   */
      real(8) :: xa, ya, za, xb, yb, zb, s1, s2, s3

!     /*   real   */
      real(8), parameter :: bohr2ang = au_length/1.d-10

!-----------------------------------------------------------------------
!     /*   print last geometry in xyz format                          */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

      open ( iounit, file = 'final.xyz' )

      do j = 1, nbead

         write( iounit, '(i8)' ) natom

         write( iounit, '(a10,2i8,f16.8)' ) 'ANGSTROM', istep, j, pot(j)

         do i = 1, natom

            xa = x(i,j) * bohr2ang
            ya = y(i,j) * bohr2ang
            za = z(i,j) * bohr2ang

            xb = x(i,j)
            yb = y(i,j)
            zb = z(i,j)

            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

            call pbc_unfold_MPI( xb, yb, zb, m1, m2, m3 )

            xb = xb * bohr2ang
            yb = yb * bohr2ang
            zb = zb * bohr2ang

            write( iounit, '(a4,3f16.8,i4)' ) &
     &         species(i)(1:4), xa, ya, za, ikind(i)

!            write( iounit, '(a4,3f16.8,i4)' )
!     &         species(i)(1:4), xb, yb, zb, ikind(i)

         end do

      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   print last geometry in poscar format                       */
!-----------------------------------------------------------------------

      open ( iounit, file = 'final.poscar' )

      do j = 1, nbead

         write( iounit, fmt = '(a)' ) 'FINAL GEOMETRY'

         write( iounit, fmt = '(f16.8)'  ) 1.d0

         write( iounit, fmt = '(3f16.8)' ) &
     &      box(1,1)*bohr2ang, box(2,1)*bohr2ang, box(3,1)*bohr2ang
         write( iounit, fmt = '(3f16.8)' ) &
     &      box(1,2)*bohr2ang, box(2,2)*bohr2ang, box(3,2)*bohr2ang
         write( iounit, fmt = '(3f16.8)' ) &
     &      box(1,3)*bohr2ang, box(2,3)*bohr2ang, box(3,3)*bohr2ang

         write( iounit, fmt = '(a)', advance='no' ) species(1)

         do i = 2, natom
            if ( species(i) .eq. species(i-1) ) cycle
            write( iounit, fmt = '(a)', advance='no' ) species(i)
         end do

         write( iounit, fmt = '(a)', advance='yes' )

         k = 1

         do i = 2, natom

            if ( species(i) .eq. species(i-1) ) then
               k = k + 1
               cycle
            else
               write( iounit, fmt = '(i8)', advance='no' ) k
               k = 1
            end if

         end do

         write( iounit, fmt = '(i8)', advance='yes' ) k

         write( iounit, fmt = '(a)' ) 'DIRECT'

         do i = 1, natom

            xa = x(i,j)
            ya = y(i,j)
            za = z(i,j)

            s1 = boxinv(1,1)*xa + boxinv(1,2)*ya + boxinv(1,3)*za
            s2 = boxinv(2,1)*xa + boxinv(2,2)*ya + boxinv(2,3)*za
            s3 = boxinv(3,1)*xa + boxinv(3,2)*ya + boxinv(3,3)*za

            write( iounit, fmt = '(3f16.8)' ) s1, s2, s3

         end do

      end do

      close( iounit )

      return
      end

