!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     prepare input files for molecular liquids
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module prep_variables
!***********************************************************************

!     //   maximum number of iterations
      integer :: maxiter = 10000

!     //   maximum number of bonds per atom
      integer :: maxbond = 10

!     //   number of bonds
      integer, dimension(:), allocatable :: mbond

!     //   atomic types
      integer, dimension(:), allocatable :: num

!     //   maximum bond distances
      real(8), dimension(:,:), allocatable :: rbond

!     //   maximum bond distances
      real(8), dimension(:), allocatable :: rbond_spec

!     //   maximum atomic radius
      real(8), dimension(:), allocatable :: radius_symbol

!     //   list of bonded atoms
      integer, dimension(:,:), allocatable :: ibond

!     //   number of bonds
      integer, dimension(:), allocatable :: nbond

!     //   unit conversion factor
      real(8) :: bohr2ang

!***********************************************************************
      end module prep_variables
!***********************************************************************





!***********************************************************************
      program prep_liquid
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      write ( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   set parameters                                             */
!-----------------------------------------------------------------------

      call setparams

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      call setallocation

!-----------------------------------------------------------------------
!     /*   species                                                    */
!-----------------------------------------------------------------------

      call read_species

!-----------------------------------------------------------------------
!     /*   boundary condition                                         */
!-----------------------------------------------------------------------

      call boundary

!-----------------------------------------------------------------------
!     /*   molecules                                                  */
!-----------------------------------------------------------------------

      call init_molcomp

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      call prep_condition

!-----------------------------------------------------------------------
!     /*   random geometry                                            */
!-----------------------------------------------------------------------

      call prep_geometry

!-----------------------------------------------------------------------
!     /*   bond topology                                              */
!-----------------------------------------------------------------------

      call prep_bond

!-----------------------------------------------------------------------
!     /*   print files                                                */
!-----------------------------------------------------------------------

      call prep_print

!-----------------------------------------------------------------------
!     /*   finalize                                                   */
!-----------------------------------------------------------------------

      write ( 6, '(a)' ) 'Normal of termination of PREP_LIQUID.'
      write ( 6, '(a)' )

      stop
      end





!***********************************************************************
      subroutine prep_condition
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, natom, mspec

      use prep_variables, only : &
     &   rbond, rbond_spec, bohr2ang, ibond, nbond, num, maxbond, mbond

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!-----------------------------------------------------------------------
!     //   unit conversion factor
!-----------------------------------------------------------------------

      bohr2ang = au_length / 1.d-10

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      allocate( num(natom) )

      allocate( rbond(natom,natom) )

      allocate( ibond(natom,maxbond) )

      allocate( nbond(natom) )

      allocate( mbond(natom) )

      allocate( rbond_spec(mspec) )

!-----------------------------------------------------------------------
!     //   atomic types
!-----------------------------------------------------------------------

      call get_types

!-----------------------------------------------------------------------
!     //   maximum bond distances
!-----------------------------------------------------------------------

      call get_rbond

      return
      end





!***********************************************************************
      subroutine get_rbond
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, mspec, natom_spec, spec

      use prep_variables, only : &
     &   rbond, rbond_spec

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: i, j, k, l, m, n, ierr

!     //   number of atomic symbols
      integer :: nsymbol

!     //   atomic symbols
      character(len=8), dimension(:), allocatable :: symbol

!     //   atomic numbers
      integer, dimension(:), allocatable :: num_symbol

!     //   atomic masses
      real(8), dimension(:), allocatable :: physmass_symbol

!     //   atomic radii
      real(8), dimension(:), allocatable :: radius_symbol

!-----------------------------------------------------------------------
!     /*   read atomic symbols                                        */
!-----------------------------------------------------------------------

!     /*   number of atoms   */
      call read_int1 ( nsymbol, '<nsymbol>', 9, iounit )

!     /*   memory allocation: atomic symbols   */
      allocate( symbol(nsymbol) )

!     /*   memory allocation: atomic symbols   */
      allocate( num_symbol(nsymbol) )

!     /*   memory allocation: atomic masses   */
      allocate( physmass_symbol(nsymbol) )

!     /*   memory allocation: atomic radii   */
      allocate( radius_symbol(nsymbol) )

!-----------------------------------------------------------------------
!     /*   read atomic symbols                                        */
!-----------------------------------------------------------------------

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<nsymbol>', 9, iounit, ierr )

!     /*   number of atomic symbols   */
      read ( iounit, *, iostat=ierr ) nsymbol

!     /*   loop of symbols   */
      do i = 1, nsymbol

!        /*   read symbol, atomic number, atomic mass   */
         read ( iounit, *, iostat=ierr ) &
     &      symbol(i), num_symbol(i), physmass_symbol(i), &
     &      radius_symbol(i)

!     /*   loop of symbols   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   on error, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<nsymbol>', 9, iounit, ierr )

!        /*   number of atomic symbols   */
         read ( iounit, *, iostat=ierr )

!        /*   loop of symbols   */
         do i = 1, nsymbol

!           /*   read symbol, atomic number, atomic mass   */
            read ( iounit, *, iostat=ierr ) &
     &         symbol(i), num_symbol(i), physmass_symbol(i), &
     &         radius_symbol(i)

!        /*   loop of symbols   */
         end do

!        /*   file close   */
         close( iounit )

!     /*   on error, read default values   */
      end if

!-----------------------------------------------------------------------
!     /*   loop of symbols                                            */
!-----------------------------------------------------------------------

      do k = 1, mspec

         do i = 1, nsymbol

            if ( symbol(i)(1:3) .eq. spec(k)(1:3) ) then

               rbond_spec(k) = 2.d0 * radius_symbol(i)

               exit

            end if

         end do

      end do

!-----------------------------------------------------------------------
!     //   maximum bond lengths
!-----------------------------------------------------------------------

      i = 0

      do k = 1, mspec
      do l = 1, natom_spec(k)

         i = i + 1

         j = 0

         do m = 1, mspec
         do n = 1, natom_spec(m)

            j = j + 1

            rbond(i,j) = 0.5d0 * ( rbond_spec(k) + rbond_spec(m) )

         end do
         end do

      end do
      end do

      return
      end





!***********************************************************************
      subroutine get_types
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : iounit, mspec, natom_spec

      use prep_variables, only : num, mbond

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: i, j, k, ierr, natom_spec_k, num_spec_k, mbond_spec_k

!     //   characters
      character(len=3) :: spec_k

!     //   real numbers
      real(8) :: physmass_spec_k

!-----------------------------------------------------------------------
!     //   initial settings
!-----------------------------------------------------------------------

!     //   open file
      open( iounit, file = 'input.dat' )

!     //   search for keyword
      call search_tag( '<nspec>', 7, iounit, ierr )

!     //   species
      read( iounit, *, iostat=ierr )

!     //   counter
      j = 0

!     //   loop of species
      do k = 1, mspec

!        //   species
         read( iounit, *, iostat=ierr ) &
     &      spec_k, physmass_spec_k, natom_spec_k, &
     &      num_spec_k, mbond_spec_k

!        //   loop of atoms per species
         do i = 1, natom_spec(k)

!           //   counter
            j = j + 1

!           //   number in force field
            num(j) = num_spec_k

!           //   number of bonds
            mbond(j) = mbond_spec_k

!        //   loop of atoms per species
         end do

!     //   loop of species
      end do

!     //   close file
      close( iounit )

!     //   error handling
      call error_handling( ierr, 'subroutine get_types', 20 )

      return
      end





!***********************************************************************
      subroutine prep_geometry
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, x, y, z, box, au_length, nbead, iounit, physmass, natom

      use rotor_variables, only : &
     &   x_mol, y_mol, z_mol, nmol, list_atom_mol, natom_per_mol

      use prep_variables, only : &
     &   rbond, maxiter

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   random numbers
      real(8) :: s(3), psi, theta, phi, ranf1

!     //   integers
      integer :: ibead, i, j, k, l, m, n, iter, iflag

!     //   real numbers
      real(8) :: xg, yg, zg, xk, yk, zk, rx, ry, rz, r2, r2max

!     //   real numbers
      real(8) :: xc, yc, zc, pc

!-----------------------------------------------------------------------
!     //   shift each molecule to origin
!-----------------------------------------------------------------------

      do i = 1, nmol

         xc = 0.d0
         yc = 0.d0
         zc = 0.d0
         pc = 0.d0

         do j = 1, natom_per_mol(i)

            k = list_atom_mol(j,i)

            xc = xc + physmass(k) * x_mol(j,i)
            yc = yc + physmass(k) * y_mol(j,i)
            zc = zc + physmass(k) * z_mol(j,i)

            pc = pc + physmass(k)

         end do

         xc = xc / pc
         yc = yc / pc
         zc = zc / pc

         do j = 1, natom_per_mol(i)

            x_mol(j,i) = x_mol(j,i) - xc
            y_mol(j,i) = y_mol(j,i) - yc
            z_mol(j,i) = z_mol(j,i) - zc

         end do

      end do

!-----------------------------------------------------------------------
!     //   loop of beads
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!        //   loop of molecules
!-----------------------------------------------------------------------

         do i = 1, nmol

!-----------------------------------------------------------------------
!           //   a trial geometry of new molecule
!-----------------------------------------------------------------------

            do iter = 1, maxiter

!              //   molecule: first molecule in original position
               if ( i .eq. 1 ) then

!                 //   random numbers for molecular translation

                  s(1)  = 0.d0
                  s(2)  = 0.d0
                  s(3)  = 0.d0

!                 //   random numbers for molecular rotation

                  phi   = 0.d0
                  theta = 0.d0
                  psi   = 0.d0

!              //   molecule: other molecules in random position
               else

!                 //   random numbers for molecular translation

                  s(1)  = ranf1() - 0.5d0
                  s(2)  = ranf1() - 0.5d0
                  s(3)  = ranf1() - 0.5d0

!                 //   random numbers for molecular rotation

                  phi   = ranf1() * 2.d0*pi
                  theta = ranf1() * 2.d0*pi
                  psi   = ranf1() * 2.d0*pi

!              //   molecule
               end if

!-----------------------------------------------------------------------
!              //   creat new geometry
!-----------------------------------------------------------------------

!              //   loop of atoms
               do j = 1, natom_per_mol(i)

!                 //  atom
                  k = list_atom_mol(j,i)

!-----------------------------------------------------------------------
!                 //   random translation within box
!-----------------------------------------------------------------------

!                 //   atoms in molecular frame

                  x(k,ibead) = x_mol(j,i)
                  y(k,ibead) = y_mol(j,i)
                  z(k,ibead) = z_mol(j,i)

!                 //   random center of mass

                  xg = box(1,1)*s(1) + box(1,2)*s(2) + box(1,3)*s(3)
                  yg = box(2,1)*s(1) + box(2,2)*s(2) + box(2,3)*s(3)
                  zg = box(3,1)*s(1) + box(3,2)*s(2) + box(3,3)*s(3)

!                 //   shift positions

                  x(k,ibead) = x(k,ibead) + xg
                  y(k,ibead) = y(k,ibead) + yg
                  z(k,ibead) = z(k,ibead) + zg

!-----------------------------------------------------------------------
!                 //   random rotation around z axis
!-----------------------------------------------------------------------

                  xk =   cos(phi)*x(k,ibead) + sin(phi)*y(k,ibead)
                  yk = - sin(phi)*x(k,ibead) + cos(phi)*y(k,ibead)

                  x(k,ibead) = xk
                  y(k,ibead) = yk

!-----------------------------------------------------------------------
!                 //   random rotation around x axis
!-----------------------------------------------------------------------

                  yk =   cos(theta)*y(k,ibead) + sin(theta)*z(k,ibead)
                  zk = - sin(theta)*y(k,ibead) + cos(theta)*z(k,ibead)

                  y(k,ibead) = yk
                  z(k,ibead) = zk

!-----------------------------------------------------------------------
!                 //   random rotation around z axis
!-----------------------------------------------------------------------

                  xk =   cos(psi)*x(k,ibead) + sin(psi)*y(k,ibead)
                  yk = - sin(psi)*x(k,ibead) + cos(psi)*y(k,ibead)

                  x(k,ibead) = xk
                  y(k,ibead) = yk

!-----------------------------------------------------------------------
!              //   create new geometry
!-----------------------------------------------------------------------

               end do

!-----------------------------------------------------------------------
!              //   check interatomic distances
!-----------------------------------------------------------------------

!              //   loop of atoms
               do j = 1, natom_per_mol(i)

!                 //  atom
                  k = list_atom_mol(j,i)

!                 //   loop of molecules
                  do l = 1, i-1

!                    //   loop of atoms
                     do m = 1, natom_per_mol(l)

!                       //   atom
                        n = list_atom_mol(m,l)

!                       //   maximum bond distance
                        r2max = rbond(k,n) * rbond(k,n)

!                       //   distance
                        rx = x(k,ibead) - x(n,ibead)
                        ry = y(k,ibead) - y(n,ibead)
                        rz = z(k,ibead) - z(n,ibead)

!                       //   apply periodic boundary condition
                        call pbc_atom( rx, ry, rz )

!                       //   distance
                        r2 = rx*rx + ry*ry + rz*rz

!                       //   reject if too close
                        if ( r2 .le. r2max ) go to 100

!                    //   loop of atoms
                     end do

!                 //   loop of molecules
                  end do

!              //   loop of atoms
               end do

!              //   check successful
               iflag = 1

!              //   proceed to new molecule
               exit

!              //   check failed
  100          continue

!              //   error flag
               iflag = 0

!-----------------------------------------------------------------------
!           //   loop of trial geometry
!-----------------------------------------------------------------------

            end do

!-----------------------------------------------------------------------
!           //   error handling
!-----------------------------------------------------------------------

            if ( iflag .eq. 0 ) then

               write( 6, '(a,i6,a,i6,a)' ) &
     &            'Error - Failed to place molecule:', i, &
     &            ' after ', maxiter, ' iterations.'

               write( 6, '(a)' ) &
     &            'Error termination: ' // &
     &            'subroutine prep_geometry.'

               write( 6, '(a)' )

               stop

             end if

!-----------------------------------------------------------------------
!        //  loop of molecules
!-----------------------------------------------------------------------

         end do

!-----------------------------------------------------------------------
!     //   loop of beads
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     //   shift each bead configuration to origin
!-----------------------------------------------------------------------

      xg = 0.5d0 * ( box(1,1) + box(1,2) + box(1,3) )
      yg = 0.5d0 * ( box(2,1) + box(2,2) + box(2,3) )
      zg = 0.5d0 * ( box(3,1) + box(3,2) + box(3,3) )

      do ibead = 1, nbead

         xc = 0.d0
         yc = 0.d0
         zc = 0.d0
         pc = 0.d0

         do k = 1, natom

            xc = xc + physmass(k) * x(k,ibead)
            yc = yc + physmass(k) * y(k,ibead)
            zc = zc + physmass(k) * z(k,ibead)
            pc = pc + physmass(k)

         end do

         xc = xc / pc
         yc = yc / pc
         zc = zc / pc

         do k = 1, natom

            x(k,ibead) = x(k,ibead) - xc + xg
            y(k,ibead) = y(k,ibead) - yc + yg
            z(k,ibead) = z(k,ibead) - zc + zg

         end do

      end do

      return
      end





!***********************************************************************
      subroutine prep_bond
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, natom

      use prep_variables, only : &
     &   rbond, ibond, nbond, maxbond, mbond

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: ibead, i, j, k

!     //   real numbers
      real(8) :: rx, ry, rz, r2, r2max

      write( 6, '(a)' ) &
     &   '------------------------------------------------'
      write( 6, '(a)' ) &
     &   '  atom 1  atom 2  bond order   dist/au  r1+r2/au'
      write( 6, '(a)' ) &
     &   '------------------------------------------------'

      do ibead = 1, 1

         do i = 1, natom

            k = 0

            do j = 1, natom

               if ( i .eq. j ) cycle

!              //   distance
               rx = x(i,ibead) - x(j,ibead)
               ry = y(i,ibead) - y(j,ibead)
               rz = z(i,ibead) - z(j,ibead)

!              //   apply periodic boundary condition
               call pbc_atom( rx, ry, rz )

!              //   distance
               r2 = rx*rx + ry*ry + rz*rz

!              //   maximum bond distance
               r2max = rbond(i,j) * rbond(i,j)

!              //   look for bonds
               if ( r2 .le. r2max ) then

!                 //   number of bonds
                  k = k + 1

!                 //   list of bonded atoms
                  if ( k .le. maxbond ) ibond(i,k) = j

                  write( 6, '(2i8,2i6,2f10.5)' ) &
     &               i, j, k, mbond(i), sqrt(r2), sqrt(r2max)

!              //   look for bonds
               end if

            end do

!           //   number of bonds
            nbond(i) = k

!           //   checking number of bonds

            if ( nbond(i) .ne. mbond(i) ) then

               write( 6, '(a,i2,a,i2,a,i6,a)' ) &
     &            'Warning - Number of bonds do not match: ', &
     &             nbond(i), ' and ', mbond(i), ' for atom:', i, '.'

            end if

!           //   error handling

            if ( nbond(i) .gt. maxbond ) then

               write( 6, '(a,i1,a,i6,a)' ) &
     &            'Error - Number of bonds greater than ', &
     &             maxbond, ' for atom:', i, '.'
               write( 6, '(a)' ) &
     &            'Error termination: ' // &
     &            'subroutine prep_bond.'
               write( 6, '(a)' )

               stop

            end if

         end do

      end do

      return
      end





!***********************************************************************
      subroutine prep_print
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   species, x, y, z, nbead, natom, iounit, ikind

      use prep_variables, only : &
     &   bohr2ang, ibond, nbond, num

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: ibead, i, k

!     //   real numbers
      real(8) :: xk, yk, zk

!-----------------------------------------------------------------------
!     //   print geometry: in xyz format
!-----------------------------------------------------------------------

      open ( iounit, file = 'structure.dat' )

      do i = 1, 1

         write( iounit, '(i6)' ) natom
         write( iounit, '(a)' ) 'ANGSTROM'

         do k = 1, natom

            xk = x(k,i) * bohr2ang
            yk = y(k,i) * bohr2ang
            zk = z(k,i) * bohr2ang

            write( iounit, '(a4,3f12.6,i4)' ) &
     &         species(k), xk, yk, zk, ikind(k)

         end do

      end do

      close( iounit )

      write ( 6, '(a)' )
      write ( 6, '(a)' ) 'Created: structure.dat.'
      write ( 6, '(a)' )

!-----------------------------------------------------------------------
!     //   print geometry: in centroid format
!-----------------------------------------------------------------------

      open ( iounit, file = 'centroid.dat' )

      do i = 1, 1
      do k = 1, natom

         xk = x(k,i)
         yk = y(k,i)
         zk = z(k,i)

         write( iounit, '(3f12.6)' ) xk, yk, zk

      end do
      end do

      close( iounit )

      write ( 6, '(a)' ) 'Created: centroid.dat.'
      write ( 6, '(a)' )

!-----------------------------------------------------------------------
!     //   print geometry: in geometry.ini format
!-----------------------------------------------------------------------

      open ( iounit, file = 'geometry.ini' )

      do i = 1, nbead
      do k = 1, natom

         xk = x(k,i)
         yk = y(k,i)
         zk = z(k,i)

         write( iounit, '(i3,6f24.16,3i4)' ) &
     &      0, xk, yk, zk, 0.d0, 0.d0, 0.d0, 0, 0, 0

      end do
      end do

      close( iounit )

      write ( 6, '(a)' ) 'Created: geometry.ini.'
      write ( 6, '(a)' )

!-----------------------------------------------------------------------
!     //   print geometry: in txyz format
!-----------------------------------------------------------------------

      open ( iounit, file = 'structure.txyz' )

      write( iounit, '(i5,a)' ) natom, ' Created by PIMD.'

      do ibead = 1, 1

         do i = 1, natom

!           //   number of bonds
            k = nbond(i)

!           //   unit conversion to angstroms

            xk = x(i,ibead) * bohr2ang
            yk = y(i,ibead) * bohr2ang
            zk = z(i,ibead) * bohr2ang

!           //   print in tinker xyz format

            write( iounit, '(i5,1x,a3,1x,3f10.4,i6,4i6)' ) &
     &         i, species(i), xk, yk, zk, num(i), ibond(i,1:k)

         end do

      end do

      close( iounit )

      write ( 6, '(a)' ) 'Created: structure.txyz.'
      write ( 6, '(a)' )

      return
      end
