!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     BEST method for aqueous solutions
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      program prep_best
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: iounit = 10

      integer :: natom, nwater, i, j, nwater_qm, narg, natom_mol, iobest

      integer :: iargc

      integer :: ispec_o

      real(8), dimension(:), allocatable :: x, y, z, d, dmin

      real(8) :: boxl, dx, dy, dz

      real(8) :: bohr2ang = 0.529177249d0

      integer, dimension(:), allocatable :: ispec

      integer, dimension(:), allocatable :: iflag, imin, iox, ih1, ih2

      character(len=4), dimension(:), allocatable :: symbol

      character(len=24) :: argv

      character(len=2) :: acharge

!-----------------------------------------------------------------------
!     /*   input data                                                 */
!-----------------------------------------------------------------------

      narg = iargc()

      if ( narg .ne. 7 ) then
         write( 6, '(a)' )
         write( 6, '(a)' ) 'USAGE OF PREP_BEST.X'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'RUN WITH FOLLOWING ARGUMENTS.'
         write( 6, '(a)' ) '(1) number of atoms of the solute'
         write( 6, '(a)' ) '(2) number of water molecules'
         write( 6, '(a)' ) '(3) number of QM water molecules'
         write( 6, '(a)' ) '(4) central atom or ion in best'
         write( 6, '(a)' ) '(5) oxygen species number'
         write( 6, '(a)' ) '(6) side length of cubic box [angstrom]'
         write( 6, '(a)' ) '(7) charge of the system'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'STRUCTURE.DAT IS READ.'
         write( 6, '(a)' )
         write( 6, '(a)' ) 'FORMAT OF STRUCTURE.DAT MUST BE AS FOLLOWS.'
         write( 6, '(a)' ) 'The 1st   M atoms must be the solute.'
         write( 6, '(a)' ) 'The next  N atoms must be O atom of water.'
         write( 6, '(a)' ) 'The next 2N atoms must be H atom of water.'
         write( 6, '(a)' )
         stop
      end if

      call getarg( 1, argv )
      read( argv, * ) natom_mol

      call getarg( 2, argv )
      read( argv, * ) nwater

      call getarg( 3, argv )
      read( argv, * ) nwater_qm

      call getarg( 4, argv )
      read( argv, * ) iobest

      call getarg( 5, argv )
      read( argv, * ) ispec_o

      call getarg( 6, argv )
      read( argv, * ) boxl

      call getarg( 7, argv )
      acharge = argv(1:2)

!-----------------------------------------------------------------------
!     /*   boundary condition                                         */
!-----------------------------------------------------------------------

      natom = 3 * nwater + natom_mol

      allocate( x(natom) )
      allocate( y(natom) )
      allocate( z(natom) )
      allocate( symbol(natom) )
      allocate( ispec(natom) )

      allocate( d(nwater) )
      allocate( iflag(nwater) )
      allocate( dmin(nwater) )
      allocate( imin(nwater) )
      allocate( iox(nwater) )
      allocate( ih1(nwater) )
      allocate( ih2(nwater) )

!-----------------------------------------------------------------------
!     /*   read atoms                                                 */
!-----------------------------------------------------------------------

      open ( iounit, file = 'structure.dat' )

      read ( iounit, * )
      read ( iounit, * )

      do i = 1, natom
         read ( iounit, * ) symbol(i), x(i), y(i), z(i), ispec(i)
      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   distance between o atom and ion                            */
!-----------------------------------------------------------------------

      nwater = ( natom - natom_mol ) / 3

      do i = 1, nwater

         dx = x(i+natom_mol) - x(iobest)
         dy = y(i+natom_mol) - y(iobest)
         dz = z(i+natom_mol) - z(iobest)

         dx = dx - nint(dx/boxl) * boxl
         dy = dy - nint(dy/boxl) * boxl
         dz = dz - nint(dz/boxl) * boxl

         d(i) = sqrt( dx*dx + dy*dy + dz*dz )

         iflag(i) = 0

      end do

!-----------------------------------------------------------------------
!     /*   print results                                              */
!-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) '------------------------------'
      write( 6, '(a)' ) 'water    O    H    H  O-M dist'
      write( 6, '(a)' ) '------------------------------'

      do j = 1, nwater

         imin(j) = 0
         dmin(j) = boxl

         do i = 1, nwater

            if ( iflag(i) .eq. 1 ) cycle

            if ( d(i) .lt. dmin(j) ) then

               imin(j) = i
               dmin(j) = d(i)

            end if

         end do

         iox(j) = natom_mol + imin(j)
         ih1(j) = natom_mol + imin(j) + nwater
         ih2(j) = natom_mol + imin(j) + 2*nwater

         write( 6, '(4i5,f10.5)' ) j, iox(j), ih1(j), ih2(j), dmin(j)

         iflag(imin(j)) = 1

      end do

!-----------------------------------------------------------------------
!     /*   print results                                              */
!-----------------------------------------------------------------------

      open ( iounit, file = 'structure.best' )

      write( iounit, '(i6)' ) natom
      write( iounit, '(a)' ) 'ANGSTROM'

      do i = 1, natom_mol

         write( iounit, '(a4,3f10.5,i2,1x,a1)' ) &
     &      symbol(i), x(i), y(i), z(i), ispec(i), 'A'

      end do

      do j = 1, nwater

         i = iox(j)

         if ( j .le. nwater_qm ) then

            write( iounit, '(a4,3f10.5,i2,1x,a1)' ) &
     &         symbol(i), x(i), y(i), z(i), ispec(i), 'A'

            if ( ispec_o .ne. ispec(i) ) then
               write( 6, '(a)' ) 'Error - oxygen species number.'
            end if

         else

            write( iounit, '(a4,3f10.5,i2,1x,a1)' ) &
     &         symbol(i), x(i), y(i), z(i), ispec(i), 'B'

            if ( ispec_o .ne. ispec(i) ) then
               write( 6, '(a)' ) 'Error - oxygen species number.'
            end if

         end if

      end do

      do j = 1, nwater

         i = ih1(j)

         if ( j .le. nwater_qm ) then

            write( iounit, '(a4,3f10.5,i2,1x,a1)' ) &
     &         symbol(i), x(i), y(i), z(i), ispec(i), 'A'

         else

            write( iounit, '(a4,3f10.5,i2,1x,a1)' ) &
     &         symbol(i), x(i), y(i), z(i), ispec(i), 'B'

         end if

      end do

      do j = 1, nwater

         i = ih2(j)

         if ( j .le. nwater_qm ) then

            write( iounit, '(a4,3f10.5,i2,1x,a1)' ) &
     &         symbol(i), x(i), y(i), z(i), ispec(i), 'A'

         else

            write( iounit, '(a4,3f10.5,i2,1x,a1)' ) &
     &         symbol(i), x(i), y(i), z(i), ispec(i), 'B'

         end if

      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   end of code                                                */
!-----------------------------------------------------------------------

      open ( iounit, file = 'input.best' )

      write( iounit, '(a)' ) '<method>'
      write( iounit, '(a)' ) 'STATIC'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<ipotential>'
      write( iounit, '(a)' ) 'QMMM'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<qmmm_embedding>'
      write( iounit, '(a)' ) 'ME'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<qmmm_potential>'
      write( iounit, '(a)' ) 'SMASH'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<smash_options>'
      write( iounit, '(a,i2)' ) 'job method=hf basis=6-31g(d) ' // &
     &                          'memory=1gb charge=' // trim(acharge)
      write( iounit, '(a)' ) 'control iprint=1'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<iboundary>'
      write( iounit, '(a)' ) '1'
      write( iounit, '(3f10.5)' ) boxl/bohr2ang, 0.d0, 0.d0
      write( iounit, '(3f10.5)' ) 0.d0, boxl/bohr2ang, 0.d0
      write( iounit, '(3f10.5)' ) 0.d0, 0.d0, boxl/bohr2ang
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<ioption_best>'
      write( iounit, '(a)' ) '1'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<iobest>'
      write( iounit, '(i5)' ) iobest
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<ikind_best>'
      write( iounit, '(i5)' ) ispec_o
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<fc_best>'
      write( iounit, '(a)' ) '25.0'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<eps_best>'
      write( iounit, '(a)' ) '1.e-16'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<iprint_best>'
      write( iounit, '(a)' ) '1'
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<iread_exit>'
      write( iounit, '(a)' ) '1'
      write( iounit, '(a)' )

      close( iounit )

!-----------------------------------------------------------------------
!     /*   end of code                                                */
!-----------------------------------------------------------------------

      write( 6, '(a)' )
      write( 6, '(a)' ) 'A new file structure.best created.'
      write( 6, '(a)' ) 'Rename structure.best to structure.dat.'
      write( 6, '(a)' )
      write( 6, '(a)' ) 'A new file input.best created.'
      write( 6, '(a)' ) 'Rename input.best to input.dat.'
      write( 6, '(a)' )
      write( 6, '(a)' ) 'Normal termination.'
      write( 6, '(a)' )

      stop
      end

